<?php

App::uses('EvCheckoutAppController', 'EvCheckout.Controller');
App::uses('InvalidStageProcessException', 'EvCheckout.Lib/Exceptions');

class CheckoutController extends EvCheckoutAppController {

	/**
	 * the active stage process either from config or session override
	 *
	 * @var string
	 */
	public $activeStageProcess = null;

	/**
	 * the current active stage
	 *
	 * @var string
	 */
	public $activeStage = null;

	public $helpers = [
		'EvCheckout.Order',
	];

	/**
	 * override constructor - set our components
	 */
	public function __construct(CakeRequest $Request, CakeResponse $Response) {
		$this->components['PreStage'] = array(
			'className' => 'EvCheckout.PreStage'
		);
		$this->components['PostStage'] = array(
			'className' => 'EvCheckout.PostStage'
		);

		parent::__construct($Request, $Response);
	}

	/**
	 * set the authorised methods
	 */
	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(
			array('index', 'stage', 'successful', 'fail')
		);

		// default stage process
		$stageProcess = Configure::read('EvCheckout.stageProcess');
		$stages = Configure::read('EvCheckout.stages');

		// check for an overriding process
		if ($this->Session->check('EvCheckout.stageProcess')) {
			$stageProcess = $this->Session->read('EvCheckout.stageProcess');
		}

		// check the selected process to use is valid
		if (empty($stages[$stageProcess]) || ! is_array($stages[$stageProcess])) {
			throw new InvalidStageProcessException(array($stageProcess));
		}

		$this->activeStageProcess = $stageProcess;

		// Don't allow caching of the checkout pages.
		// This prevents old data being shown on navigating back.
		$this->response->expires('now');
		$this->response->sharable(false, 0);
		$this->response->disableCache();
	}

/**
 * Called after the controller action is run, but before the view is rendered. You can use this method
 * to perform logic or set view variables that are required on every request.
 *
 * @return void
 * @link http://book.cakephp.org/2.0/en/controllers.html#request-life-cycle-callbacks
 */
	public function beforeRender() {
		parent::beforeRender();

		// if the session is set to omit over VAT then set a view template to
		// show ex VAT pricings on basket templates. The session variable is
		// set in EvCheckout.PreStageComponent::processBasketVATFromAddress
		// once the customer have chosen / entered their delivery address
		if (CakeSession::check('EvCheckout.omitOrderVat')) {
			$omitOrderVat = CakeSession::read('EvCheckout.omitOrderVat');
			$this->set(compact('omitOrderVat'));
		}
	}

/**
 * Loop the given array and try to run the methods on the given component - for processing the stages.
 *
 * @param string $componentKey The component key to run methods on.
 * @param array  $processors   Array of any methods to run.
 * @return bool True if all methods return true, false otherwise.
 */
	protected function _callStageProcessors($componentKey, $processors) {
		$result = true;
		$processorAttributes = [
			'skipAll' => false,
			'skipNext' => false,
		];

		if (! empty($processors)) {
			foreach ($processors as $method) {
				if (!$result) {
					return $result;
				}

				if (!empty($processorAttributes['skipAll'])) {
					return $result;
				}

				if (!empty($processorAttributes['skipNext'])) {
					$processorAttributes['skipNext'] = false;
					continue;
				}

				if (method_exists($this->{$componentKey}, $method)) {
					$methodResult = $this->{$componentKey}->{$method}();

					if (is_array($methodResult) && isset($methodResult['success'])) {
						//Override any values currently stored in the processor attributes.
						$processorAttributes = Hash::merge($processorAttributes, $methodResult);

						//Check to see if the method returned a success value
						$methodResult = !empty($methodResult['success']);
					}

					$result = $result && $methodResult;
				}
			}
		}

		return $result;
	}

	/**
	 * preStage failed, calculate the redirect route
	 *
	 * @param 	string 	$stageId 	The current stage ID
	 * @return 	array 	$route 		The route array to redirect to
	 */
	public function calculatePreStageRedirect($stageId) {
		$stages = Configure::read('EvCheckout.stages.' . $this->activeStageProcess);

		// all processors returned true, proceed
		$position = array_search($stageId, array_keys($stages));
		$position--;

		if ($position >= 0) {
			$stageIterator = new ArrayIterator($stages);
			$stageIterator->seek($position);

			$prevStageKey = $stageIterator->key();

			// redirect to the next stage
			if (! empty($prevStageKey)) {
				$route = $this->Routable->getListingRoute('EvCheckout', 'Checkout', 'stage');
				$route[] = $prevStageKey;
			}
		}

		if (empty($route)) {
			// no stage found - start from beginning
			$route = $this->Routable->getListingRoute('EvCheckout', 'Checkout');
		}

		return $route;
	}

	/**
	 * postStage succeeded, calculate the redirect route
	 *
	 * @param 	string 	$stageId 	The current stage ID
	 * @return 	array 	$route 	The route array to redirect to
	 */
	public function calculatePostStageRedirect($stageId) {
		$stages = Configure::read('EvCheckout.stages.' . $this->activeStageProcess);

		// all processors returned true, proceed
		$position = array_search($stageId, array_keys($stages));

		$stageIterator = new ArrayIterator($stages);
		$stageIterator->seek($position);
		$stageIterator->next();

		$nextStageKey = $stageIterator->key();

		// redirect to the next stage
		if (! empty($nextStageKey)) {
			$route = $this->Routable->getListingRoute('EvCheckout', 'Checkout', 'stage');
			$route[] = $nextStageKey;
		} else {
			// no more stages but everything was successful.
			// complete
			$route = $this->Routable->getListingRoute('EvCheckout', 'Checkout', 'successful');
		}

		return $route;
	}

	/**
	 * default checkout page - just get the first stage and redirect to it
	 *
	 */
	public function index() {
		$stages = Configure::read('EvCheckout.stages.' . $this->activeStageProcess);
		reset($stages);
		$stageId = key($stages);

		$route = $this->Routable->getListingRoute('EvCheckout', 'Checkout', 'stage');
		$route[] = $stageId;

		$this->redirect(
			$route
		);
	}

	/**
	 * the main stage method to load each stage of the checkout
	 *
	 */
	public function stage($stageId) {
		$stageInfo = Configure::read('EvCheckout.stages.' . $this->activeStageProcess . '.' . $stageId);

		if (empty($stageInfo)) {
			$this->redirect(
				array(
					'action' => 'index'
				)
			);
		}

		// set the active stage
		$this->activeStage = $stageId;

		// check the preStage calls
		$result = $this->_callStageProcessors('PreStage', $stageInfo['pre']);

		// go back a step if the results are false
		if ($result === false) {
			$route = $this->calculatePreStageRedirect($stageId);

			$this->redirect(
				$route
			);
		}

		if ($this->request->is('post') || $this->request->is('put')) {
			$result = $this->_callStageProcessors('PostStage', $stageInfo['post']);

			// proceed to next step
			if ($result === true) {
				$route = $this->calculatePostStageRedirect($stageId);

				$this->redirect(
					$route
				);
			}
		}

		if (!empty($stageInfo['assignPage'])) {
			$this->assignPage($stageInfo['assignPage'], 'checkoutPage');
		}

		$this->view = 'EvCheckout.' . $stageInfo['template'];
	}
}
