<?php

App::uses('AppShell', 'Console/Command');

class OrdersShell extends AppShell {

/**
 * Gets the option parser instance and configures it.
 *
 * @return ConsoleOptionParser
 * @link http://book.cakephp.org/2.0/en/console-and-shells.html#Shell::getOptionParser
 */
	public function getOptionParser() {
		$parser = parent::getOptionParser();

		$parser->addSubcommand(
			'convertNotes',
			[
				'help' => 'Convert order note strings into separate order notes',
				'parser' => $this->_getConverNotesOptionParser(),
			]
		);

		return $parser;
	}

/**
 * Gets the option parser for the convert notes command.
 *
 * @return ConsoleOptionParser
 * @link http://book.cakephp.org/2.0/en/console-and-shells.html#Shell::getOptionParser
 */
	protected function _getConverNotesOptionParser() {
		return [
			'options' => [
				'admin' => [
					'short' => 'a',
					'help' => 'Convert admin notes instead of customer notes. Boolean Defaults to false.',
					'boolean' => true,
				],
				'separator' => [
					'short' => 's',
					'help' => 'The string that separates current order notes so they can be split up into individual notes.'
				],
				'orderId' => [
					'short' => 'i',
					'help' => 'The id of a specific order to convert.'
				]
			]
		];
	}

/**
 * Convert a single string of notes into separate individual notes.
 * Available options:
 * admin - Converting admin notes or customer notes, defaults to false (customer notes).
 * separator - The separator between each note. Use if you have multiple notes in the same string.
 * orderId - An order ID to be used to convert notes of a single order.
 *
 * @return bool|void.
 */
	public function convertNotes() {
		$Order = EvClassRegistry::init('EvCheckout.Order');

		$fields = [
			'Order.id'
		];

		$conditions = [];

		if ($this->param('admin')) {
			$fields[] = 'Order.admin_notes';
			$conditions[] = 'Order.admin_notes IS NOT NULL';
		} else {
			$fields[] = 'Order.notes';
			$conditions[] = 'Order.notes IS NOT NULL';
		}

		if (!empty($this->param('orderId'))) {
			$conditions['Order.id'] = $this->param('orderId');
		}

		$ordersToConvert = $Order->find(
			'all',
			[
				'fields' => $fields,
				'conditions' => $conditions
			]
		);

		if (!empty($ordersToConvert)) {
			$orderNotes = [];
			foreach ($ordersToConvert as $order) {
				$orderNote = [
					'order_id' => $order['Order']['id']
				];

				if ($this->param('admin')) {
					$note = $order['Order']['admin_notes'];
					$orderNote['is_admin_note'] = true;
				} else {
					$note = $order['Order']['notes'];
					$orderNote['is_admin_note'] = false;
				}

				if (!empty($this->param('separator'))) {
					//If a separator has been provided then attempt to split the notes into indiviual notes
					//and save each one.
					$splitNotes = explode($this->param('separator'), $note);
					if (!empty($splitNotes)) {
						foreach ($splitNotes as $splitNote) {
							//Check the split note isn't an empty string.
							if (!empty($splitNote)) {
								$splitOrderNote = [
									'order_id' => $orderNote['order_id'],
									'is_admin_note' => $orderNote['is_admin_note'],
									'note' => $splitNote,
								];

								$orderNotes[] = $splitOrderNote;
							}
						}
					}
				} else {
					$orderNote['note'] = $note;
					$orderNotes[] = $orderNote;
				}

			}

			if (!empty($orderNotes)) {
				return EvClassRegistry::init('EvCheckout.OrderNote')->saveMany($orderNotes);
			}
		}
	}
}
