<?php

App::uses('PluginInstallShell', 'shells');
App::uses('File', 'Utility');
App::uses('MigrationUtil', 'EvMigrationUtil.Lib');

class EvCheckoutInstallShell extends PluginInstallShell {

	public $uses = [
		'EvTemplates.Template',
		'EvSiteSettings.SiteSetting',
	];

/**
 * call parent run() This will auto run any migrations
 *
 * Use the run() method here to insert any content
 */
	public function run() {
		parent::run();

		$this->_createPages();
		$this->_createMenu();
		$this->_createOrderStatuses();
		$this->_createSiteSettings();
		$this->_createEmails();
		$this->_createConfig();
	}

/**
 * Create menu items
 *
 * @return void
 */
	protected function _createMenu() {
		$MenuMigrationUtil = MigrationUtil::init('EvNavigation.Menu');
		$MenuMigrationUtil->createOnce([
			'plugin' => 'EvCheckout',
			'controller' => 'ev_checkout.orders',
			'action' => 'index',
		], [
			'parent_id' => 1,
			'name' => 'Orders',
			'pattern' => '/^\/admin\/ev_checkout\//',
			'plugin' => 'EvCheckout',
			'controller' => 'ev_checkout.orders',
			'model' => 'Order',
			'action' => 'index',
			'class' => 'icon-gbp',
			'is_active' => true,
		]);
	}

/**
 * Create pages and their base templates
 *
 * @return void
 */
	protected function _createPages() {
		// Create the base templates
		$TemplateMigrationUtil = MigrationUtil::init('EvTemplates.Template');
		$templates = [
			[
				'name' => 'Order Successful',
				'template' => 'EvCheckout./Templates/Order/successful',
				'helperFunction' => 'N/A',
				'cfDataLabel' => 'Custom Data',
				'model' => 'Page',
				'is_active' => true,
			],
			[
				'name' => 'Order Failed',
				'template' => 'EvCheckout./Templates/Order/fail',
				'helperFunction' => 'N/A',
				'cfDataLabel' => 'Custom Data',
				'model' => 'Page',
				'is_active' => true,
			],
			[
				'name' => 'Order Cancelled',
				'template' => 'EvCheckout./Templates/Order/cancelled',
				'helperFunction' => 'N/A',
				'cfDataLabel' => 'Custom Data',
				'model' => 'Page',
				'is_active' => true,
			],
		];
		$TemplateMigrationUtil->createAllOnce(['template', 'model'], $templates);

		$templateIds = $this->Template->find('list', [
			'fields' => ['name', 'id'],
			'callbacks' => false,
			'limit' => 3,
			'order' => 'Template.id DESC'
		]);

		$PageMigrationUtil = MigrationUtil::init('EvCore.Page');
		$pages = [
			[
				'title' => 'Order Successful',
				'internal_title' => 'order_successful',
				'body' => '<p>Your order (#ORDERID) was placed successfully. You shall receive a confirmation email shortly.</p>',
				'is_active' => true,
				'is_protected' => true,
				'template_id' => $templateIds['Order Successful'],
			],
			[
				'title' => 'Order Failed',
				'internal_title' => 'order_failed',
				'body' => '<p>There was a problem completing your order, please try again. If the problem persists, please contact us.</p>',
				'is_active' => true,
				'is_protected' => true,
				'template_id' => $templateIds['Order Failed'],
			],
			[
				'title' => 'Order Cancelled',
				'internal_title' => 'order_cancelled',
				'body' => '<p>The checkout process was cancelled.</p>',
				'is_active' => true,
				'is_protected' => true,
				'template_id' => $templateIds['Order Cancelled'],
			],
		];
		$PageMigrationUtil->createAllOnce(['internal_title'], $pages);
	}

/**
 * Create order statuses
 *
 * @return void
 */
	protected function _createOrderStatuses() {
		$OrderStatusMigrationUtil = MigrationUtil::init('EvCheckout.OrderStatus');
		$statuses = [
			[
				'name' => 'Awaiting Payment',
				'slug' => 'awaiting_payment',
				'sequence' => 10,
				'is_selectable' => true,
				'is_active' => true,
			],
			[
				'name' => 'Paid',
				'slug' => 'paid',
				'sequence' => 20,
				'is_selectable' => true,
				'is_active' => true,
			],
			[
				'name' => 'Shipped',
				'slug' => 'shipped',
				'sequence' => 30,
				'is_selectable' => true,
				'is_active' => true,
			],
		];
		$OrderStatusMigrationUtil->createAllOnce(['slug'], $statuses);
	}

/**
 * Create site settings for EvCheckout
 *
 * @return void
 */
	protected function _createSiteSettings() {
		$CategoryMigrationUtil = MigrationUtil::init('EvSiteSettings.SiteSettingCategory');
		$categoryId = $CategoryMigrationUtil->createOnce(['array_slug' => 'ev_checkout'], [
			'name' => 'Checkout',
			'array_slug' => 'ev_checkout',
			'sequence' => 0,
			'prevent_edit' => false,
		]);

		$SettingMigrationUtil = MigrationUtil::init('EvSiteSettings.SiteSetting');
		$settings = [
			[
				'name' => 'New Order Notification Email',
				'array_slug' => 'new_order_email',
				'value' => 'admin@evoluted.net',
				'type' => 'string',
				'sequence' => 0,
				'site_setting_category_id' => $categoryId,
				'prevent_edit' => false,
			],
			[
				'name' => 'New Order Notification Name',
				'array_slug' => 'new_order_name',
				'value' => 'Admin',
				'type' => 'string',
				'sequence' => 0,
				'site_setting_category_id' => $categoryId,
				'prevent_edit' => false,
			],
			[
				'name' => 'Enable Adwords Conversion Tracking?',
				'array_slug' => 'enable_conversion_tracking',
				'value' => '1',
				'type' => 'boolean',
				'sequence' => 0,
				'site_setting_category_id' => $categoryId,
				'prevent_edit' => false,
			],
			[
				'name' => 'Adwords Conversion ID',
				'array_slug' => 'google_conversion_id',
				'value' => '',
				'type' => 'string',
				'sequence' => 0,
				'site_setting_category_id' => $categoryId,
				'prevent_edit' => false,
			],
			[
				'name' => 'Adwords Conversion Label',
				'array_slug' => 'google_conversion_label',
				'value' => '',
				'type' => 'string',
				'sequence' => 0,
				'site_setting_category_id' => $categoryId,
				'prevent_edit' => false,
			],
		];
		$SettingMigrationUtil->createAllOnce(['array_slug', 'site_setting_category_id'], $settings);
	}

/**
 * Create email templates
 *
 * @return void
 */
	protected function _createEmails() {
		$GroupMigrationUtil = MigrationUtil::init('EvEmail.EmailGroup');
		$groupId = $GroupMigrationUtil->createOnce(['name' => 'EvCheckout'], ['name' => 'EvCheckout']);

		$EmailMigrationUtil = MigrationUtil::init('EvEmail.Email');
		$emails = [
			[
				'name' => 'EvCheckout Status Update Customer Paid',
				'system_name' => 'EvCheckoutStatusUpdateCustomerPaid',
				'email_group_id' => $groupId,
				'subject' => $this->SiteSetting->fieldOnly('value', ['array_slug' => 'site_title']),
				'content' => '
					<p>Dear {customerName},</p>
					<p>Thank you for your order (#{orderNumber}). See below for confirmation of what you have ordered.</p>
					<p>{==content==}</p>
				',
				'required_tokens' => 'customerName, orderNumber, ==content==',
				'optional_tokens' => '',
				'from_name' => '',
				'from_email' => '',
				'override_to' => false,
				'to_name' => null,
				'to_email' => null,
				'is_active' => true,
			],
			[
				'name' => 'EvCheckout Status Update Admin Paid',
				'system_name' => 'EvCheckoutStatusUpdateAdminPaid',
				'email_group_id' => $groupId,
				'subject' => $this->SiteSetting->fieldOnly('value', ['array_slug' => 'site_title']),
				'content' => '
					<p>A new order has been made by {customerName}</p>
					<p>The order number is (#{orderNumber}). See below for confirmation of what they have ordered.</p>
					<p>{==content==}</p>
				',
				'required_tokens' => 'customerName, orderNumber, ==content==',
				'optional_tokens' => '',
				'from_name' => '',
				'from_email' => '',
				'override_to' => false,
				'to_name' => null,
				'to_email' => null,
				'is_active' => true,
			],
			[
				'name' => 'EvCheckout Status Update Customer Shipped',
				'system_name' => 'EvCheckoutStatusUpdateCustomerShipped',
				'email_group_id' => $groupId,
				'subject' => $this->SiteSetting->fieldOnly('value', ['array_slug' => 'site_title']),
				'content' => '
					<p>Dear {customerName},</p>
					<p>Your order (#{orderNumber}) has been shipped. You can expect it to arrive in 14 days. As a reminder your order is being sent to the following address.</p>
					<p>{==content==}</p>
				',
				'required_tokens' => 'customerName, orderNumber, ==content==',
				'optional_tokens' => '',
				'from_name' => '',
				'from_email' => '',
				'override_to' => false,
				'to_name' => null,
				'to_email' => null,
				'is_active' => true,
			],
		];
		$EmailMigrationUtil->createAllOnce(['system_name'], $emails);
	}

/**
 * Create the plugin's config file
 *
 * @return void
 */
	protected function _createConfig() {
		$filepath = APP . 'Config/Plugin/ev_checkout.php';
		if (file_exists($filepath)) {
			return;
		}

		$configContents = <<<'EOF'
<?php
$config = [
	'EvCheckout' => [
		/**
		 * pageIds for success / fail pages
		 */
		'pageIds' => [
			'success' => 'order_successful',
			'fail' => 'order_failed',
			'cancel' => 'order_cancelled',
		],
	],
];
EOF;

		$File = new File($filepath, true);
		$File->write($configContents);
	}
}
