<?php

App::uses('AppComponent', 'Controller/Component');
App::uses('OrderStatus', 'EvCheckout.Model');
App::uses('BasketLib', 'EvBasket.Lib');
App::uses('RouterUtil', 'Routable.Lib');
App::uses('AddressBookLib', 'EvAddressBook.Lib');

class OrderManagerComponent extends AppComponent {

	public $order = array();

	public $orderItems = array();

	public $orderTotals = array();

	public $orderData = array();

	/**
	 * create order totals row
	 *
	 * @param   string  $name          The name of the total (label)
	 * @param   float   $amount        The amount of the total
	 * @param   int     $sequence      The sequence order number
	 * @param   float   $amountExTax   The amount of the total not including tax for display purposes
	 * @param   float   $amountIncTax  The amount of the total including tax for display purposes
	 * @return  bool
	 */
	public function setupOrderTotal(
		$name,
		$amount,
		$sequence = 10,
		$amountExTax = null,
		$amountIncTax = null
	) {
		$orderTotal = [
			'name' => $name,
			'amount' => $amount,
			'sequence' => $sequence
		];

		if (!empty($amountExTax)) {
			$orderTotal['display_ex_tax'] = $amountExTax;
		}

		if (!empty($amountIncTax)) {
			$orderTotal['display_inc_tax'] = $amountIncTax;
		}

		$this->orderTotals[$name] = $orderTotal;

		return true;
	}

/**
 * create order item row
 *
 * @param string $name            Name string or array from basket of model / model_id
 * @param int    $quantity        Number of items
 * @param float  $unitPrice       Unit Price of the item
 * @param float  $unitPriceIncTax Unit Price of the item including tax
 * @param float  $rowTotal        Row total (unitprice X quantity)
 * @param float  $rowTotalIncTax  Row total (unitprice X quantity) including tax
 * @param float  $taxRate         The tax rate (percentage) to apply
 * @param array  $orderItemData   Array of any extra data to store
 * @return bool
 */
	public function setupOrderItem(
		$name,
		$quantity,
		$unitPrice,
		$unitPriceIncTax,
		$rowTotal,
		$rowTotalIncTax,
		$taxRate,
		$model,
		$modelId,
		$orderItemData = array()
	) {
		$newOrderItem = array(
			'name' => $name,
			'quantity' => $quantity,
			'unit_price' => $unitPrice,
			'unit_price_inc_tax' => $unitPriceIncTax,
			'row_total' => $rowTotal,
			'row_total_inc_tax' => $rowTotalIncTax,
			'tax_rate' => $taxRate,
			'model' => $model,
			'model_id' => $modelId,
			'OrderItemData' => $orderItemData
		);

		$this->setupOrderItemExt($newOrderItem);

		return true;
	}

/**
 * Create order item row. Extend this method if you want to add additional data to an order item.
 *
 * @param array $orderItem Name string or array from basket of model / model_id
 * @return void.
 */
	public function setupOrderItemExt($orderItem) {
		$this->orderItems[] = $orderItem;
	}

	/**
	 * Zero tax on order items
	 * When selling to outside the EU, shops sometimes want to keep the VAT costs
	 * Should be called when setting up an order before the order is created
	 */
	public function zeroOrderItemTax() {
		$subtotal = 0;
		foreach ($this->orderItems as &$orderItem) {
			$orderItem['tax_rate'] = 0;
			$orderItem['row_total'] = $orderItem['row_total_inc_tax'];
			$orderItem['unit_price'] = $orderItem['unit_price_inc_tax'];
			$subtotal += $orderItem['row_total_inc_tax'];
		}

		// Move Delivery VAT
		$deliveryTotals = $this->orderTotals[Configure::read('EvBasket.labels.delivery')];
		$this->orderTotals[Configure::read('EvBasket.labels.delivery')]['amount'] = $deliveryTotals['display_inc_tax'];
		$this->orderTotals[Configure::read('EvBasket.labels.delivery')]['display_ex_tax'] = $deliveryTotals['display_inc_tax'];

		// Move Discount VAT
		if (isset($this->orderTotals[Configure::read('EvBasket.labels.discount')])) {
			$discountTotals = $this->orderTotals[Configure::read('EvBasket.labels.discount')];
			$this->orderTotals[Configure::read('EvBasket.labels.discount')]['amount'] = $discountTotals['display_inc_tax'];
			$this->orderTotals[Configure::read('EvBasket.labels.discount')]['display_ex_tax'] = $deliveryTotals['display_inc_tax'];
		}

		// Move Voucher VAT
		if (isset($this->orderTotals[Configure::read('EvBasket.labels.voucher')])) {
			$voucherTotals = $this->orderTotals[Configure::read('EvBasket.labels.voucher')];
			$this->orderTotals[Configure::read('EvBasket.labels.voucher')]['amount'] = $voucherTotals['display_inc_tax'];
			$this->orderTotals[Configure::read('EvBasket.labels.voucher')]['display_ex_tax'] = $deliveryTotals['display_inc_tax'];
		}

		// Update Subtotal to include Tax
		$subtotal = $this->orderTotals[Configure::read('EvBasket.labels.subtotal')]['display_inc_tax'];
		$this->orderTotals[Configure::read('EvBasket.labels.subtotal')]['amount'] = $subtotal;
		$this->orderTotals[Configure::read('EvBasket.labels.subtotal')]['display_ex_tax'] = $subtotal;

		// Clear VAT row
		$this->setupOrderTotal(Configure::read('EvBasket.labels.tax'), 0, 30, 0, 0);
	}

/**
 * Remove VAT from all order items - this can be called when orders are placed
 * by customers outside the EU when the appropriate config setting is toggled
 */
	public function removeOrderVAT() {
		$grandtotal = 0;
		$subtotal = 0;

		foreach ($this->orderItems as &$orderItem) {
			// apply the new ex vat row_total value to the subtotal calculation
			$subtotal += $orderItem['row_total'];

			// zero the tax_rate field
			$orderItem['tax_rate'] = 0;

			// Set the totals to use the item price without tax
			$orderItem['unit_price_inc_tax'] = $orderItem['unit_price'];
			$orderItem['row_total_inc_tax'] = $orderItem['row_total'];
		}

		// Remove Delivery VAT
		if (isset($this->orderTotals[Configure::read('EvBasket.labels.delivery')])) {
			$deliveryTotals = $this->orderTotals[Configure::read('EvBasket.labels.delivery')];
			// mark the inc tax and amoung fields as the same value
			$this->orderTotals[Configure::read('EvBasket.labels.delivery')]['amount'] = $deliveryTotals['display_ex_tax'];
			$this->orderTotals[Configure::read('EvBasket.labels.delivery')]['display_inc_tax'] = $deliveryTotals['amount'];
		}

		// Remove Discount VAT
		if (isset($this->orderTotals[Configure::read('EvBasket.labels.discount')])) {
			$discountTotals = $this->orderTotals[Configure::read('EvBasket.labels.discount')];
			// mark the inc tax and amoung fields as the same value
			$this->orderTotals[Configure::read('EvBasket.labels.discount')]['amount'] = $discountTotals['display_ex_tax'];
			$this->orderTotals[Configure::read('EvBasket.labels.discount')]['display_inc_tax'] = $discountTotals['amount'];
		}

		// Remove Voucher VAT
		if (isset($this->orderTotals[Configure::read('EvBasket.labels.voucher')])) {
			$voucherTotals = $this->orderTotals[Configure::read('EvBasket.labels.voucher')];
			// mark the inc tax and amoung fields as the same value
			$this->orderTotals[Configure::read('EvBasket.labels.voucher')]['amount'] = $voucherTotals['display_ex_tax'];
			$this->orderTotals[Configure::read('EvBasket.labels.voucher')]['display_inc_tax'] = $voucherTotals['amount'];
		}

		// Update Subtotal to include Tax
		if (isset($this->orderTotals[Configure::read('EvBasket.labels.subtotal')])) {
			// mark the inc tax and amoung fields as the same value
			$subtotal = $this->orderTotals[Configure::read('EvBasket.labels.subtotal')]['display_ex_tax'];
			$this->orderTotals[Configure::read('EvBasket.labels.subtotal')]['amount'] = $subtotal;
			$this->orderTotals[Configure::read('EvBasket.labels.subtotal')]['display_inc_tax'] = $this->orderTotals[Configure::read('EvBasket.labels.subtotal')]['amount'];
		}

		// Clear VAT row
		$this->setupOrderTotal(Configure::read('EvBasket.labels.tax'), 0, 30, 0, 0);

		// re-calculate and assign grand total
		$grandTotal = $this->__recalcuateGrandTotal($this->orderTotals);
		$this->setupOrderTotal(Configure::read('EvBasket.labels.grandtotal'), $grandTotal, 50);
	}

/**
 * Recalculate the grand total field for the order after all vat amount
 * @param array $orderItems Contains the items being ordered
 * @param array $orderTotals Contains the order totals
 * @return int Containing the recalculated order grand total
 */
	private function __recalcuateGrandTotal($orderTotals = array()) {
		$grandTotal = 0;

		if (! empty($orderTotals)) {
			// loop and sum the order totals, excluding the grand total field
			foreach ($orderTotals as $label => $totals) {
				if ($label != Configure::read('EvBasket.labels.grandtotal')) {
					$grandTotal += number_format($totals['amount'], 2, '.', '');
				}
			}

			if ($grandTotal < 0) {
				$grandTotal = 0;
			}
		}

		return $grandTotal;
	}

	/**
	 * generate the order item data array
	 *
	 * @param   array   $item           The array of data to search in (usual basketItemRow)
	 * @return  array   $orderItemData  Correct format for storing in the database
	 */
	public function setupOrderItemData($item) {
		$orderItemPaths = Configure::read('EvCheckout.orderItemData');
		$orderItemData = array();

		if (isset($orderItemPaths[$item['model']])) {
			foreach ($orderItemPaths[$item['model']] as $name => $path) {
				$orderItemData[] = array(
					'name' => $name,
					'item_data' => $this->findItemData($item, $path)
				);
			}
		}

		if (! empty($item['BasketItemData'])) {
			foreach ($item['BasketItemData'] as $itemData) {
				$orderItemData[] = array(
					'name' => $itemData['name'],
					'item_data' => $itemData['item_data']
				);
			}
		}

		return $orderItemData;
	}

	/**
	 * build transaction totals from order in cache
	 *
	 * @return 	array 	$transactionTotals
	 */
	public function buildTransactionTotals() {
		$grandTotal = $this->orderTotals[Configure::read('EvBasket.labels.grandtotal')];
		$subTotal = $this->orderTotals[Configure::read('EvBasket.labels.subtotal')];
		$taxTotal = $this->orderTotals[Configure::read('EvBasket.labels.tax')];
		$deliveryTotal = $this->orderTotals[Configure::read('EvBasket.labels.delivery')];
		$discount = !empty($this->orderTotals[Configure::read('EvBasket.labels.discount')]) ? $this->orderTotals[Configure::read('EvBasket.labels.discount')] : false;
		$voucher = !empty($this->orderTotals[Configure::read('EvBasket.labels.voucher')]) ? $this->orderTotals[Configure::read('EvBasket.labels.voucher')] : false;

		return array(
			'delivery' => $deliveryTotal['amount'],
			'tax' => $taxTotal['amount'],
			'subtotal' => $subTotal['amount'],
			'grandtotal' => $grandTotal['amount'],
			'discount' => (! empty($discount['amount'])) ? $discount['amount'] : false,
			'voucher' => (! empty($voucher['amount'])) ? $voucher['amount'] : false
		);
	}

	/**
	 * build transaction items from order in cache ($this->orderItems)
	 * Used In PostStage component when building the transaction
	 *
	 * @return 	array 	$transactionItems
	 */
	public function buildTransactionItems() {
		$items = array();

		foreach ($this->orderItems as $OrderItem) {
			$items[] = array(
				'model' => $OrderItem['model'],
				'model_id' => $OrderItem['model_id'],
				'description' => $OrderItem['name'],
				'amount' => $OrderItem['row_total'],
				'quantity' => $OrderItem['quantity'],
				'unitPrice' => $OrderItem['unit_price']
			);
		}

		return $items;
	}

/**
 * Build the transaction extras array from provided addresses, user and additional data.
 * The totals are found and added to the extras as well.
 *
 * @param array $addresses The delivery and billing addresses to be used for this transaction
 * @param array $user      The current user to be used for this transaction
 * @param array $data      Additional provided data that can be used to add custom extras
 * @return array The array to be used as transaction extras
 */
	public function buildTransactionExtras($addresses, $user, $data) {
		return [
			'totals' => $this->buildTransactionTotals(),
			'delivery' => (!empty($addresses['delivery'])) ? $addresses['delivery'] : null,
			'user' => $user
		];
	}

	/**
	 * given the array of data and the path - extract the data we want to save
	 *
	 * @param   array   $itemData   Item data to search in
	 * @param   string  $path       The path to search data for
	 * @return  mixed|bool
	 */
	public function findItemData($itemData, $path) {
		return Hash::get($itemData, $path, false);
	}

	/**
	 * sets the order data rows up within the model so it can be used later when saving
	 * @param  string   $name   name of the data record
	 * @param  string   $data   value of the data record
	 * @return void
	 */
	public function setupOrderData($name, $data, $isVisible = false) {
		$this->orderData[] = array(
			'name' => $name,
			'data' => $data,
			'is_visible' => $isVisible
		);
	}

	/**
	 * create main order details
	 *
	 * @param   array   $user           Array of user data
	 * @param   array   $addresses      Array of address data
	 * @param   int     $orderStatusId  Id of the order status to set
	 * @param   bool    $isPaid         Whether the order has been paid or not
	 * @return  bool
	 */
	public function setupOrder($user, $addresses, $currencyId, $orderStatusId = null, $isPaid = false) {
		// get a default status
		if ($orderStatusId === null) {
			$OrderStatus = EvClassRegistry::init('EvCheckout.OrderStatus');
			$orderStatusId = $OrderStatus->getConstant('UNPAID');
		}

		$this->order = array(
			'name' => (! empty($user['name'])) ? $user['name'] : '',
			'email' => (! empty($user['email'])) ? $user['email'] : '',
			'phone' => (! empty($user['phone'])) ? $user['phone'] : '',
			'billing_address' => (! empty($addresses['billing'])) ? AddressBookLib::display($addresses['billing']) : '',
			'delivery_address' => (! empty($addresses['delivery'])) ? AddressBookLib::display($addresses['delivery']) : '',
			'order_status_id' => $orderStatusId,
			'is_paid' => $isPaid,
			'is_guest_checkout' => (isset($user['is_guest_user'])) ? $user['is_guest_user'] : '1',
			'user_id' => (! empty($user['id'])) ? $user['id'] : null,
			'currency_id' => $currencyId
		);

		if (!empty($addresses['delivery'])) {
			$this->setupOrderData('delivery_address', json_encode($addresses['delivery']));
		}

		if (!empty($addresses['billing'])) {
			$this->setupOrderData('billing_address', json_encode($addresses['billing']));
		}

		return true;
	}

	/**
	 * save the order we've built up
	 *
	 * @param   object      $Order      Order model object
	 * @return  int|bool    $orderId    Order ID or false on fail
	 */
	public function saveOrder($Order) {
		$result = $Order->saveAll(
			array(
				'Order' => $this->order,
				'OrderItem' => $this->orderItems,
				'OrderTotal' => $this->orderTotals,
				'OrderData' => $this->orderData
			),
			array(
				'deep' => true
			)
		);

		if ($result === true) {
			return $Order->id;
		}

		return false;
	}

	/**
	 * mark the order as paid
	 *
	 * @param   int     $orderId    The order ID to mark as paid
	 * @return  bool
	 */
	public function markAsPaid($orderId) {
		$Order = EvClassRegistry::init('EvCheckout.Order');

		if ($Order->paid($orderId) !== false) {
			// dispatch order paid event
			$this->_controller->getEventManager()->dispatch(
				new CakeEvent('EvCheckout.Component.Order.paid', $this, array(
					'orderId' => $orderId
				))
			);

			return true;
		}

		return false;
	}

	/**
	 * update the orders status
	 *
	 * @param   int     $orderId    The order ID to update
	 * @param   int     $statusId   The status ID
	 * @return  bool
	 */
	public function updateStatus($orderId, $statusId) {
		$Order = EvClassRegistry::init('EvCheckout.Order');

		if ($Order->updateStatus($orderId, $statusId) !== false) {
			// dispatch order paid event
			$this->_controller->getEventManager()->dispatch(
				new CakeEvent('EvCheckout.Component.Order.statusChanged', $this, array(
					'orderId' => $orderId,
					'statusId' => $statusId
				))
			);

			return true;
		}

		return false;
	}

	/**
	 * return the order complete url
	 *
	 * @return string   $orderCompleteUrl
	 */
	public function getCompleteUrl() {
		return Router::url(
			RouterUtil::getListingRoute('EvCheckout', 'Order', 'complete'),
			true
		);
	}

	/**
	 * get the order cancelled url
	 *
	 * @return string   $orderCancelledUrl
	 */
	public function getCancelledUrl() {
		return Router::url(
			RouterUtil::getListingRoute('EvCheckout', 'Order', 'cancel'),
			true
		);
	}

	/**
	 * build the order from the given transactionId
	 *
	 * @param   int         $transactionId      The successful transaction ID just completed
	 * @return  int|bool    $orderId            The order we've just built or false on fail
	 */
	public function buildOrder($transactionId) {
		$selectedBuilder = Configure::read('EvCheckout.orderBuilder');

		$OrderBuilder = $this->loadBuilder($selectedBuilder);

		if (is_object($OrderBuilder)) {
			return $OrderBuilder->build($transactionId);
		}

		return false;
	}

	/**
	 * load the builder to use
	 *
	 * @param   string          $selectedBuilder    The selected builder to use
	 * @return  object|bool     $OrderBuilder       The builder object
	 */
	public function loadBuilder($selectedBuilder) {
		list($plugin, $builder) = pluginSplit($selectedBuilder, true, '');

		App::uses($builder, $plugin . 'OrderBuilders');

		if (class_exists($builder)) {
			return new $builder($this->_controller);
		}

		return false;
	}

	/**
	 * assign the order to the original transaction
	 *
	 * @param   int     $orderId        The order ID
	 * @param   int     $transactionId  The transaction ID
	 * @return  bool
	 */
	public function assignToTransaction($orderId, $transactionId) {
		$Transaction = EvClassRegistry::init('EvTransactions.Transaction');

		return $Transaction->save(
			array(
				'Transaction' => array(
					'id' => $transactionId,
					'model' => 'EvCheckout.Order',
					'model_id' => $orderId
				)
			)
		);
	}

	/**
	 * Takes an order id and reduces the stock for all the order items
	 * @param int $orderId 	The ID of the order
	 * @return  bool
	 */
	public function reduceStockForOrder($orderId) {
		$Order = EvClassRegistry::init('EvCheckout.Order');
		$orderDetails = $Order->readForEdit($orderId);

		foreach ($orderDetails['OrderItem'] as $item) {
			// If there is no model or model_id we can't reduce stock
			if (empty($item['model']) || empty($item['model_id'])) {
				continue;
			}
			// dispatch event to lower stock levels
			$this->_controller->getEventManager()->dispatch(
				new CakeEvent('EvInventory.Inventory.reduce', $this, array(
					'model' => $item['model'],
					'model_id' => $item['model_id'],
					'quantity' => $item['quantity']
				))
			);
		}
		return true;
	}

	/**
	 * send order confirmation emails
	 *
	 * @param   int     $orderId    The order ID
	 * @return  bool
	 */
	public function sendOrderConfirmEmails($orderId) {
		$QueuedTask = EvClassRegistry::init('Queue.QueuedTask');

		$defaultHelpers = [
			'Html',
			'Number'
		];

		$helpers = Hash::merge($defaultHelpers, Configure::read('EvCheckout.orderEmailHelpers'));

		$Order = EvClassRegistry::init('EvCheckout.Order');
		$orderDetails = $Order->readForEdit($orderId);

		$orderCurrency = EvClassRegistry::init('EvCurrency.Currency')->find(
			'first',
			[
				'conditions' => [
					'Currency.id' => $orderDetails['Order']['currency_id']
				]
			]
		);

		// queue the confirmation email to customer
		$QueuedTask->createJob(
			'CustomEmail',
			array(
				'from' => array(
					Configure::read('SiteSetting.general.admin_email') => Configure::read('SiteSetting.general.site_title')
				),
				'to' => array(
					$orderDetails['Order']['email'] => $orderDetails['Order']['name']
				),
				'subject' => $this->_customerEmailSubject($orderDetails),
				'viewVars' => array(
					'order' => $orderDetails,
					'orderCurrency' => $orderCurrency,
				),
				'helpers' => $helpers,
				'template' => 'EvCheckout.Orders/customerNewOrder'
			)
		);

		// queue the confirmation email to admin
		$manageOrder = $this->_controller->Routable->getListingRoute(
			'EvCheckout',
			'Order',
			'edit'
		);
		$manageOrder[] = $orderId;
		$manageOrder['admin'] = true;

		$QueuedTask->createJob(
			'CustomEmail',
			array(
				'from' => array(
					Configure::read('SiteSetting.general.admin_email') => Configure::read('SiteSetting.general.site_title')
				),
				'to' => array(
					Configure::read('SiteSetting.ev_checkout.new_order_email') => Configure::read('SiteSetting.ev_checkout.new_order_name')
				),
				'subject' => $this->_adminEmailSubject($orderDetails),
				'viewVars' => array(
					'order' => $orderDetails,
					'orderCurrency' => $orderCurrency,
					'manageOrderUrl' => Router::url($manageOrder, true)
				),
				'helpers' => $helpers,
				'template' => 'EvCheckout.Orders/adminNewOrder'
			)
		);

		return true;
	}

	/**
	 * Get the subject for the customer order email
	 */
	protected function _customerEmailSubject($order) {
		return __('Order Confirmation #%s', array($order['Order']['id']));
	}

	/**
	 * Get the subject for the admin order email
	 */
	protected function _adminEmailSubject($order) {
		return __('New Order #%s', array($order['Order']['id']));
	}

	/**
	 * Send an email to an individual and using the template set in the config.
	 * @param  int $orderId  The order id
	 * @param  int $statusId The status id
	 * @return bool
	 */
	public function sendStatusUpdateEmail($orderId, $statusId) {
		//Get the order and the status using the ids supplied
		$Order = EvClassRegistry::init('EvCheckout.Order');
		$Status = EvClassRegistry::init('EvCheckout.OrderStatus');

		$orderDetails = $Order->readForEdit($orderId);

		$statusDetails = $Status->find('first', array(
			'conditions' => array('OrderStatus.id' => $statusId)
		));

		$configStatusEmails = Configure::read('EvCheckout.statusUpdateEmails');
		if (empty($configStatusEmails)) {
			return null;
		}

		//loop through the config statueses to see if the current status has any emails to send
		foreach ($configStatusEmails as $statusName => $emails) {
			if ($statusName == $statusDetails['OrderStatus']['name']) {
				//Setup and send each email
				foreach ($emails as $emailIndex => $email) {
					$emailTokens = $this->getDefaultEmailTokens($orderDetails);

					$helpers = [];
					if (!empty(Configure::read('EvCheckout.orderEmailHelpers'))) {
						// Merge generic checkout email helpers into the available helpers for this email
						$helpers = Hash::merge($helpers, Configure::read('EvCheckout.orderEmailHelpers'));
					}

					if (!empty($email['helpers'])) {
						// Merge checkout status template specific email helpers into the available helpers for this email
						$helpers = Hash::merge($helpers, $email['helpers']);
					}

					if (isset($email['viewTemplates']) && $email['viewTemplates'] !== null) {
						$emailTokens['content'] = $this->generateOrderEmailContent($orderDetails, $email['viewTemplates'], $helpers);

						//Add tokens defined in the config to the tokens array
						foreach ($email['tokens'] as $tokenName => $tokenValue) {
							$emailTokens[$tokenName] = Hash::get($orderDetails, $tokenValue);
						}
					}

					$EvEmail = ClassRegistry::init('EvEmail.Email');
					$newEmail = $EvEmail->generateEmailData(
						// Either pass the integer ID of the template or the template's `system_name`
						$email['emailTemplate'],
						$emailTokens
					);

					$newEmail['Email']['headContent'] = null;
					if (isset($email['headViewTemplates']) && $email['headViewTemplates'] !== null) {
						$newEmail['Email']['headContent'] = $this->generateOrderEmailContent($orderDetails, $email['headViewTemplates'], $helpers);
					}

					$emailTarget = [$orderDetails['Order']['email'] => $orderDetails['Order']['name']];
					if (isset($email['target'])) {
						$newTarget = $this->getEmailTarget($email['target']);

						if (!empty($newTarget)) {
							$emailTarget = $newTarget;
						}
					}

					$newEmail['Email']['cc'] = null;
					if (!empty($email['cc'])) {
						$newEmail['Email']['cc'] = $email['cc'];
					}

					$newEmail['Email']['bcc'] = null;
					if (!empty($email['bcc'])) {
						$newEmail['Email']['bcc'] = $email['bcc'];
					}

					//Add email to queue
					$EvEmail->queueEmail(
						$newEmail['Email']['subject'],
						$newEmail['Email']['content'],
						$emailTarget,
						$newEmail['Email']['from'],
						null,
						$newEmail['Email']['cc'],
						$newEmail['Email']['bcc'],
						null,
						$helpers,
						$newEmail['Email']['headContent'],
						!empty($email['viewLayout']) ? $email['viewLayout'] : null
					);
				}
			}
		}

		return true;
	}

	/**
	 * Get the default tokens for an EvCheckout email.
	 * In seperate function for extendability? Haven't done this before so I dunno
	 * @param  array $orderData An array containing the order data
	 * @return array            array of tokens that each email contains
	 */
	public function getDefaultEmailTokens($orderData) {
		return [
			'content' => ' ',
			'customerName' => $orderData['Order']['name'],
			'orderNumber' => $orderData['Order']['id']
		];
	}

	/**
	 * Generate html content for an order email. Content that needs to be formatted e.g. tables
	 * is best to be added here using a template.
	 * @param  array $orderDetails The details about an order that will be supplied to each template
	 * @param  array $templates    An array of strings that contain each template to append to the email.
	 * @return string              The full html string containing the rendered templates.
	 */
	public function generateOrderEmailContent($orderDetails, $templates, $helpers = null) {
		$content = '';

		//Create email content from view templates supplied from the config
		$Controller = new Controller();

		if (!empty($helpers)) {
			$Controller->helpers = $helpers;
		}

		//Get manageOrderUrl
		$manageOrderUrl = $this->_controller->Routable->getListingRoute(
			'EvCheckout',
			'Order',
			'edit'
		);
		$manageOrderUrl[] = $orderDetails['Order']['id'];
		$manageOrderUrl['admin'] = true;

		$content = '';

		foreach ($templates as $template) {
			$View = new View($Controller);
			$View->set('order', $orderDetails);

			$View->set(
				'orderCurrency',
				EvClassRegistry::init('EvCurrency.Currency')->find(
					'first',
					[
						'conditions' => [
							'Currency.id' => $orderDetails['Order']['currency_id']
						]
					]
				)
			);

			$View->set('manageOrderUrl', Router::url($manageOrderUrl, true));

			$content .= $View->render($template, 'ajax');
		}

		return $content;
	}

	/**
	 * Get the "to" data for the email. Pretty simple at the moment but put it in it's own
	 * function so that it can be overridden if there are more types of users that will
	 * require their own emails.
	 * @param  string $target The string description of a user group.
	 * @return array          An array containing the target email and name
	 */
	public function getEmailTarget($target) {
		//If target is admin then get their admin email from general site settings
		$returnAddress = array();
		if ($target == 'admin') {
			$returnAddress = [Configure::read('SiteSetting.ev_checkout.new_order_email') => Configure::read('SiteSetting.ev_checkout.new_order_name')];
		}

		return $returnAddress;
	}

	/**
	 * get all the orders belonging to a user
	 *
	 * @param   int     $userId     User id number
	 * @return  array   $orders     Paginated users order
	 */
	public function getOrdersByUser($userId, $params = []) {
		$OrderModel = EvClassRegistry::init('EvCheckout.Order');

		$defaultParams = [
			'conditions' => array(
				$OrderModel->alias . '.user_id' => $userId
			),
			'contain' => array(
				'OrderItem',
				'OrderTotal',
				'OrderStatus'
			),
			'order' => $OrderModel->alias . '.id DESC',
			'limit' => 10
		];

		$this->_controller->Paginator->settings = Hash::merge($defaultParams, $params);

		return $this->_controller->Paginator->paginate(
			$OrderModel
		);
	}
}
