<?php

App::uses('EvCheckoutAppController', 'EvCheckout.Controller');
App::uses('ArrayUtil', 'EvCore.Lib');

class OrdersController extends EvCheckoutAppController {

	public function beforeFilter() {
		$this->adminActions[] = 'admin_pay';
		$this->adminActions[] = 'admin_status';
		$this->adminActions[] = 'admin_invoice';

		parent::beforeFilter();

		$this->Auth->allow(array('complete', 'cancel'));
	}

	/**
	 * complete an order
	 * this could display success or fail messages.
	 * This is where the payment is checked and turned into an order
	 *
	 */
	public function complete() {
		if (CakePlugin::loaded('EvTransactions') && ! empty($this->request->query['transaction'])) {
			$this->TransactionsManager = $this->loadComponent('EvTransactions.Transactions');
			$this->OrderManager = $this->loadComponent('EvCheckout.OrderManager');

			if ($this->TransactionsManager->checkPayment()) {
				// build the order. OrderBuilder is specified within Config
				// builds from an EvBasket Row as default
				$orderId = $this->OrderManager->buildOrder($this->request->query['transaction']);

				// trigger the success hook
				$this->getEventManager()->dispatch(
					new CakeEvent('EvCheckout.Controller.Order.success', $this, array(
						'orderId' => $orderId
					))
				);

				$this->view = 'EvCheckout./Fallbacks/Order/success';
				$pageId = Configure::read('EvCheckout.pageIds.success');
			} else {

				// trigger the fail hook
				$this->getEventManager()->dispatch(
					new CakeEvent('EvCheckout.Controller.Order.fail', $this)
				);

				$this->view = 'EvCheckout./Fallbacks/Order/fail';
				$pageId = Configure::read('EvCheckout.pageIds.fail');
			}
		}

		if (empty($this->view)) {
			// trigger the fail hook
			$this->getEventManager()->dispatch(
				new CakeEvent('EvCheckout.Controller.Order.fail', $this)
			);

			$this->view = 'EvCheckout./Fallbacks/Order/fail';
			$pageId = Configure::read('EvCheckout.pageIds.fail');
		}

		// check if we have a specific content page set
		// if so - set the template
		if (! empty($pageId)) {
			$pageData = $this->assignPage($pageId);

			if (! empty($pageData)) {
				if (! empty($orderId)) {
					$pageData['Page']['title'] = str_replace('ORDERID', $orderId, $pageData['Page']['title']);
					$pageData['Page']['body'] = str_replace('ORDERID', $orderId, $pageData['Page']['body']);
					$this->set('data', $pageData);
				}

				$this->view = $this->Tpl->getTemplate($pageData, 'Page');
			}
		}
	}

	/**
	 * cancel an order.
	 * usually used on gateways for the "cancel and return to store" links
	 *
	 */
	public function cancel() {
		if (CakePlugin::loaded('EvTransactions') && ! empty($this->request->query['transaction'])) {
			$this->TransactionsManager = $this->loadComponent('EvTransactions.Transactions');

			/*$this->TransactionsManager->failTransaction(
				$this->request->query['transaction'],
				'The order was cancelled by the user'
			);*/
		}

		// trigger the cancelled hook
		$this->getEventManager()->dispatch(
			new CakeEvent('EvCheckout.Controller.Order.cancel', $this)
		);

		$this->view = 'EvCheckout./Fallbacks/Order/cancel';
		$pageId = Configure::read('EvCheckout.pageIds.cancel');

		// check if we have a specific content page set
		// if so - set the template
		if (! empty($pageId)) {
			$pageData = $this->assignPage($pageId);

			if (! empty($pageData)) {
				$this->view = $this->Tpl->getTemplate($pageData, 'Page');
			}
		}
	}

	/**
	 * redefine index columns
	 * to add more order info
	 */
	protected function _adminIndexColumns() {
		$columns = parent::_adminIndexColumns();

		$columns = ArrayUtil::addAfter(
			$columns,
			'Order.name',
			array(
				'Order.email' => array(
					'type' => 'string',
					'label' => __('Email')
				),
				'Order.phone' => array(
					'type' => 'string',
					'label' => __('Phone')
				),
				'OrderStatus.name' => array(
					'type' => 'string',
					'label' => __('Status')
				),
				'Order.is_paid' => array(
					'type' => 'display_boolean',
					'label' => __('Paid')
				),
				'Currency.name' => array(
					'type' => 'string',
					'label' => __('Currency')
				)
			)
		);

		unset($columns['Order.modified']);
		return $columns;
	}

	/*
		----------------------------------------------------------------
		Admin methods
		----------------------------------------------------------------
	*/

	/**
	 * redefine to remove the add button
	 *
	 */
	protected function _adminIndexToolbar() {
		return array();
	}

	/**
	 * redefine index paginate
	 * to contain status
	 */
	protected function _adminIndexPaginate() {
		$paginate = parent::_adminIndexPaginate();

		$paginate['contain'][] = 'OrderStatus';
		$paginate['contain'][] = 'Currency';
		$paginate['order'] = 'Order.id DESC';

		return $paginate;
	}

	/**
	 * define index filters
	 * to make it more usable
	 *
	 */
	protected function _adminFilterFields() {
		$fields = parent::_adminFilterFields();

		$fields = ArrayUtil::addAfter(
			$fields,
			'Order.name',
			array(
				'Order.email' => array(
					'label' => __('Email'),
					'type' => 'string',
					'compare' => array(
						'Order.email LIKE' => '%%%s%%'
					)
				),
				'Order.order_status_id' => array(
					'label' => __('Status'),
					'type' => 'select',
					'compare' => array(
						'Order.order_status_id' => '%s'
					)
				),
				'Order.is_paid' => array(
					'label' => __('Paid'),
					'type' => 'select',
					'compare' => array(
						'Order.is_paid' => '%s'
					)
				),
				'Order.currency_id' => array(
					'label' => __('Currency'),
					'type' => 'select',
					'compare' => array(
						'Order.currency_id' => '%s'
					)
				)
			)
		);

		unset($fields['Order.modified']);

		return $fields;
	}

	/**
	 * redefine filter populate
	 * to bring in order status list
	 *
	 */
	protected function _adminPopulateFilterLookups() {
		$OrderStatus = EvClassRegistry::init('EvCheckout.OrderStatus');

		$this->set(
			'orderStatuses',
			$OrderStatus->getForDropDown()
		);

		$this->set(
			'isPaids',
			array(
				'1' => __('Yes'),
				'0' => __('No')
			)
		);
	}

	/**
	 * redefine filter populate
	 * to bring in order status list
	 *
	 */
	protected function _adminPopulateLookups() {
		$OrderStatus = EvClassRegistry::init('EvCheckout.OrderStatus');

		$this->set(
			'orderStatuses',
			$OrderStatus->getForDropDown()
		);
	}

	/**
	 * redefine to remove the add button
	 *
	 */
	protected function _adminFormToolbar($id = null) {
		$toolbar = parent::_adminFormToolbar($id);

		unset($toolbar['Add New']);

		return $toolbar;
	}

	/**
	 * redefine the admin edit to have a nicer order management screen
	 *
	 * @param 	int 	$id 	The order ID
	 */
	public function admin_edit($id = null) {
		if ($id === null) {
			$this->Flash->fail(
				array(
					'title' => 'Error',
					'description' => 'No order was selected for viewing.'
				)
			);

			$this->redirect(
				array('action' => 'index')
			);
		}

		parent::admin_edit($id);

		$this->view = 'EvCheckout.Orders/admin_edit';
	}

	/**
	 * mark an order as paid
	 *
	 * @param 	int 	$id 	the order ID to mark as paid
	 */
	public function admin_pay($id) {
		$this->OrderManager = $this->loadComponent('EvCheckout.OrderManager');

		if ($this->OrderManager->markAsPaid($id)) {
			$this->Flash->success(
				array(
					'title' => 'Paid',
					'description' => 'The order (#' . $id . ') has been marked as paid'
				)
			);
		} else {
			$this->Flash->fail(
				array(
					'title' => 'Failed',
					'description' => 'There was a problem marking the order (#' . $id . ') as paid'
				)
			);
		}

		$this->redirect(
			array(
				'action' => 'edit',
				$id
			)
		);
	}

	/**
	 * change an orders status
	 *
	 * @param 	int 	$id 	the order ID to change
	 */
	public function admin_status($id) {
		$this->OrderManager = $this->loadComponent('EvCheckout.OrderManager');

		$statusSet = ! empty($this->request->data['Order']['order_status_id']);

		if ($statusSet && $this->OrderManager->updateStatus($id, $this->request->data['Order']['order_status_id'])) {
			$this->Flash->success(
				array(
					'title' => 'Status Changed',
					'description' => 'The order status for #' . $id . ' has been updated'
				)
			);
		} else {
			$this->Flash->fail(
				array(
					'title' => 'Failed',
					'description' => 'There was a problem updating the status for order #' . $id
				)
			);
		}

		$this->redirect(
			array(
				'action' => 'edit',
				$id
			)
		);
	}

	/**
	 * view printable picking / invoice list
	 *
	 * @param 	int 	$id 	the order ID to view
	 */
	public function admin_invoice($id) {
		parent::admin_edit($id);

		$this->Meta->set(
			array(
				'Page' => array(
					'title' => 'Order #' . $id . ' Invoice',
				)
			),
			'EvCore.Page'
		);

		$currencyModel = EvClassRegistry::init('EvCurrency.Currency');
		$this->set('currencyList', $currencyModel->getForDropdown());

		$this->view = 'EvCheckout.Orders/admin_invoice';
		$this->layout = 'print';
	}

	/**
	 * frontend - show users order history
	 *
	 */
	public function history() {
		$this->OrderManager = $this->loadComponent('EvCheckout.OrderManager');

		$this->set(
			'orders',
			$this->OrderManager->getOrdersByUser($this->Auth->user('User.id'))
		);

		$this->Meta->set(
			array(
				'Page' => array(
					'title' => 'Order History'
				)
			),
			'EvCore.Page'
		);

		$this->view = 'EvCheckout.Orders/history';
	}

	/**
	 * frontend - show the user their order
	 *
	 * @param 	int 	$orderId 	The id of the order to view
	 */
	public function view($id) {
		$OrderModel = EvClassRegistry::init('EvCheckout.Order');
		$order = $OrderModel->readForView($id);

		if (! empty($order['Order']['user_id']) && (int)$order['Order']['user_id'] !== (int)$this->Auth->user('User.id')) {
			$route = $this->Routable->getItemRoute('EvCheckout', 'Order');
			$route['action'] = 'history';

			$this->Flash->fail('You do not have permission to view that order');

			$this->redirect($route);
		}

		$this->set(
			'order',
			$order
		);

		$this->Meta->set(
			array(
				'Page' => array(
					'title' => 'View Order #' . $id
				)
			),
			'EvCore.Page'
		);

		$this->view = 'EvCheckout.Orders/view';
	}
}
