<?php

App::uses('AppComponent', 'Controller/Component');

class PreStageComponent extends AppComponent {

	/**
	 * check to see if a user is already logged in
	 * used to skip to the next stage if the user is logged in
	 *
	 * @return bool
	 */
	public function alreadyLoggedIn() {
		if ($this->_controller->Auth->user() !== null) {
			$this->_controller->redirect(
				$this->_controller->calculatePostStageRedirect(
					$this->_controller->activeStage
				)
			);
		}

		return true;
	}

	/**
	 * authorise a user
	 *
	 * @return 	bool
	 */
	public function authUser() {
		if ($this->_controller->Auth->user() !== null) {
			return true;
		}

		return false;
	}

	/**
	 * attempt to load a logged in users addresses
	 *
	 * @return bool
	 */
	public function loadAddresses() {
		if (! CakePlugin::loaded('EvAddressBook')) {
			$this->_controller->Flash->fail('EvAddressBook is not loaded. Cannot retrieve addresses');
			return false;
		}

		if (empty($this->_controller->AddressManager)) {
			$this->_controller->AddressManager = $this->_controller->loadComponent('EvAddressBook.AddressManager');
		}

		// laod the address book helper to aid with formatting of addresses
		$this->_controller->helpers['AddressBook'] = array(
			'className' => 'EvAddressBook.AddressBook'
		);

		$this->_controller->set(
			'addressBook',
			$this->_controller->AddressManager->getAddresses(
				EvClassRegistry::init('EvAddressBook.Address'),
				'EvCore.User',
				$this->_controller->Auth->user('User.id')
			)
		);

		// get the countries for add address dropdown
		$Country = EvClassRegistry::init('EvCountry.Country');
		$this->_controller->set(
			'countries',
			$Country->getForDropDown()
		);

		return true;
	}

	/**
	 * load the shipping plugin and try to auto calculate any shipping
	 *
	 * @return 	bool
	 */
	public function processBasketShippingFromAddress() {
		if (! CakePlugin::loaded('EvBasket')) {
			$this->_controller->Flash->fail('EvBasket is not loaded. Cannot add shipping to basket');
			return false;
		}

		if (! CakePlugin::loaded('EvAddressBook')) {
			$this->_controller->Flash->fail('EvAddressBook is not loaded. Cannot retrieve address');
			return false;
		}

		if (! CakePlugin::loaded('EvShipping')) {
			$this->_controller->Flash->fail('EvShipping is not loaded. Cannot calculate shipping');
			return false;
		}

		// load the mbasket manager so we can get the basket
		if (empty($this->_controller->BasketManager)) {
			$this->_controller->BasketManager = $this->_controller->loadComponent('EvBasket.Basket');
		}

		// load the shipping manager so we can calculate shipping
		if (empty($this->_controller->ShippingManager)) {
			$this->_controller->ShippingManager = $this->_controller->loadComponent('EvShipping.ShippingManager');
		}

		// load the address manager component
		if (empty($this->_controller->AddressManager)) {
			$this->_controller->AddressManager = $this->_controller->loadComponent('EvAddressBook.AddressManager');
		}

		$basket = $this->_controller->BasketManager->getBasket();
		$grandTotalRow = $this->_controller->BasketManager->findTotalRow(
			Configure::read('EvBasket.labels.subtotal')
		);

		$this->_controller->ShippingManager->addData('basket', $basket);
		$this->_controller->ShippingManager->addData(
			'total-amount',
			$grandTotalRow['amount']
		);

		// find the addresses to save to the transaction
		$addresses = $this->_controller->AddressManager->findAddresses();

		if (empty($addresses['delivery'])) {
			$this->_controller->Flash->fail(
				array(
					'title' => 'No Delivery Address',
					'description' => 'No delivery address could be found for calculating shipping, please go back and make sure you have created or select a delivery address'
				)
			);
			return false;
		}

		$rate = $this->_controller->ShippingManager->calculateShippingFromAddress(
			$addresses['delivery']['Address']
		);

		if ($rate !== false) {
			$this->_controller->BasketManager->manageTotalRow(
				Configure::read('EvBasket.labels.delivery'),
				$rate
			);

			$this->_controller->BasketManager->rebuildTotals();
		}

		return true;
	}

	/**
	 * load EvBasket and load a basket summary
	 * for the user to confirm before payment
	 *
	 * @return 	bool
	 */
	public function loadBasket() {
		if (! CakePlugin::loaded('EvBasket')) {
			$this->_controller->Flash->fail('EvBasket is not loaded. Cannot retrieve basket');
			return false;
		}

		$return = true;

		// load the mbasket manager so we can get the basket
		if (empty($this->_controller->BasketManager)) {
			$this->_controller->BasketManager = $this->_controller->loadComponent('EvBasket.Basket');
		}

		// laod the basket heper incase any of it is in use (it should be!)
		$this->_controller->helpers['Basket'] = array(
			'className' => 'EvBasket.Basket'
		);

		$this->_controller->set(
			'basket',
			$this->_controller->BasketManager->getBasket(true)
		);

		return $return;
	}

	/**
	 * load the selected address (if any)
	 * and set to template for displaying
	 *
	 * @return bool
	 */
	public function loadSelectedAddresses() {
		if (! CakePlugin::loaded('EvAddressBook')) {
			$this->_controller->Flash->fail('EvAddressBook is not loaded. Cannot retrieve addresses');
			return false;
		}

		// load the address manager component
		if (empty($this->_controller->AddressManager)) {
			$this->_controller->AddressManager = $this->_controller->loadComponent('EvAddressBook.AddressManager');
		}

		// laod the address book helper to aid with formatting of addresses
		$this->_controller->helpers['AddressBook'] = array(
			'className' => 'EvAddressBook.AddressBook'
		);

		// read for the session and process each id
		$addressIds = $this->_controller->Session->read('EvCheckout.address');
		$addresses = array();

		foreach ($addressIds as $key => $addressId) {
			$addresses[$key] = $this->_controller->AddressManager->getAddress(
				EvClassRegistry::init('EvAddressBook.Address'),
				$addressIds[$key]
			);
		}

		$this->_controller->set(
			'selectedAddresses',
			$addresses
		);

		return true;
	}
}
