<?php

App::uses('AppComponent', 'Controller/Component');
App::uses('OrderStatus', 'EvCheckout.Model');
App::uses('BasketLib', 'EvBasket.Lib');
App::uses('RouterUtil', 'Routable.Lib');

class OrderManagerComponent extends AppComponent {

	public $order = array();

	public $orderItems = array();

	public $orderTotals = array();

	/**
	 * create order totals row
	 *
	 * @param 	string 	$name 		The name of the total (label)
	 * @param 	float 	$amount 	The amount of the total
	 * @param 	int 	$sequence 	The sequence order number
	 * @return 	bool
	 */
	public function setupOrderTotal($name, $amount, $sequence = 10) {
		$this->orderTotals[$name] = array(
			'name' => $name,
			'amount' => $amount,
			'sequence' => $sequence
		);

		return true;
	}

	/**
	 * create order item row
	 *
	 * @param 	string 			$name 			Name string or array from basket of model / model_id
	 * @param 	int 			$quantity 		Number of items
	 * @param 	float 			$unitPrice 		Unit Price of the item
	 * @param 	float 			$rowTotal 		Row total (unitprice X quantity)
	 * @param 	float 			$taxRate 		The tax rate (percentage) to apply
	 * @param 	array 			$orderItemData 	Array of any extra data to store
	 * @return 	bool
	 */
	public function setupOrderItem($name, $quantity, $unitPrice, $rowTotal, $taxRate, $orderItemData = array()) {
		$this->orderItems[] = array(
			'name' => $name,
			'quantity' => $quantity,
			'unit_price' => $unitPrice,
			'row_total' => $rowTotal,
			'tax_rate' => $taxRate,
			'OrderItemData' => $orderItemData
		);

		return true;
	}

	/**
	 * generate the order item data array
	 *
	 * @param 	array 	$item 			The array of data to search in (usual basketItemRow)
	 * @return 	array 	$orderItemData 	Correct format for storing in the database
	 */
	public function setupOrderItemData($item) {
		$orderItemPaths = Configure::read('EvCheckout.orderItemData');
		$orderItemData = array();

		if (isset($orderItemPaths[$item['model']])) {
			foreach ($orderItemPaths[$item['model']] as $name => $path) {
				$orderItemData[] = array(
					'name' => $name,
					'item_data' => $this->findItemData($item, $path)
				);
			}
		}

		return $orderItemData;
	}

	/**
	 * given the array of data and the path - extract the data we want to save
	 *
	 * @param 	array 	$itemData 	Item data to search in
	 * @param 	string 	$path 		The path to search data for
	 * @return  mixed|bool
	 */
	public function findItemData($itemData, $path) {
		return Hash::get($itemData, $path, false);
	}

	/**
	 * create main order details
	 *
	 * @param 	array 	$user 			Array of user data
	 * @param 	array 	$addresses 		Array of address data
	 * @param 	int 	$orderStatusId 	Id of the order status to set
	 * @param 	bool 	$isPaid 		Whether the order has been paid or not
	 * @return  bool
	 */
	public function setupOrder($user, $addresses, $currencyId, $orderStatusId = null, $isPaid = false) {
		// get a default status
		if ($orderStatusId === null) {
			$OrderStatus = EvClassRegistry::init('EvCheckout.OrderStatus');
			$orderStatusId = $OrderStatus->getConstant('UNPAID');
		}

		$this->order = array(
			'name' => (! empty($user['name'])) ? $user['name'] : '',
			'email' => (! empty($user['email'])) ? $user['email'] : '',
			'phone' => (! empty($user['phone'])) ? $user['phone'] : '',
			'billing_address' => (! empty($addresses['billing'])) ? $addresses['billing'] : '',
			'delivery_address' => (! empty($addresses['delivery'])) ? $addresses['delivery'] : '',
			'order_status_id' => $orderStatusId,
			'is_paid' => $isPaid,
			'is_guest_checkout' => (isset($user['is_guest_user'])) ? $user['is_guest_user'] : '1',
			'user_id' => (! empty($user['id'])) ? $user['id'] : null,
			'currency_id' => $currencyId
		);

		return true;
	}

	/**
	 * save the order we've built up
	 *
	 * @param 	object 		$Order 		Order model object
	 * @return 	int|bool 	$orderId 	Order ID or false on fail
	 */
	public function saveOrder($Order) {
		$result = $Order->saveAll(
			array(
				'Order' => $this->order,
				'OrderItem' => $this->orderItems,
				'OrderTotal' => $this->orderTotals
			),
			array(
				'deep' => true
			)
		);

		if ($result === true) {
			return $Order->id;
		}

		return false;
	}

	/**
	 * mark the order as paid
	 *
	 * @param 	int 	$orderId 	The order ID to mark as paid
	 * @return 	bool
	 */
	public function markAsPaid($orderId) {
		$Order = EvClassRegistry::init('EvCheckout.Order');

		if ($Order->paid($orderId) !== false) {
			// dispatch order paid event
			$this->_controller->getEventManager()->dispatch(
				new CakeEvent('EvCheckout.Component.Order.paid', $this, array(
					'orderId' => $orderId
				))
			);

			return true;
		}

		return false;
	}

	/**
	 * update the orders status
	 *
	 * @param 	int 	$orderId 	The order ID to update
	 * @param 	int 	$statusId 	The status ID
	 * @return 	bool
	 */
	public function updateStatus($orderId, $statusId) {
		$Order = EvClassRegistry::init('EvCheckout.Order');

		if ($Order->updateStatus($orderId, $statusId) !== false) {
			// dispatch order paid event
			$this->_controller->getEventManager()->dispatch(
				new CakeEvent('EvCheckout.Component.Order.statusChanged', $this, array(
					'orderId' => $orderId,
					'statusId' => $statusId
				))
			);

			return true;
		}

		return false;
	}

	/**
	 * return the order complete url
	 *
	 * @return string 	$orderCompleteUrl
	 */
	public function getCompleteUrl() {
		return Router::url(
			RouterUtil::getListingRoute('EvCheckout', 'Order', 'complete'),
			true
		);
	}

	/**
	 * get the order cancelled url
	 *
	 * @return string 	$orderCancelledUrl
	 */
	public function getCancelledUrl() {
		return Router::url(
			RouterUtil::getListingRoute('EvCheckout', 'Order', 'cancel'),
			true
		);
	}

	/**
	 * build the order from the given transactionId
	 *
	 * @param 	int 		$transactionId 		The successful transaction ID just completed
	 * @return 	int|bool 	$orderId 			The order we've just built or false on fail
	 */
	public function buildOrder($transactionId) {
		$selectedBuilder = Configure::read('EvCheckout.orderBuilder');

		$OrderBuilder = $this->loadBuilder($selectedBuilder);

		if (is_object($OrderBuilder)) {
			return $OrderBuilder->build($transactionId);
		}

		return false;
	}

	/**
	 * load the builder to use
	 *
	 * @param 	string 			$selectedBuilder 	The selected builder to use
	 * @return 	object|bool 	$OrderBuilder 		The builder object
	 */
	public function loadBuilder($selectedBuilder) {
		list($plugin, $builder) = pluginSplit($selectedBuilder, true, '');

		App::uses($builder, $plugin . 'OrderBuilders');

		if (class_exists($builder)) {
			return new $builder($this->_controller);
		}

		return false;
	}

	/**
	 * assign the order to the original transaction
	 *
	 * @param 	int 	$orderId 		The order ID
	 * @param 	int 	$transactionId 	The transaction ID
	 * @return 	bool
	 */
	public function assignToTransaction($orderId, $transactionId) {
		$Transaction = EvClassRegistry::init('EvTransactions.Transaction');

		return $Transaction->save(
			array(
				'Transaction' => array(
					'id' => $transactionId,
					'model' => 'EvCheckout.Order',
					'model_id' => $orderId
				)
			)
		);
	}

	/**
	 * send order confirmation emails
	 *
	 * @param 	int 	$orderId 	The order ID
	 * @return 	bool
	 */
	public function sendOrderConfirmEmails($orderId) {
		$QueuedTask = EvClassRegistry::init('Queue.QueuedTask');

		$Order = EvClassRegistry::init('EvCheckout.Order');
		$orderDetails = $Order->readForEdit($orderId);

		// queue the confirmation email to customer
		$QueuedTask->createJob(
			'CustomEmail',
			array(
				'from' => array(
					Configure::read('SiteSetting.general.admin_email') => Configure::read('SiteSetting.general.site_title')
				),
				'to' => array(
					$orderDetails['Order']['email'] => $orderDetails['Order']['name']
				),
				'subject' => __('Order Confirmation #%s', array($orderId)),
				'viewVars' => array(
					'order' => $orderDetails
				),
				'helpers' => array(
					'Html',
					'Number'
				),
				'template' => 'EvCheckout.Orders/customerNewOrder'
			)
		);

		// queue the confirmation email to admin
		$manageOrder = $this->_controller->Routable->getListingRoute(
			'EvCheckout',
			'Order',
			'edit'
		);
		$manageOrder[] = $orderId;
		$manageOrder['admin'] = true;

		$QueuedTask->createJob(
			'CustomEmail',
			array(
				'from' => array(
					Configure::read('SiteSetting.general.admin_email') => Configure::read('SiteSetting.general.site_title')
				),
				'to' => array(
					Configure::read('SiteSetting.ev_checkout.new_order_email') => Configure::read('SiteSetting.ev_checkout.new_order_name')
				),
				'subject' => __('New Order #%s', array($orderId)),
				'viewVars' => array(
					'order' => $orderDetails,
					'manageOrderUrl' => Router::url($manageOrder, true)
				),
				'helpers' => array(
					'Html',
					'Number'
				),
				'template' => 'EvCheckout.Orders/adminNewOrder'
			)
		);

		return true;
	}

	/**
	 * get all the orders belonging to a user
	 *
	 * @param 	int 	$userId 	User id number
	 * @return 	array 	$orders 	Paginated users order
	 */
	public function getOrdersByUser($userId) {
		$OrderModel = EvClassRegistry::init('EvCheckout.Order');

		$this->_controller->Paginator->settings = array(
			'conditions' => array(
				$OrderModel->alias . '.user_id' => $userId
			),
			'contain' => array(
				'OrderItem',
				'OrderTotal',
				'OrderStatus'
			),
			'order' => $OrderModel->alias . '.id DESC',
			'limit' => 10
		);

		return $this->_controller->Paginator->paginate(
			$OrderModel
		);
	}
}
