<?php

App::uses('OrderBuilderInterface', 'EvCheckout.Lib');
App::uses('OrderBuilder', 'EvCheckout.Lib');
App::uses('AddressBookLib', 'EvAddressBook.Lib');
App::uses('BasketLib', 'EvBasket.Lib');

class BasketBuilder extends OrderBuilder implements OrderBuilderInterface {

	/**
	 * build an order from a basket
	 *
	 * @param 	int 	$transactionId 		The successful transaction ID just completed
	 * @return  int 	$orderId 			The id number of the order we just created
	 */
	public function build($transactionId) {
		if (! CakePlugin::loaded('EvBasket')) {
			$this->_controller->Flash->fail('EvBasket is not loaded. Cannot process basket');
			return false;
		}

		if (! CakePlugin::loaded('EvAddressBook')) {
			$this->_controller->Flash->fail('EvAddressBook is not loaded. Cannot retrieve addresses');
			return false;
		}

		// load the mbasket manager so we can get the basket
		if (empty($this->_controller->BasketManager)) {
			$this->_controller->BasketManager = $this->_controller->loadComponent('EvBasket.Basket');
		}

		// get the addresses
		$TransactionModel = EvClassRegistry::init('EvTransactions.Transaction');
		$transaction = $TransactionModel->readForView($transactionId);

		$addresses = array();
		if (! empty($transaction['TransactionData']['addresses'])) {
			$addresses = json_decode($transaction['TransactionData']['addresses'], true);

			// format the addresses
			foreach ($addresses as $key => $address) {
				$addresses[$key] = AddressBookLib::display($address);
			}
		}

		$orderId = $this->_createFromBasket(
			$this->_controller->BasketManager->getBasket(),
			$this->_controller->Auth->user('User'),
			$addresses
		);

		$this->_controller->OrderManager->markAsPaid($orderId);

		// clear the basket
		BasketLib::deleteBasket($transaction['Transaction']['model_id']);

		// force a rebuild of the basket cache
		$this->_controller->BasketManager->getBasket(true);

		// update the transaction to reference the order
		$this->_controller->OrderManager->assignToTransaction(
			$orderId,
			$transactionId
		);

		return $orderId;
	}

	/**
	 * create and setup an order from a basket
	 *
	 * @param 	array 		$basket 	Array of Basket Data
	 * @param 	array 		$user 		array of user details
	 * @param 	array 		$addresses 	array of address details
	 * @return 	int|bool 	$orderId 	Return the order ID number
	 */
	protected function _createFromBasket($basket, $user, $addresses) {
		$this->_controller->OrderManager->setupOrder($user, $addresses);

		foreach ($basket['BasketItem'] as $item) {
			$itemData = $this->_controller->OrderManager->setupOrderItemData($item);

			$this->_controller->OrderManager->setupOrderItem(
				BasketLib::getItemName($item),
				$item['quantity'],
				$item['unit_price'],
				$item['row_total'],
				$item['tax_rate'],
				$itemData
			);

			if (CakePlugin::loaded('EvInventory')) {
				// dispatch event to lower stock levels
				$this->_controller->getEventManager()->dispatch(
					new CakeEvent('EvInventory.Inventory.reduce', $this, array(
						'model' => $item['model'],
						'model_id' => $item['model_id'],
						'quantity' => $item['quantity']
					))
				);
			}
		}

		foreach ($basket['BasketTotal'] as $total) {
			$this->_controller->OrderManager->setupOrderTotal(
				$total['name'],
				$total['amount'],
				$total['sequence']
			);
		}

		$orderId = $this->_controller->OrderManager->saveOrder(EvClassRegistry::init('EvCheckout.Order'));

		if ($orderId !== false) {
			if (! empty($basket['Basket']['id'])) {
				$Basket = EvClassRegistry::init('EvBasket.Basket');
				$Basket->delete($basket['Basket']['id']);
			}

			return $orderId;
		}

		return false;
	}
}
