<?php

App::uses('EvCheckoutAppModel', 'EvCheckout.Model');
App::uses('CakeEvent', 'Event');
App::uses('AppNonDeletableModel', 'Model');
App::uses('CustomEmail', 'Lib');
App::uses('CakeSession', 'Model/Datasource');
App::uses('CakeEmail', 'Network/Email');
App::uses('OrderStatus', 'EvCheckout.Model');

class Order extends EvCheckoutAppModel {

	public $cacheQueries = false;

	public $belongsTo = array(
		'Customer' => array(
			'className' => 'EvCheckout.Customer',
			'foreignKey' => 'user_id'
		),
		'DeliveryAddress' => array(
			'className' => 'EvCheckout.CustomerAddress',
			'foreignKey' => 'delivery_address_id'
		),
		'InvoiceAddress' => array(
			'className' => 'EvCheckout.CustomerAddress',
			'foreignKey' => 'invoice_address_id'
		),
		'OrderStatus' => array(
			'className' => 'EvCheckout.OrderStatus'
		)
	);

	public $hasMany = array(
		'OrderItem' => array(
			'className' => 'EvCheckout.OrderItem'
		),
		'OrderOrderStatusHistory' => array(
			'className' => 'EvCheckout.OrderOrderStatusHistory'
		)
	);

	public $hasOne = array(
		'Transaction' => array(
			'className' => 'Transactions.Transaction',
			'foreignKey' => 'model_id',
			'conditions' => array(
				'Transaction.model' => 'EvCheckout.Order'
			)
		)
	);

/**
 * Default order of order listings
 */
	public $order = 'Order.created DESC';

/**
 * Temporary variable used to cache the "Order.cart_id" session variable
 * @var int
 */
	protected $_cartId = null; // User's shopping cart ID

/**
 * Construct and add in virtual fields
 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		$this->virtualFields['full_name'] = 'CONCAT(' . $this->escapeField('first_name') . ', " ", ' . $this->escapeField('last_name') . ')';

		$this->virtualFields['invoice_number'] = 'CONCAT("#", LPAD(' . $this->escapeField('id') . ', 8, 0))';

		// The subtotal excludes VAT so provide a subtotal_inc_vat column for convinience.
		$this->virtualFields['subtotal_inc_vat'] = 'CAST(( ' . $this->escapeField('subtotal') . ' + ' . $this->escapeField('vat') . ' + ' . $this->escapeField('discount_vat') . ' - ' . $this->escapeField('delivery_vat') . ' - ' . $this->escapeField('surcharge_vat') . ') AS DECIMAL(7,2))';

		// The discount_total excludes VAT so provide a discount_inc_vat column for convinience.
		$this->virtualFields['discount_inc_vat'] = 'CAST(( ' . $this->escapeField('discount_total') . ' * (1 + (' . $this->escapeField('vat') . ' + ' . $this->escapeField('discount_vat') . ' - ' . $this->escapeField('delivery_vat') . ' - ' . $this->escapeField('surcharge_vat') . ') / ' . $this->escapeField('subtotal') . ')) AS DECIMAL(7,2))';
	}

/**
 * Called afterSave used to send event in the case of an order changing status
 */
	public function afterSave($created, $options = array()) {
		// Check if the order status has been modified.
		if ($created === false && !empty($this->data[$this->alias]['order_status_id'])) {

			$lastOrderStatus = $this->OrderOrderStatusHistory->findByOrderId($this->id);

			if (empty($lastOrderStatus['OrderOrderStatusHistory']['order_status_id'])
				|| $lastOrderStatus['OrderOrderStatusHistory']['order_status_id'] !== $this->data[$this->alias]['order_status_id']
			) {

				$currentOrderStatus = array(
					'order_id' => $this->id,
					'order_status_id' => $this->data[$this->alias]['order_status_id'],
					'user_id' => CakeSession::read('Auth.User.User.id')
				);

				$this->OrderOrderStatusHistory->save($currentOrderStatus);

				// Trigger order status change event.

				$orderStatus = $this->OrderStatus->findById($this->data[$this->alias]['order_status_id']);

				if (!empty($orderStatus)) {

					$eventName = 'order' . Inflector::camelize($orderStatus['OrderStatus']['system_name']);

					// Send out event to notify of order status change
					$event = new CakeEvent('Model.Order.' . $eventName, $this, array(
							'data' => $this->data
						)
					);
					$this->getEventManager()->dispatch($event);

				}

			}

		}

		return;
	}

/**
 * Adds an item to the basket.
 *
 * @param array $data An array of information about the product been added.
 *                    The following params are mandatory
 *                    $data['product_id'] - The ID of the root product
 *                    $data['order_id'] - The ID of the order to attact it to
 *                    $data['model'] - The model alias of the model the product belongs to
 *                    $data['model_id'] - The ID of the model
 * @return integer	  Returns 1 if added successfully, -1 if added with quantity adjusted
 * 					  and 0 if failed to add
 */
	public function addToBasket($data) {
		$defaults = array(
			'product_id' => null,
			'model' => null,
			'model_id' => null,
			'order_id' => null,
			'name' => '',
			'description' => '',
			'unit_cost' => 0,
			'quantity' => 1,
			'stock' => 0,
			'unlimited_stock' => false,
			'subtotal' => false,
			'vat' => false,
			'total' => false,
			'is_virtual' => false,
			'vat_rate_id' => '',
		);
		$data = array_merge($defaults, $data);

		// Check required fields are filled in
		if (empty($data['model_id']) || empty($data['model']) || !isset($data['product_id'])) {
			return false;
		}

		if (empty($data['order_id'])) {
			$cartId = $this->getCartId();
			$cartId = ($cartId === null) ? $this->createOrder() : $cartId;
			$data['order_id'] = $cartId;
		}

		$orderItem = $this->OrderItem->find('first', array(
			'conditions' => array(
				'model' => $data['model'],
				'model_id' => $data['model_id'],
				'order_id' => $data['order_id']
			)
		));

		if (!empty($orderItem)) {
			$data['id'] = $orderItem['OrderItem']['id'];
			$data['quantity'] += $orderItem['OrderItem']['quantity'];
		}

		$trackStock = Configure::read('EvCheckout.track_stock');
		if ($trackStock === true && ($data['quantity'] > $data['stock'] && $data['unlimited_stock'] === false)) {
			$data['quantity'] = $data['stock'];
			$quantityAdjusted = true;
		} else {
			$quantityAdjusted = false;
		}

		if ($data['subtotal'] === false) {
			$data['subtotal'] = $data['quantity'] * $data['unit_cost'];
		}

		if ($data['vat'] === false && $data['vat_rate_id']) {
			$data['vat'] = $this->_calculateTaxes($data['unit_cost'], $data['vat_rate_id'], $data['quantity']);
		}

		if ($data['total'] === false) {
			$data['total'] = $data['subtotal'] + $data['vat'];
		}

		// Send off event to allow listeners to manipulate the order item before save
		$event = new CakeEvent('Model.Order.addToBasket', $this, array(
			'OrderItem' => $data
		));
		$this->getEventManager()->dispatch($event);
		$data = $event->data;


		if ($this->OrderItem->save($data)) {

			$this->updateOrderTotal();

		} else {

			return 0;

		}

		return $quantityAdjusted === true ? -1 : 1;
	}

/**
 * Calculates the VAT.
 *
 * @param float $subtotal Unit price of item VAT is to be calculated for
 * @param int $vatRateId ID of VAT Rate
 * @param int $quantity Quantity of item being purchased
 * @return float VAT
 */
	protected function _calculateTaxes($subtotal, $vatRateId, $quantity = 1) {
		$vat = $this->OrderItem->VatRate->findById($vatRateId);

		$tax = ($vat['VatRate']['rate'] > 0) ? $subtotal * ($vat['VatRate']['rate'] / 100) : 0.00;

		// Now multiply the tax rate to reflect the correct quantity.
		$tax *= $quantity;

		$event = new CakeEvent('Model.Order.calculateTaxes', $this, array(
				'subtotal' => $subtotal,
				'vat_rate_id' => $vatRateId,
				'tax' => $tax,
				'quantity' => $quantity
			)
		);
		$this->getEventManager()->dispatch($event);
		$tax = $event->data['tax'];
		return $tax;
	}

/**
 * Sets the User of the given cart or session cart
 *
 * @param int $userId User::id
 * @param int $cartId The ID of the cart
 */
	public function setUser($userId, $cartId = false) {
		$this->id = ($cartId) ? $cartId : $this->getCartID();

		$data = array(
			'Order' => array(
				'id' => $this->id,
				'user_id' => $userId
			)
		);

		if ($userId > 0) {

			// Set the existing customer's details in the orders table for future
			// reference.

			$user = $this->Customer->findById($userId);

			$data['Order']['first_name'] = $user['Customer']['first_name'];
			$data['Order']['last_name'] = $user['Customer']['last_name'];
			$data['Order']['email'] = $user['Customer']['email'];
			$data['Order']['phone'] = $user['Customer']['phone'];

		}

		return (bool)$this->save($data);
	}

/**
 * Gets Cart ID from SESSION
 * @param  boolean $checkExists make sure cart exists in the database
 * @return integer Cart/Order ID
 */
	public function getCartId($checkExists = true) {
		if (empty($this->_cartId)) {

			$this->_cartId = CakeSession::read('Order.cart_id');

			if (!empty($this->_cartId) && $checkExists === true) {
				$order = $this->hasAny(array(
					'Order.id' => $this->_cartId
				));
				$this->_cartId = !empty($order) ? $this->_cartId : null;
			}

		}

		return $this->_cartId;
	}

/**
 * Clears the cart session ID used when the order has been completed
 * @return void
 */
	public function clearCartId() {
		$this->_cartId = null;
		CakeSession::write('Order.cart_id', null);
		CakeSession::delete('Order.cart_id');

		return;
	}

/**
 * Gets the current Order item either by the passed ID or by the Session variable using Order::getCartId
 * @param  mixed $id The ID of the cart
 * @param  boolean $isAdmin Indicates whether the method is being accessed from the admin
 * @param  boolean $summaryOnly Indicates whether we should return cart with all bells and whisles or only the basic info
 * @param  boolean $checkStock Indicates whether we should check stock levels and update basket quantities or not
 * @return array Find first of the Order array or false if no cart can be found
 */
	public function getCart($id = false, $isAdmin = false, $summaryOnly = false, $checkStock = false) {
		// If no $id passed find from session
		$id = ($id) ? $id : $this->getCartId();

		$params = array();

		// assign order status check, the unpaid status check is not required when
		// method is called from the admin
		if ($isAdmin === false) {

			$params['conditions'] = array($this->alias . '.id' => $id, $this->alias . '.order_status_id' => OrderStatus::IN_BASKET);

		} else {

			$params['conditions'] = array($this->alias . '.id' => $id);

		}

		if ($summaryOnly === false) {
			$params['contain'][] = 'DeliveryAddress';
			$params['contain'][] = 'InvoiceAddress';
			$params['contain'][] = 'OrderItem';
		}

		$order = $this->find('first', $params);

		// Clear cart if user has already paid for order or it has been deleted
		if (empty($order)) {

			$this->clearCartId();

			return false;

		} elseif ($isAdmin === false && (int)$order[$this->alias]['order_status_id'] === OrderStatus::IN_BASKET) {

			// Check items in basket are still in stock.
			$trackStock = Configure::read('EvCheckout.track_stock');
			if ($trackStock === true && $checkStock === true) {

				$quantityAdjusted = false;
				foreach ($order['OrderItem'] as &$orderItem) {

					$Model = isset($Model) ? $Model : ClassRegistry::init($orderItem['model']);

					if (empty($Model)) {
						throw new InternalErrorException("{$orderItem['model']} model could not be loaded");
					}

					$product = $Model->findById($orderItem['model_id']);

					if ($orderItem['quantity'] > $product[$Model->alias]['stock'] && $product[$Model->alias]['unlimited_stock'] === false) {

						$orderItem['quantity'] = $product[$Model->alias]['stock'];

						// Adjust the costs for the new quantity.
						$orderItem['unit_cost'] = $product[$Model->alias]['price'];
						$orderItem['subtotal'] = $product[$Model->alias]['price'] * $orderItem['quantity'];
						$orderItem['vat'] = $this->_calculateTaxes($orderItem['unit_cost'], $orderItem['vat_rate_id'], $orderItem['quantity']);
						$orderItem['total'] = $orderItem['subtotal'] + $orderItem['vat'];

						$quantityAdjusted = true;

					}

					// Include the stock levels in the response.
					$orderItem['stock'] = $product[$Model->alias]['stock'];
					$orderItem['unlimited_stock'] = $product[$Model->alias]['unlimited_stock'];

				}

				if ($quantityAdjusted === true) {
					// Update order with new quantities.
					$this->saveAssociated(
						array(
							'Order' => $order['Order'],
							'OrderItem' => $order['OrderItem']
						)
					);
					// Update the totals and retrieve the updated cart.
					$this->updateOrderTotal();
					$order = $this->getCart();
					$order['Order']['cart_modified'] = true;
				}

			}

		}

		return $order;
	}

/**
 * Gets a summary of the cart, does not bring back as much information as getCart() is designed to be lightweight as it is called on every page for header etc
 * @param  mixed $id      The ID of the cart
 * @param  boolean  $isAdmin Indicates whether the method is being accessed from the admin
 * @return array          Find first of the Order array or false if no cart can be found
 */
	public function getCartSummary($id = false, $isAdmin = false) {
		return $this->getCart($id, $isAdmin, true);
	}

/**
 * Creates a new cart and saves the ID into the session
 * @return int The ID of the new cart
 */
	public function createOrder() {
		$data = array(
			'Order' => array('order_status_id' => OrderStatus::IN_BASKET),
			'OrderOrderStatusHistory' => array('order_status_id' => OrderStatus::IN_BASKET)
		);

		$this->create();
		$this->saveAssociated($data);

		$this->_cartId = $this->id;
		CakeSession::write('Order.cart_id', $this->_cartId);

		return $this->_cartId;
	}

/**
 * Gets the cart information for the admin_edit page
 */
	public function readForEdit($id, $params = array()) {
		$params['contain'][] = 'OrderItem';
		$params['contain'][] = 'OrderStatus';
		$params['contain'][] = 'Transaction';
		$params['contain'][] = 'DeliveryAddress';
		$params['contain'][] = 'InvoiceAddress';
		$params['contain'][] = 'Customer';

		return parent::readForEdit($id, $params);
	}

	public function updateOrderTotal() {
		$order = $this->getCart();

		$subtotal = 0.00;
		$vat = 0.00;

		$itemCount = 0;

		$isVirtual = 1;
		if (!empty($order['OrderItem'])) {

			foreach ($order['OrderItem'] as $item) {

				$subtotal += $item['subtotal'];
				$vat += $item['vat'];
				if (!$item['is_virtual']) {
					$isVirtual = 0;
				}

				$itemCount += $item['quantity'];
			}

		}
		$total = $subtotal + $vat;
		$data = array(
			'Order' => array(
				'id' => $order['Order']['id'],
				'discount_vat' => 0, // Set by Event dispatch later in the method
				'discount_total' => 0, // Set by Event dispatch later in the method
				'delivery_subtotal' => 0, // Set by Event dispatch later in the method
				'delivery_vat' => 0, // Set by Event dispatch later in the method
				'delivery_total' => 0, // Set by Event dispatch later in the method
				'surcharge_subtotal' => 0, // Set by Event dispatch later in the method
				'surcharge_vat' => 0, // Set by Event dispatch later in the method
				'surcharge_total' => 0, // Set by Event dispatch later in the method
				'items_subtotal' => $subtotal, // Fields not saved in the database but are used by the event listeners to calculate additional costs
				'items_vat' => $vat, // Fields not saved in the database but are used by the event listeners to calculate additional costs
				'vat' => $vat,
				'subtotal' => $subtotal,
				'total' => $total,
				'items_total' => $itemCount,
				'is_virtual' => $isVirtual,
			)
		);

		//Assign user if there is a signed in one
		$userId = CakeSession::read('Auth.User.User.id');
		if (!empty($userId)) {
			$data['Order']['user_id'] = $userId;
		}

		// Delivery, discounts etc, calc'd in event
		$event = new CakeEvent('Model.Order.updateOrderTotal', $this, array(
				'Order' => array_merge($order, $data)
			)
		);
		$this->getEventManager()->dispatch($event);
		$data = $event->data['Order'];

		//Return total if success else false
		return ($this->save($data) ? $total : false);
	}

	public function setDeliveryAddress($addressId = null) {
		$deliveryAddress = null;

		$this->id = $this->getCartId();

		if ($addressId) {

			$this->DeliveryAddress->id = $addressId;
			$this->DeliveryAddress->saveField('last_used_for_delivery', gmdate('Y-m-d H:i:s'));

			$deliveryAddress = $this->saveField('delivery_address_id', $addressId);

		}

		$this->updateOrderTotal();

		return $deliveryAddress;
	}

	public function setInvoiceAddress($addressId) {
		$this->id = $this->getCartId();
		if ($addressId) {
			$this->InvoiceAddress->id = $addressId;
			$this->InvoiceAddress->saveField('last_used_for_invoice', gmdate('Y-m-d H:i:s'));
		}

		$result = $this->saveField('invoice_address_id', $addressId);

		return $result;
	}

/**
 * Removes an item from the basket and updates the order total.
 *
 * @param integer $orderItemId order item ID
 * @return boolean
 */
	public function removeItem($orderItemId) {
		$this->id = $this->getCartId();
		$response = $this->OrderItem->deleteAll(
			array(
				'OrderItem.id' => $orderItemId,
				'OrderItem.order_id' => $this->id
			),
			false,
			false
		);

		// If everything deleted without issue update the order total.
		if ($response === true) {
			$this->updateOrderTotal();
		}

		return $response;
	}

	public function updateBasketItems($order) {
		$this->id = $this->getCartId();

		$data = array();

		// Fetch all the order items from the database that we are updating, we
		// need to check that the items being posted still exist.
		if (!empty($order['OrderItem'])) {
			$result = $this->OrderItem->find('all', array(
				'conditions' => array(
					'OrderItem.id' => Hash::extract($order['OrderItem'], '{n}.id')
				)
			));
			$orderItems = Hash::combine($result, '{n}.OrderItem.id', '{n}');
		}

		foreach ($order['OrderItem'] as $itemKey => $item) {

			if ($item['quantity'] > 0) {

				$orderItem = !empty($orderItems[$item['id']]) ? $orderItems[$item['id']] : null;

				// Make sure the item still exists in the basket before
				// recalculating the item costs.
				if ($orderItem !== null) {

					$subtotal = $orderItem['OrderItem']['unit_cost'] * $item['quantity'];
					$vat = $this->_calculateTaxes($orderItem['OrderItem']['unit_cost'], $orderItem['OrderItem']['vat_rate_id'], $item['quantity']);
					$data[] = array(
						'id' => $item['id'],
						'order_id' => $this->id,
						'quantity' => $item['quantity'],
						'subtotal' => $subtotal,
						'vat' => $vat,
						'total' => $subtotal + $vat
					);

				}

			} elseif ($item['quantity'] <= 0) {

				// Remove item from basket.
				$this->OrderItem->delete($item['id']);
				unset($order['OrderItem'][$itemKey]);

			}

		}

		$response = $this->OrderItem->saveMany($data);

		// If everything updated without issue update the order total.
		if ($response === true) {
			$this->updateOrderTotal();
		}

		return;
	}

	public function markAsPaid($id = false, $success = true, $method = false) {
		$id = $id ? $id : $this->getCartId();

		$order = $this->getCart($id);

		// Only mark as paid once
		if (isset($order['Order']['is_paid']) && (bool)$order['Order']['is_paid'] !== true) {

			if ($success) {

				$dbo = $this->getDataSource();

				$data = array(
					'Order' => array(
						'id' => $id,
						'order_status_id' => OrderStatus::COMPLETED,
						'is_paid' => true,
						// The order_time needs to be in GMT, Cake's TimeHelper
						// will display this in the correct timezone in the Views
						'order_time' => gmdate('Y-m-d H:i:s')
					)
				);

				$this->save($data);

				// Adjust the stock.
				$this->_adjustStockLevels($order);

			}

		}

		$this->clearCartId();

		return;
	}

/**
 * Gets the most recent orders of the given user
 * @param  $userId
 */
	public function getRecentOrders($userId, $limit = 10) {
		$orders = $this->find('all', array(
				'conditions' => array(
					$this->alias . '.user_id' => $userId,
					$this->alias . '.order_status_id !=' => OrderStatus::IN_BASKET
				),
				'contain' => array(
					'OrderStatus'
				),
				'limit' => $limit
			)
		);

		return $orders;
	}

/**
 * Goes through the order items and calls each model to adjust each's stock
 * @param  array $order The order array including order items
 * @param  integer $modifier
 * @return void
 */
	protected function _adjustStockLevels($order, $modifier = -1) {
		// Check that we're tracking stock before updating stock levels.
		$trackStock = Configure::read('EvCheckout.track_stock');
		if ($trackStock !== true) {
			return;
		}

		foreach ($order['OrderItem'] as $orderItem) {

			$Model = isset($Model) ? $Model : ClassRegistry::init($orderItem['model']);

			if (empty($Model)) {

				throw new InternalErrorException("{$orderItem['model']} model could not be loaded");

			} else {

				$Model->adjustStock($orderItem, $modifier);

			}

		}

		return;
	}

/**
 * Check if the order contains only virtual items.
 *
 * @return boolean
 */
	public function isCartVirtual() {
		$cart = $this->getCart();
		foreach ($cart['Items'] as $item) {
			if (!$item['Product']['is_virtual']) {
				return false;
			}
		}
		return true;
	}

}
