<?php

App::uses('EvCheckoutAppModel', 'EvCheckout.Model');
App::uses('CakeEvent', 'Event');

class CustomerAddress extends EvCheckoutAppModel {

	public $hasMany = array(
		'Order' => array(
			'className' => 'EvCheckout.Order'
		)
	);

	public $belongsTo = array(
		'Customer' => array(
			'className' => 'EvCheckout.Customer'
		)
	);

	public $validate = array(
		'address_line_1' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'First address line cannot be blank'
			)
		),
		'city' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Please enter a valid city'
			)
		),
		'postcode' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Please enter a valid postcode'
			)
		)
	);

/**
 * Creates a customer address
 * @param  int  	$userId  	User ID to link address to
 * @param  array  	$data    	Address information
 * @param  int  	$orderId 	Optional param to link to order. Adds as delivery address by default unless model alias is InvoiceAddress
 * @return mixed 				Returns Address ID if successfully saved else false
 */
	public function createAddress($userId, $data, $orderId = null) {
		$data[$this->alias]['user_id'] = $userId;

		$this->set($data);

		if ($this->save()) {

			//If orderID is set then link to order
			if ($orderId !== null) {

				$this->Order->id = $orderId;
				if ($this->alias == 'InvoiceAddress') {

					$this->Order->saveField('invoice_address_id', $this->id);

				} else {

					$this->Order->saveField('delivery_address_id', $this->id);

				}

				$this->Order->updateOrderTotal();

			}

			return $this->id;

		} else {

			return false;

		}
	}

/**
 * Get recent addresses for a user
 *
 * @param integer $userId
 * @param integer $currentAddressId
 * @param string $type either 'delivery' or 'invoice'
 * @param integer $limit number of addresses to return
 * @return array
 */
	protected function _recentAddresses($userId, $currentAddressId, $type = 'delivery', $limit = 5) {
		$params = array(
			'conditions' => array(
				$this->alias . '.user_id' => $userId,
				$this->alias . '.is_active' => true
			),
			'limit' => $limit,
			'joins' => array(
				//Only get addresses if the user is NOT a guest
				array(
					'table' => 'users',
					'alias' => 'Customer',
					'type' => 'RIGHT',
					'conditions' => array(
						$this->alias . '.user_id = Customer.id',
						'OR' => array(
							'Customer.is_guest_user <>' => true,
							'Customer.is_guest_user IS NULL'
						)
					)
				)
			)
		);

		if ($type == 'delivery') {
			$params['order'] = array($this->alias . '.last_used_for_delivery' => 'DESC');
		} elseif ($type == 'invoice') {
			$params['order'] = array($this->alias . '.last_used_for_invoice' => 'DESC');
		}

		$addresses = $this->find('all', $params);

		$addressList = Hash::combine($addresses, '{n}.' . $this->alias . '.id');

		// Make sure the current selected address is returned for all customers.
		if (!empty($currentAddressId) && !in_array($currentAddressId, array_keys($addressList))) {
			$currentAddress = $this->findById($currentAddressId);
			array_unshift($addresses, $currentAddress);
			array_slice($addresses, 0, $limit);
		}

		return $addresses;
	}

	public function recentDeliveryAddresses($userId, $currentDeliveryAddress, $limit) {
		return $this->_recentAddresses($userId, $currentDeliveryAddress, 'delivery', $limit);
	}

	public function recentInvoiceAddresses($userId, $currentInvoiceAddress, $limit) {
		return $this->_recentAddresses($userId, $currentInvoiceAddress, 'invoice', $limit);
	}

	public function beforeSave($options = []) {
		if (!empty($this->data[$this->alias]['postcode'])) {
			$this->data[$this->alias]['postcode'] = strtoupper($this->data[$this->alias]['postcode']);
		}

		return parent::beforeSave($options);
	}

}
