<?php

App::uses('EvCheckoutAppController', 'EvCheckout.Controller');
App::uses('OrderStatus', 'EvCheckout.Model');
/**
 * Checkout Orders Controller
 *
 * @package EvCheckout/Orders
 */
class OrdersController extends EvCheckoutAppController {

/**
 * This will store the currently enabled checkout process which will be populated from one of the below variables
 * $checkoutFlowOnSite, $checkoutFlowOffSite, $checkoutFlowOnePageOffSite, $checkoutFlowOnePageOnSite
 * @var array
 */
	public $checkoutProcess = null;

/**
 * On site default checkout flow
 * @var array
 */
	public $checkoutFlowOnSite = array(
		'register' => array(
			'next' => 'delivery',
			'process' => array(
				'_processRegistration'
			)
		),
		'delivery' => array(
			'next' => 'invoice',
			'process' => array(
				'_processDeliveryAddress'
			)
		),
		'invoice' => array(
			'next' => 'success',
			'process' => array(
				'_processInvoiceAddress',
				'_processOnsitePayment'
			),
			'title' => 'Invoice Address  '
		),
		'confirm' => array(
			'step' => 'confirm',
			'next' => 'success',
			'process' => array(
			)
		)
	);

/**
 * Off site default checkout flow
 * @var array
 */
	public $checkoutFlowOffSite = array(
		'register' => array(
			'next' => 'delivery',
			'process' => array(
				'_processRegistration'
			)
		),
		'delivery' => array(
			'next' => 'invoice',
			'process' => array(
				'_processDeliveryAddress'
			),
			'title' => 'Delivery Address'
		),
		'invoice' => array(
			'next' => 'confirm',
			'process' => array(
				'_processInvoiceAddress'
			),
			'title' => 'Invoice Address'
		),
		'confirm' => array(
			'next' => 'offsite_payment',
			'process' => array(
			),
			'title' => 'Confirm Order'
		),
		'offsite_payment' => array(
			'next' => 'success',
			'process' => array(
				'_processOffsitePayment'
			)
		),
		'success' => array(
			'next' => false,
			'process' => array(
			)
		)
	);

/**
 * One page on site checkout flow
 * @var array
 */
	public $checkoutFlowOnePageOnSite = array(
		'register' => array(
			'next' => 'onepage',
			'process' => array(
				'_processRegistration'
			)
		),
		'onepage' => array(
			'next' => 'success',
			'process' => array(
				'_processDeliveryAddress',
				'_processInvoiceAddress',
				'_processCreditCard'
			),
			'title' => 'Checkout'
		)
	);

/**
 * One page off site checkout flow
 * @var array
 */
	public $checkoutFlowOnePageOffSite = array(
		'register' => array(
			'next' => 'onepage',
			'process' => array(
				'_processRegistration'
			)
		),
		'onepage' => array(
			'next' => 'success',
			'process' => array(
				'_processDeliveryAddress',
				'_processInvoiceAddress',
				'_processCreditCard'
			),
			'title' => 'Checkout'
		)
	);

	public $components = array(
		'Transactions.Transactions'
	);

/**
 * Used to force SSL on any pages specified, similar to $this->Auth->allow
 */
	public $sslProtect = array(
		'invoice'
	);

/**
 * Set up default checkout process
 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		if ($defaultCheckoutProcess = Configure::read('EvCheckout.default_checkout_process')) {

			$checkoutProcessVariable = "checkoutFlow" . $defaultCheckoutProcess;

			if (isset($this->{$checkoutProcessVariable})) {

				$this->checkoutProcess = $this->{$checkoutProcessVariable};

			} else {

				throw new Exception("Checkout process does not exist. Please create ", 1);

			}

		} else {

			$this->checkoutProcess = $this->checkoutFlowOnSite;

		}

		$this->helpers[] = 'EvCheckout.CustomerAddress';
	}

	public function beforeFilter() {
		//Work out if any steps MUST be SSL protected, e.g. handle credit card info.
		//If there is an SSL certificate available for the site then we should have the
		//entire checkout under SSL but as this isn't very common at the moment only ensure
		//those that have credit card forms are protected
		if (isset($this->checkoutProcess[$this->action])) {

			$processesInStage = $this->checkoutProcess[$this->action]['process'];

			if (in_array('_processCreditCard', $processesInStage)) {

				$this->sslProtect[] = $this->action;

			}

		}


		if (in_array($this->action, $this->sslProtect)) {

			$this->forceSsl();

		}

		$this->adminActions[] = 'admin_print';

		$this->Auth->allow(array(
			'basket',
			'register',
			'remove_item',
			'paymentCallback',
			'sagepayServerCallback',
			'failure',
			'success'
		));

		parent::beforeFilter();
	}

/**
 * Switch to SSL.
 *
 * @return mixed
 */
	public function forceSsl() {
		// Check if protocol is already SSL?
		//
		// Check if we're not currently using SSL. We need to check for HTTP_X_SSL
		// which needs defining in the nginx config for the site as Apache is
		// telling us that we are not on SSL even when we are so $_SERVER['HTTPS']
		// cannot be relied upon.
		if (empty($_SERVER['HTTP_X_SSL']) && empty($_SERVER['HTTPS'])) {

			$sslOn = Configure::read('enableSSL');

			return $sslOn ? $this->redirect('https://' . env('HTTP_HOST') . $this->here) : false;

		}

		return;
	}

/**
 * Defines the fields displayed in a filter form.
 *
 * Defaults to id, display name, is_active, created and modified fields.
 *
 * Override in your controller to customise.
 *
 * Format:
 *
 * 'id' => array(
 *         'label' => label text,
 *         'type' => data type of control
 *         'compare' => SQL query. e.g. Model.field='%s' (where %s is replace by the value of the field)
 *         'default' => value you want the data filtering by by default
 * )
 */
	protected function _adminFilterFields() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$fields = parent::_adminFilterFields();

		unset($fields[$Model->alias . '.name']);
		unset($fields[$Model->alias . '.created']);
		unset($fields[$Model->alias . '.modified']);

		$fields[$Model->alias . '.id']['label'] = 'Order Ref';

		$fields[$Model->alias . '.order_status_id'] = array(
			'label' => 'Order Status',
			'type' => 'select',
			'compare' => array(
				$Model->alias . '.order_status_id' => '%s'
			)
		);

		$fields[$Model->alias . '.email'] = array(
			'label' => 'Customer Email',
			'type' => 'string',
			'compare' => array(
				$Model->alias . '.email LIKE' => '%%%s%%'
			)
		);

		$fields[$Model->alias . '.last_name'] = array(
			'label' => 'Customer Surname',
			'type' => 'string',
			'compare' => array(
				$Model->alias . '.last_name LIKE' => '%%%s%%'
			)
		);

		$fields[$Model->alias . '.order_time_from'] = array(
			'label' => 'Placed After',
			'type' => 'daterange_from',
			'compare' => array(
				$Model->alias . '.order_time >' => '%s'
			)
		);

		$fields[$Model->alias . '.order_time_to'] = array(
			'label' => 'Placed Before',
			'type' => 'daterange_to',
			'compare' => array(
				$Model->alias . '.order_time <' => '%s'
			)
		);

		return $fields;
	}

	protected function _processFilter($adminFilter = true) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$conditions = parent::_processFilter($adminFilter);

		if ($adminFilter === true) {

			// We only want to view orders that have a user assigned and contain items.
			$conditions[$modelAlias . '.user_id <>'] = null;
			$conditions[$modelAlias . '.items_total >'] = 0;

		}

		return $conditions;
	}

	protected function _adminIndexColumns() {
		$Model = $this->{$this->modelClass};

		return array(
			$Model->alias . '.id' => array(
				'type' => 'integer',
				'null' => false,
				'default' => null,
				'length' => (int)11,
				'key' => 'primary',
				'label' => 'Order Ref'
			),
			$Model->alias . '.order_time' => array(
				'type' => 'datetime',
				'null' => true,
				'default' => null,
				'length' => null,
				'label' => 'Order Placed'
			),
			$Model->alias . '.first_name' => array(
				'type' => 'string',
				'label' => 'First Name'
			),
			$Model->alias . '.last_name' => array(
				'type' => 'string',
				'label' => 'Last Name'
			),
			$Model->alias . '.email' => array(
				'type' => 'string',
				'label' => 'Email'
			),
			'OrderStatus.name' => array(
				'type' => 'string',
				'label' => 'Status'
			),
			$Model->alias . '.total' => array(
				'type' => 'currency',
				'label' => 'Total'
			),
			$Model->alias . '.modified' => array(
				'type' => 'datetime',
				'null' => true,
				'default' => null,
				'length' => null,
				'label' => 'Last Modified'
			),
		);
	}

	protected function _adminPopulateLookups() {
		$Model = $this->{$this->modelClass};

		parent::_adminPopulateLookups();

		$this->set('orderStatuses', $Model->OrderStatus->find('list'));
	}

/**
 * Defines the buttons in the toolbar displayed on an admin_form
 *
 * Override in your controller to customise
 */
	protected function _adminFormToolbar($id = null) {
		$toolbar = array();

		if ($id !== null) {

			$toolbar['Print'] = array(
				'url' => array(
					'admin' => false,
					'plugin' => 'ev_checkout',
					'controller' => 'orders',
					'action' => 'receipt',
					$id
				)
			);

		}

		return $toolbar;
	}

	protected function _adminIndexActions() {
		$actions = array(
			'Edit'
		);

		return $actions;
	}

	protected function _adminIndexToolbar() {
		return array();
	}

/**
 * Pagination settings for admin_index
 *
 * Override in your own controller to customise
 *
 * @return array
 */
	protected function _adminIndexPaginate() {
		$Model = $this->{$this->modelClass};

		$query = parent::_adminIndexPaginate();

		// Hide orders in basket unless filtering by the order status (user may want to view
		// these orders).
		if (empty($query['conditions']['Order.order_status_id'])) {
			$query['conditions'][$Model->alias . '.order_status_id <>'] = OrderStatus::IN_BASKET;
		}

		$query['contain'][] = 'Customer';
		$query['contain'][] = 'OrderStatus';

		return $query;
	}

	public function admin_edit($id = null) {
		// No add so redirect if $id is null
		if ($id === null) {

			$this->Session->setFlash(array(
				'title' => 'Cannot find order',
				'description' => 'Could not find order, please search for the order in the listing below.'
			), 'flash_fail');

			return $this->redirect(array('action' => 'index'));

		}

		parent::admin_edit($id);

		// If the order is no longer in the basket remove the option from the order statuses.
		if ((int)$this->request->data['Order']['order_status_id'] !== OrderStatus::IN_BASKET) {
			unset($this->viewVars['orderStatuses'][OrderStatus::IN_BASKET]);
		}

		$this->view = 'EvCheckout.Orders/admin_form';

		return;
	}

	public function admin_print($id = null) {
		$this->view($id);

		$this->view = 'view';
		$this->layout = 'print';

		$this->set('title_for_layout', 'Print Order | ' . Configure::read('SiteSetting.site_title'));
	}

	public function admin_resendConfirmationEmail($id) {
		$Model = $this->{$this->modelClass};
		$Model->emailConfirmation($id);
		$this->Session->setFlash('Email Sent', 'flash_success');
		return $this->redirect(array('action' => 'edit', $id));
	}

	public function admin_confirmationemail($id) {
		$Model = $this->{$this->modelClass};
		$order = $Model->readForEdit($id);
		$this->view = 'confirmation';
		$this->set(array(
			'order' => $order,
			'delayed' => false,
			'telephone' => Configure::read('SiteSetting.telephone_number'),
			'sortCode' => Configure::read('SiteSetting.bank_sortcode'),
			'accountNumber' => Configure::read('SiteSetting.bank_account')
		));
	}

/*
 * BASKET AND CHECKOUT METHODS
 */

/**
 * Displays the cart
 *
 * @return void
 */
	public function basket() {
		$Model = $this->{$this->modelClass};

		if (!empty($this->request->data)) {
			$Model->updateBasketItems($this->request->data);
		}

		$cart = $Model->getCart(null, false, false, true);

		if (!empty($cart['Order']['cart_modified'])) {
			$this->Session->setFlash(__('The quantities of some items in your basket have changed due to availability'), 'flash_info');
		}

		$this->set('cart', $cart);
		$this->set('continueShoppingUrl', $this->_getContinueShoppingUrl());

		$this->view = 'EvCheckout.Orders/basket';

		return;
	}

/**
 * Gets the continue shopping url, this will either be defined by the Session
 * EvCheckout.lastListingPage or the default listing url ("/").
 *
 * @return string
 */
	protected function _getContinueShoppingUrl() {
		if ($this->Session->read('EvCheckout.lastListingPage')) {

			return $this->Session->read('EvCheckout.lastListingPage');

		} else {

			return "/";

		}
	}

/**
 * Removes an item from the cart which has the given item ID
 * @param  integer $itemId OrderItem::id
 * @return void
 */
	public function remove_item($itemId) {
		$Model = $this->{$this->modelClass};

		if ($Model->removeItem($itemId)) {
			$this->Session->setFlash(__('Item removed'), 'flash_success');
		} else {
			$this->Session->setFlash(__('Item cannot be removed'), 'flash_fail');
		}

		$this->redirect(array('action' => 'basket'));

		return;
	}

/**
 * Handles user login and registration
 */
	public function register() {
		$Model = $this->{$this->modelClass};

		$cart = $this->_populateCart();

		// If the cart is empty redirect to the basket page.
		if (empty($cart['OrderItem'])) {
			return $this->redirect(array('action' => 'basket'));
		}

		// If user is logged in skip this step and move onto the next step
		$user = $this->Auth->user();

		if (!empty($user)) {

			$this->redirect(array(
				'action' => $this->checkoutProcess['register']['next']
			));

			return;

		}

		// Else show the login and registration page.
		// We setup the login redirect here because if the customer chooses to login that
		// is handled by the UsersController
		$this->Session->write('Auth.redirect',
			Router::url(array(
				'action' => $this->checkoutProcess['register']['next']
			))
		);

		$Model = $this->{$this->modelClass};

		// User has submitted registration or login form
		if (!empty($this->request->data['Customer'])) {

			// Process user registration
			$userRegistrationData = $this->request->data;

			$userId = $Model->Customer->register($userRegistrationData);

			// Check the data validates
			if ($userId !== false) {

				if ($userId === 0) {
					$data = array(
						'Order' => array(
							'id' => $cart['Order']['id'],
							'first_name' => $this->request->data['Customer']['first_name'],
							'last_name' => $this->request->data['Customer']['last_name'],
							'email' => $this->request->data['Customer']['email'],
							'phone' => $this->request->data['Customer']['phone']
						)
					);
					$Model->save($data);
				}

				// Sign in user and send to delivery page
				$fullUser = $Model->Customer->getUserForLogin($userId);

				// Log user in and link order
				$this->Auth->login($fullUser);
				$Model->setUser($fullUser['User']['id']);

				return $this->redirect($this->Auth->redirect());

			} else {

				$this->set('errors', $Model->Customer->validationErrors);

			}

		}

		$this->view = 'EvCheckout.Orders/register';

		return;
	}

	public function delivery() {
		$Model = $this->{$this->modelClass};
		$cart = $this->_populateCart();

		// If the cart is empty redirect to the basket page.
		if (empty($cart['OrderItem'])) {
			return $this->redirect(array('action' => 'basket'));
		}

		$this->_processStage('delivery');

		$currentDeliveryAddress = !empty($cart['DeliveryAddress']) ? $cart['DeliveryAddress']['id'] : null;

		$this->_populatePreviousDeliveryAddresses($currentDeliveryAddress);

		return;
	}

	public function invoice() {
		$Model = $this->{$this->modelClass};
		$cart = $this->_populateCart();

		// If the cart is empty redirect to the basket page.
		if (empty($cart['OrderItem'])) {
			return $this->redirect(array('action' => 'basket'));
		}

		// Make sure we've set the order's customer before proceeding with payment.
		$Model->setUser(CakeSession::read('Auth.User.User.id'));

		$this->_processStage('invoice');

		$currentInvoiceAddress = !empty($cart['InvoiceAddress']) ? $cart['InvoiceAddress']['id'] : null;

		$this->_populateLookupsInvoice();
		$this->_populatePreviousInvoiceAddresses($currentInvoiceAddress);

		return;
	}

	public function offsite_payment() {
		$this->_processStage('offsite_payment');
		return;
	}

	protected function _processStage($stage, $redirect = true) {
		$success = true;

		$stages = $this->checkoutProcess;
		$cart = $this->_populateCart();

		foreach ($stages[$stage]['process'] as $process) {

			if (!$this->{$process}()) {

				$success = false;

			}

		}

		if ($success && $redirect) {

			if ($stages[$stage]['next'] == 'success') {

				$this->redirect(array(
					'action' => $stages[$stage]['next'],
					$cart['Order']['id']
				));

			} else {

				$this->redirect(array(
					'action' => $stages[$stage]['next']
				));

			}
		}

		if (isset($stages[$stage]['template'])) {

			$template = $stages[$stage]['template'];

		} else {

			$template = $stage;

		}

		$this->view = 'EvCheckout.Orders/' . $template;

		return $success;
	}

	public function confirm() {
		$Model = $this->{$this->modelClass};
		$cart = $this->_populateCart();

		// If the cart is empty redirect to the basket page.
		if (empty($cart['OrderItem'])) {
			return $this->redirect(array('action' => 'basket'));
		}

		$this->_processStage('confirm');

		$stages = $this->checkoutProcess;
		$this->set('nextStage', $stages['confirm']['next']);

		$this->set('cart', $cart);

		$this->view = 'EvCheckout.Orders/confirm';

		return;
	}

/**
 * Checkout complete.
 *
 * @param integer $id
 * @return void
 */
	public function success($id) {
		$Model = $this->{$this->modelClass};

		$cart = $Model->readForView($id);

		// Check the order has been completed before displaying.
		if (empty($cart[$Model->alias]['order_status_id']) || $cart[$Model->alias]['order_status_id'] === OrderStatus::IN_BASKET) {

			throw new NotFoundException();

		}

		// Check the user is authorised to view the page.

		$user = $this->Auth->user();

		if (empty($user) || ($user['User']['id'] != $cart[$Model->alias]['user_id'] && (!isset($user['User']['user_group_id']) || $user['User']['user_group_id'] > 2))) {

			throw new ForbiddenException();

		}

		$this->set('cart', $cart);

		// Get the content for the page
		if (Configure::read('EvCheckout.confirmation_page_id')) {

			$Page = ClassRegistry::init('Page');
			$data = $Page->readForView(Configure::read('EvCheckout.confirmation_page_id'));

			$this->set('data', $data);

		}

		$this->view = 'EvCheckout.Orders/success';

		return;
	}

/**
 * @return void
 */
	public function onepage() {
		$Model = $this->{$this->modelClass};
		$cart = $Model->getCart();

		$this->_populateCart();
		$this->_populateLookupsInvoice();

		$this->_processDeliveryAddress();

		$this->_processInvoiceAddress();

		if ($this->_processCreditCard()) {

			// Payment taken
			$this->redirect(
				array(
					'action' => 'success',
					$cart['Order']['id']
				)
			);

		}

		return;
	}

/**
 * Processes a delivery address passed in from request->data['DeliveryAddress']
 * Handles either setting order address to previous address or adding a new one and attaching to order
 * @return boolean True on success
 */
	protected function _processDeliveryAddress() {
		$Model = $this->{$this->modelClass};

		if (!empty($this->request->data)) {

			$userId = $this->Auth->User('User.id');
			$orderId = $Model->getCartId();

			if (!empty($this->request->data['DeliveryAddress']['id'])) {

				return $Model->setDeliveryAddress($this->request->data['DeliveryAddress']['id']);

			} else {

				return $Model->DeliveryAddress->createAddress($userId, $this->request->data, $orderId);

			}

		}

		return false;
	}

/**
 * Payment callback
 *
 * @param integer $orderId
 * @return void
 */
	public function paymentCallback($orderId) {
		$Model = $this->{$this->modelClass};

		$transaction = $this->Transactions->checkPayment();

		if ($transaction) {

			$Model->markAsPaid($orderId);

			// Payment taken
			$this->redirect(
				array(
					'action' => 'success',
					$orderId
				)
			);

			return;

		} else {

			$this->Session->setFlash(__('Processing your payment failed, please check your details and try again.'), 'flash_fail');

			$this->redirect(array('action' => 'invoice'));

			return;

		}

		die();
	}

/**
 * Callback for SagePay Server
 */
	public function sagepayServerCallback() {
		$paymentSuccessful = $this->Transactions->checkPayment();

		$Model = $this->{$this->modelClass};

		$this->SagepayServer = $this->Components->load('Transactions.SagepayServer');

		if ($paymentSuccessful) {

			$this->loadModel('Transactions.Transaction');
			$orderId = $this->Transaction->field('model_id', array('payment_token' => $this->request->data['VPSTxId']));

			$Model->markAsPaid($orderId);

			$redirect = Router::url(
				array(
					'action' => 'success',
					$orderId
				),
				true
			);

		} else {

			$redirect = Router::url(
				array(
					'action' => 'failure',
					(! empty($this->request->data['VPSTxId'])) ? $this->request->data['VPSTxId'] : ''
				),
				true
			);
		}


		// Tell sagepay where to redirect the user to
		$this->SagepayServer->postBack($redirect);

		exit;
	}

	public function failure($transactionRef = false) {
		$this->view = 'EvCheckout.Orders/failure';

		return;
	}

/**
 * Processes OnSite credit card payments
 *
 * @return boolean
 */
	protected function _processCreditCard() {
		$Model = $this->{$this->modelClass};

		// If credit card information is not set we cannot process
		if (empty($this->request->data['CreditCard'])) {
			return false;
		}

		// Split name into first and second
		$fullName = trim($this->request->data['CreditCard']['name']);
		$splitName = explode(" ", $fullName);
		$this->request->data['CreditCard']['last_name'] = array_pop($splitName);

		if (count($splitName)) {

			$this->request->data['CreditCard']['first_name'] = implode(" ", $splitName);

		} else {

			$this->request->data['CreditCard']['first_name'] = '';

		}


		//Check that card validates
		$this->loadModel('EvCheckout.CreditCard');
		$this->CreditCard->set($this->request->data);
		if ($this->CreditCard->validates()) {

			$cart = $Model->getCart();
			$transaction = $this->Transactions->takePayment(
				array(
					'return' => Router::url(array(
						'action' => 'paymentCallback',
						$cart['Order']['id']
					), true),
					'cancel' => null
				),
				array(
					'model' => 'EvCheckout.Order',
					'model_id' => $cart['Order']['id']
				),
				$cart['Order']['total'],
				array(

				),
				array(
					'card' => $this->request->data['CreditCard'],
					'address' => $cart['InvoiceAddress'],
					'user' => $this->Auth->user('User'),
					'description' => __('[%s] Order %s', Configure::read('SiteSetting.site_title'), $cart['Order']['invoice_number'])
				),
				$cart['Order']['user_id']
				// 'sagepay_direct'
			);

			if (isset($transaction['result']) && $transaction['result']) {

				$Model->markAsPaid($cart['Order']['id']);
				return true;

			} else {

				$this->Session->setFlash(__('Processing your payment failed, please check your details and try again.'), 'flash_fail');

			}

		}

		return false;
	}

	protected function _processOffsitePayment() {
		$Model = $this->{$this->modelClass};

		$cart = $Model->getCart();

		$user = $this->Auth->user('User');

		$transaction = $this->Transactions->takePayment(
			array(
				'return' => Router::url(array(
					'action' => 'sagepayServerCallback'
				), true),
				'error' => Router::url(array(
					'action' => 'confirm'
				), true)
			),
			array(
				'model' => 'EvCheckout.Order',
				'model_id' => $cart['Order']['id']
			),
			$cart['Order']['total'],
			array(

			),
			array(
				'user_details' => array(
					'firstname' => $user['first_name'],
					'lastname' => $user['last_name'],
					'email' => $user['email']
				),
				'billing_address' => array(
					'address1' => $cart['InvoiceAddress']['address_line_1'],
					'address2' => $cart['InvoiceAddress']['address_line_2'],
					'city' => $cart['InvoiceAddress']['city'],
					'postcode' => $cart['InvoiceAddress']['postcode'],
					'country_code' => 'GB'
				),
				'user' => $this->Auth->user('User'),
				'description' => __('[%s] Order %s', Configure::read('SiteSetting.site_title'), $cart['Order']['invoice_number'])
			),
			$cart['Order']['user_id'],
			'sagepay_server'
		);

		if (isset($transaction['result']) && $transaction['result']) {

			$Model->markAsPaid($cart['Order']['id']);
			return true;

		} else {

			$this->Session->setFlash(__('Processing your payment failed, please check your details and try again.'), 'flash_fail');

		}

		return;
	}

	protected function _processOnsitePayment() {
		return $this->_processCreditCard();
	}

	protected function _processInvoiceAddress() {
		$invoiceAddressSaved = false;

		if (!empty($this->request->data)) {

			$Model = $this->{$this->modelClass};

			$orderId = $Model->getCartId();
			$userId = $this->Auth->user('User.id');

			$cart = $Model->getCart();

			//Either mark delivery address as invoice as well or create new address
			if (
				isset($this->request->data['InvoiceAddress']['match_delivery_address'])
				&& $this->request->data['InvoiceAddress']['match_delivery_address'] == 1
			) {

				$Model->setInvoiceAddress($cart['DeliveryAddress']['id']);
				$invoiceAddressSaved = true;

			} elseif (! empty($this->request->data['InvoiceAddress']['invoice_address_id'])) {

				$Model->setInvoiceAddress($this->request->data['InvoiceAddress']['invoice_address_id']);
				$invoiceAddressSaved = true;

			} else {

				if ($Model->InvoiceAddress->createAddress($userId, $this->request->data, $orderId)) {

					$invoiceAddressSaved = true;

				}

			}

		}

		return $invoiceAddressSaved;
	}

/**
 * View a completed order. Only authorised users can view an order.
 *
 * @param integer $id order ID
 * @return void
 */
	public function view($id) {
		$Model = $this->{$this->modelClass};

		$cart = $Model->readForView($id);

		if (empty($cart[$Model->alias]['order_status_id']) || $cart[$Model->alias]['order_status_id'] === OrderStatus::IN_BASKET) {

			throw new NotFoundException();

		}

		$user = $this->Auth->user();

		if (empty($user) || ($user['User']['id'] !== $cart[$Model->alias]['user_id'] && $user['User']['user_group_id'] > 2)) {

			throw new ForbiddenException();

		}

		$this->set('data', $cart);

		$this->view = 'EvCheckout.Orders/view';

		return;
	}

/**
 * Order receipt view for printing out an order.
 *
 * @param integer $id
 * @return void
 */
	public function receipt($id) {
		$this->view($id);

		$this->view = 'EvCheckout.Orders/view';
		$this->layout = 'EvCheckout.print';

		$this->set('title_for_layout', 'Print Order');

		return;
	}

	protected function _populateCart() {
		$Model = $this->{$this->modelClass};
		$cart = $Model->getCart();
		$this->set('cart', $cart);

		return $cart;
	}

	protected function _populatePreviousInvoiceAddresses($currentInvoiceAddress) {
		$Model = $this->{$this->modelClass};
		$userId = $this->Auth->User('User.id');
		$invoice_addresses = $Model->InvoiceAddress->recentInvoiceAddresses($userId, $currentInvoiceAddress, 5);
		$this->set(compact('invoice_addresses'));
	}

	protected function _populatePreviousDeliveryAddresses($currentDeliveryAddress) {
		$Model = $this->{$this->modelClass};
		$userId = $this->Auth->User('User.id');
		$delivery_addresses = $Model->DeliveryAddress->recentDeliveryAddresses($userId, $currentDeliveryAddress, 5);
		$this->set(compact('delivery_addresses'));
	}

	protected function _populateLookupsInvoice() {
		//Available months
		$months = $this->_getMonthsList();
		$this->set('expiryMonths', $months);
		$this->set('startMonths', $months);
		//Expiry months
		$this->set('expiryYears', $this->_getExpiryYearsList());
		//Start months
		$this->set('startYears', $this->_getStartYearsList());
	}

	protected function _getMonthsList() {
		$months = array_map(array($this, '_padArray'), range(1, 12));
		return array_combine($months, $months);
	}

	protected function _getExpiryYearsList() {
		$expiryYears = range(date("y"), date("y") + 10);
		$expiryYearsLong = range(date("Y"), date("Y") + 10);
		return array_combine($expiryYears, $expiryYearsLong);
	}

	protected function _getStartYearsList() {
		$startYears = range(date("y"), date("y") - 10);
		$startYearsLong = range(date("Y"), date("Y") - 10);
		return array_combine($startYears, $startYearsLong);
	}

/**
 * Used to pad out month and year arrays from 1 to 01
 * @param  mixed 	$number [description]
 * @return string 	Padded array
 */
	protected function _padArray($number) {
		return str_pad($number, 2, '0', STR_PAD_LEFT);
	}

}
