<?php

App::uses('EvCategoryAppModel', 'EvCategory.Model');

class Category extends EvCategoryAppModel {

	public $actsAs = array(
		'Tree',
		'MetaData.Meta',
		'Routable.Routable' => array(
			'alias' => 'category/:displayField',
			'config' => 'EvCategory'
		)
	);

	public $belongsTo = array(
		'ProductAttribute' => array(
			'className' => 'EvShop.ProductAttribute'
		),
		'ParentCategory' => array(
			'className' => 'EvCategory.Category',
			'foreignKey' => 'parent_id',
		)
	);

	public $hasMany = array(
		'ChildCategory' => array(
			'className' => 'EvCategory.Category',
			'foreignKey' => 'parent_id',
			'cascade' => true,
			'dependent' => true
		)
	);

	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'Category name is mandatory'
			),
			'unique' => array(
				'rule' => 'isUnique',
				'message' => 'Category name must be unique'
			)
		)
	);

	public $imageSlots = array(
		'main' => array(
			'slots' => 1,
			'fields' => array(
				'alt'
			)
		)
	);



/**
 * We need to save additional routes for the category- product attributes
 *
 * @see RoutableBehavior::afterSave()
 */
	public function afterSave($created, $options=array()) {

		$category = $this->data;

		$this->rebuildCategoryProductAttributeRoutes($category);

		return;

	}

/**
 * Rebuilds routes for a category including any product attribute - category combinations the category may have
 * @param  array $category The category array
 */
	public function rebuildCategoryProductAttributeRoutes($category) {

		//If this category has attributes underneath it instead of subcategories we need to build routes for those.
		if(!empty($category['Category']['product_attribute_id'])) {

			//Get Product attribute options
			$productAttributeOptions = $this->ProductAttribute->ProductAttributeOption->find('all', array(
					'conditions' => array(
						'ProductAttributeOption.product_attribute_id' => $category['Category']['product_attribute_id']
					)
				)
			);

			$Route = ClassRegistry::init('Routable.Route');

			foreach($productAttributeOptions as $key => $productAttributeOption) {

				$Route->create();

				$data = array();

				$data['Route']['actual'] = "ev_shop/products/index/category:{$category['Category']['id']}/ProductAttributeOption:{$productAttributeOption['ProductAttributeOption']['id']}";

				$data['Route']['alias'] = $category['Route']['alias'].'/'.strtolower(Inflector::slug($productAttributeOption['ProductAttributeOption']['name'], "-"));

				$data['Route']['model'] = 'Category';


				//Check if there is an existing route if so we will update it instead
				$existingRoute = $Route->findByActual($data['Route']['actual']);
				if($existingRoute) {

					$Route->id = $existingRoute['Route']['id'];

				}


				$Route->save($data);

			}

		}

	}


/**
 * Gets the list of available categories
 */
	public function getCategories($findType = 'all') {

		$categories = $this->find($findType, array(
				'conditions' => array(
					$this->alias.".is_active" => 1
				)
			)
		);

		return $categories;

	}

/**
 * Builds the menu for the front end including subcategories and attributes
 */
	public function buildMenu() {

		$categories = $this->find('threaded', array(
				'conditions' => array(
					$this->alias.".is_active" => 1
				),
				'order' => 'lft ASC'
			)
		);


		foreach($categories as $key=>$category) {

			if($category['Category']['product_attribute_id']) {

				$categories[$key]['ProductAttributeOptions'] = $this->ProductAttribute->getAttributeOptions($category['Category']['product_attribute_id'], $category['Category']['id']);

			}

		}

		return $categories;

	}

}
