<?php

App::uses('EvCaseStudiesAppController', 'EvCaseStudies.Controller');

class CaseStudiesController extends EvCaseStudiesAppController {

/**
 * Paginator defaults for the app
 *
 * @var array
 */
	public $paginate = [
		'conditions' => array(
			'CaseStudy.is_active' => 1,
			'CaseStudy.created <= NOW()'
		),
		'order' => 'CaseStudy.sequence ASC',
		'contain' => array(
			'CaseStudyCategory'
		)
	];

/**
 * Paginator defaults for admin.
 *
 * @var array
 */
	public $adminPaginate = [
		'contain' => [
			'CaseStudyCategory' => [
				'fields' => [
					'CaseStudyCategory.id',
					'CaseStudyCategory.name'
				]
			]
		]
	];

/**
 * Constructor.
 *
 * @param CakeRequest $Request Request object for this controller. Can be null for testing,
 *  but expect that features that use the request parameters will not work.
 * @param CakeResponse $Response Response object for this controller.
 * @return void
 */
	public function __construct(CakeRequest $Request, CakeResponse $Response) {
		parent::__construct($Request, $Response);

		// add Paginator component
		$this->components[] = 'Paginator';

		// update the paginate limit from the plugin config
		$this->paginate['limit'] = Configure::read('EvCaseStudies.page_limit');
	}

	protected function _adminPopulateLookups() {
		$this->loadModel('EvCaseStudies.CaseStudyCategories');
		$caseStudyCategories = $this->CaseStudyCategories->find('list');

		$this->set('caseStudyCategories', $caseStudyCategories);
	}

	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(array(
			'index',
			'view'
		));
	}

/**
 * Defines the columns displayed in the admin_index results table
 * using a columns whitelist.
 *
 * @return array containing the index columns
 */
	protected function _adminIndexColumns() {
		$indexColumns = parent::_adminIndexColumns();

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		// add additional index columns
		$additionalIndexColumns = [
			'CaseStudyCategory.name' => [
				'label' => 'Category',
				'type' => 'string',
			]
		];

		return ArrayUtil::addAfter($indexColumns, "$modelAlias.name", $additionalIndexColumns);
	}

/**
 * Defines the fields displayed in an admin_form for this model,
 * defaulting to all fields in the db table
 *
 * @return array containing the form fields for the template
 */
	protected function _adminFormFields() {
		$formFields = parent::_adminFormFields();

		$formFields[$this->{$this->modelClass}->alias . '.body']['type'] = 'html';

		// remove the template field when templates are not being used
		if (! $this->{$this->modelClass}->hasBehavior('EvTemplates.Template')) {
			unset($formFields[$this->{$this->modelClass}->alias . '.template_id']);
		}

		return $formFields;
	}

/**
 * Displays all active Case Studies with pagination
 *
 * @return void
 */
	public function index() {
		$Model = $this->{$this->modelClass};

		// see if there is a linked content page
		$pageId = Configure::read('EvCaseStudies.page_id');

		if (! empty($pageId)) {
			$this->assignPage($pageId, 'page');
		}

		$this->Paginator->settings = array_merge_recursive(
			$this->paginate,
			$this->_processCaseStudyParams($this->request->params)
		);

		// add images to the paginate query when used in the plugin
		if (! empty(Configure::read('EvCaseStudies.imageSlots'))) {
			$this->Paginator->settings = array_merge_recursive(
				$this->Paginator->settings,
				$Model->containSlotAssociations(
					['Image' => Configure::read('EvCaseStudies.imageSlots')]
				)
			);
		}

		// add documents to the paginate query when used in the plugin
		if (! empty(Configure::read('EvCaseStudies.documentSlots'))) {
			$this->Paginator->settings = array_merge_recursive(
				$this->Paginator->settings,
				$Model->containSlotAssociations(
					['Document' => Configure::read('EvCaseStudies.documentSlots')]
				)
			);
		}

		$this->set('data', $this->Paginator->paginate());

		// call the widgets
		$this->widgetLatestCaseStudies();
		$this->widgetCategories();
		$this->widgetFilterByCategory();
		$this->widgetFilterByDate();

		$this->view = 'EvCaseStudies.CaseStudies/index';
	}

/**
 * Displays a specific case study based on its id
 *
 * @param  int $id ID of the case study to display
 * @throws NotFoundException when no case study data is returned for the requested id
 * @return void
 */
	public function view($id = null) {
		$Model = $this->{$this->modelClass};

		if ($id) {
			$data = $Model->readForView($id);
		}

		if (empty($data[$Model->alias])) {
			throw new NotFoundException();
		}

		$this->Meta->set($data);

		$this->set('data', $data);

		// call the widgets
		$this->widgetLatestCaseStudies();
		$this->widgetCategories();

		if ($Model->hasBehavior('EvTemplates.Template')) {
			$templateView = $this->Tpl->getTemplate($data, $Model->alias);
		}

		$this->view = ! empty($templateView) ? $templateView : 'EvCaseStudies.CaseStudies/view';
	}

/**
 * Loop around the named portion of the supplied request parameters and append
 * requests to the current params array, in preparation for the database query
 *
 * @param array $requestParams Contains the current request parameters
 * @param array $currentParams Contains the current parameters
 * @return array Contains the combined named request and current parameters
 */
	protected function _processCaseStudyParams(array $requestParams = [], array $currentParams = []) {
		$ignoreParams = [
			'page',
			'named'
		];

		$dateParams = [
			'date',
			'startDate',
			'endDate'
		];

		$categoryIds = [];

		if (! empty($requestParams['named'])) {
			foreach ($requestParams['named'] as $param => $paramValue) {
				if (!in_array($param, $ignoreParams)) {

					//Check it isn't a date filter
					if (in_array($param, $dateParams)) {
						if ($param == 'startDate') {
							//Query for case studies published after this date
							$currentParams['conditions'][]['CaseStudy.published >='] = $paramValue;
						} elseif ($param == 'endDate') {
							//Query for case studies published before this date
							$currentParams['conditions'][]['CaseStudy.published <='] = $paramValue;
						}
					} else {
						$categoryIds[] = $paramValue;
					}

				}
			}
		}

		if (! empty($categoryIds)) {
			$currentParams['conditions']['CaseStudy.case_study_category_id'] = $categoryIds;
		}

		return $currentParams;
	}

}
