<?php

namespace Nomensa\BulkInserter;

use Storage;
use DB;
use Nomensa\BulkInserter\Exceptions\InvalidRowException;

/**
 * A super-fast way of inserting many items by creating a .sql dump file and importing it.
 * This method is more fragile than the Eloquent methods and not intended for normal app logic
 * but in the context of a mass-item seeder for tests it's a very quick.
 */
class BulkInserter
{

    const EXTENSION = '.sql';

    public $chunk_limit = 50;

    public $foldername = 'sql';

    /** @var string table name */
    public $table = '';

    /** @var bool Whether or not to completely empty the table first */
    public $truncate_first = false;

    public $columns = [];

    /** @var array  */
    public $rows = [];


    /**
     * SQLDumpImporter constructor.
     *
     * @param string $table - Name of the table
     * @param array $columns
     */
    public function __construct($table, array $columns)
    {
        $this->table = $table;
        $this->columns = $columns;
    }


    public function makeTruncateStatement()
    {
        return 'TRUNCATE TABLE `' . $this->table . '`; ' . PHP_EOL . PHP_EOL;
    }


    public function makeFirstBit(): string
    {
        return 'INSERT INTO `' . $this->table . '` (' . implode(',', $this->columns) . ') VALUES ' . PHP_EOL;
    }


    /**
     * @param array $rows
     *
     * @return string
     */
    private function makeRowBit($rows): string
    {
        return implode(',' . PHP_EOL, $rows) . ';' . PHP_EOL;
    }


    public function makeInsertBits(): string
    {
        $contents = '';

        if (count($this->rows) > $this->chunk_limit) {
            $chunks = array_chunk($this->rows, $this->chunk_limit);
            foreach ($chunks as $chunk) {
                $contents .= $this->makeFirstBit() . $this->makeRowBit($chunk) . PHP_EOL . PHP_EOL;
            }
        } else {
            $contents .= $this->makeFirstBit() . $this->makeRowBit($this->rows) . PHP_EOL;
        }

        return $contents;
    }


    public function makeContent()
    {
        $fileContents = '# This file generated by ' . self::class . PHP_EOL . PHP_EOL;

        if ($this->truncate_first) {
            $fileContents .= $this->makeTruncateStatement();
        }

        $fileContents .= $this->makeInsertBits();

        return $fileContents;
    }


    /**
     * @param bool $delete_file - Set to false during debugging
     *
     * @throws \Illuminate\Contracts\Filesystem\FileNotFoundException
     */
    public function insert(bool $delete_file=true)
    {
        $fileContents = $this->makeContent();

        $path = $this->foldername . '/' . $this->table . '_table' . self::EXTENSION;

        dump('Writing MySQL dump file (' . $path . ')');
        Storage::put($path, $fileContents);

        dump('Importing MySQL dump file');
        DB::unprepared(Storage::get($path));

        if ($delete_file) {
            dump('Deleting file (' . $path . ')');
            Storage::delete($path);
        }

        dump('Finished importing into ' . $this->table);
    }


    /**
     * @param string $row - Some like "( 23, 'Horse Racing', 'Giddy up' )"
     *
     * @throws InvalidRowException
     */
    public function addRow($row)
    {
        if (substr_count($row,',') < count($this->columns)-1) {
            throw new InvalidRowException('Row contains fewer values than there are columns');
        }
        $this->rows[] = $row;
    }


}
