<?php

App::uses('CakeEventListener', 'Event');
App::uses('BulkDiscountLib', 'EvBulkDiscount.Lib');

class BulkDiscountBasketListener implements CakeEventListener {

	protected $_subject = null;

	protected $_itemFields = [
		'model',
		'model_id',
		'unitPrice' => 'unit_price',
		'taxRate' => 'tax_rate',
		'unitTax' => 'unit_tax',
		'unit_price_incTax',
		'quantity',
		'row_total',
		'row_total_incTax',
	];

	public function implementedEvents() {
		return array(
			'EvBasket.Component.Basket.itemAdd' => array(
				'callable' => 'bulkDiscount',
				'priority' => 1
			),
			'EvBasket.Component.Basket.itemUpdated' => array(
				'callable' => 'bulkDiscount',
				'priority' => 1
			),
			'EvBasket.Component.Basket.itemDeleted' => array(
				'callable' => 'bulkDiscount',
				'priority' => 1
			)
		);
	}

	/**
	 * rebuild the bulk discount in the basket
	 * $Event->subject() = BasketManager Component
	 */
	public function bulkDiscount(CakeEvent $Event) {
		if (! CakePlugin::loaded('EvBasket')) {
			return false;
		}

		$this->_subject = $Event->subject();

		$basket = $Event->subject()->getBasket(true);

		if (empty($basket['BasketItem'])) {
			return;
		}

		$productIds = $this->_getBasketProductIds($basket);

		$discounts = $this->_getDiscounts($productIds);

		$basketQuantities = $this->_getBasketQuantities($basket);

		$originalUnitPrices = $this->_getOriginalUnitPrices();

		$itemsToUpdate = $this->_calculateBulkDiscounts(
			$basket,
			$basketQuantities,
			$discounts,
			$originalUnitPrices
		);

		if (!empty($itemsToUpdate)) {
			$this->_subject->updateItem($itemsToUpdate, null, null, 0, false);
		}

		$this->_setOriginalUnitPrices($originalUnitPrices);
	}

	protected function _getBasketProductIds($basket) {
		// get the product ids from the basket
		return Hash::extract($basket, 'BasketItem.{s}.Variant.product_id');
	}

	protected function _getDiscounts($productIds = []) {
		$this->_subject->_controller->loadModel('EvBulkDiscount.BulkDiscount');

		// get any bulk discountg data
		// $BulkDiscount = EvClassRegistry::init('EvBulkDiscount.BulkDiscount');
		$discounts = $this->_subject->_controller->BulkDiscount->find(
			'all',
			array(
				'conditions' => array(
					'BulkDiscount.model' => 'EvShop.Product',
					'BulkDiscount.model_id' => $productIds
				),
				'order' => 'BulkDiscount.model_id ASC, BulkDiscount.quantity ASC'
			)
		);

		$discounts = Hash::combine(
			$discounts,
			'{n}.BulkDiscount.quantity',
			'{n}',
			'{n}.BulkDiscount.model_id'
		);

		return $discounts;
	}

	protected function _getBasketQuantities($basket) {
		return Hash::combine(
			$basket['BasketItem'],
			array(
				'%s.%s',
				'{s}.model',
				'{s}.model_id'
			),
			'{s}.quantity',
			'{s}.Variant.product_id'
		);
	}

	protected function _getOriginalUnitPrices() {
		$originalUnitPrices = $this->_subject->_controller->BasketCookie->read('originalUnitPrices');

		if (empty($originalUnitPrices) && !is_array($originalUnitPrices)) {
			$originalUnitPrices = [];
		}

		return $originalUnitPrices;
	}

	protected function _setOriginalUnitPrices($originalUnitPrices) {
		$this->_subject->_controller->set('originalUnitPrices', $originalUnitPrices);

		if (empty($originalUnitPrices)) {
			$this->_subject->_controller->BasketCookie->delete('originalUnitPrices');
			$this->_subject->_controller->BasketManager->deleteData('original_price', false);
			return;
		}

		$this->_subject->_controller->BasketCookie->write(
			'originalUnitPrices',
			$originalUnitPrices
		);

		$this->_subject->_controller->BasketManager->setBasketData(
			'original_price',
			json_encode($originalUnitPrices),
			false
		);
	}

	protected function _calculateBulkDiscounts($basket, $basketQuantities, $discounts, &$originalUnitPrices) {
		$itemsToUpdate = [];

		foreach ($basketQuantities as $productId => $variants) {
			$bulkDiscount = $this->_findBulkDiscount($discounts, $productId, $variants);

			if (empty($bulkDiscount)) {
				$this->_resetAllOriginalUnitPrices($basket, $variants, $originalUnitPrices);
				continue;
			}

			foreach ($variants as $basketItemId => $quantity) {
				if (!empty($originalUnitPrices) && isset($originalUnitPrices[$basketItemId])) {
					$originalUnitPrice = $originalUnitPrices[$basketItemId];

					if ($basket['BasketItem'][$basketItemId]['unit_price'] != $originalUnitPrice['unit_price']) {
						$this->_resetOriginalUnitPrice($basket, $basketItemId, $originalUnitPrice);
					}
				} else {
					$originalUnitPrices = $this->_setOriginalUnitPrice($basket, $basketItemId, $originalUnitPrices);
				}

				$itemsToUpdate = array_merge(
					$itemsToUpdate,
					$this->_calculateBulkDiscount($basket, $basketItemId, $bulkDiscount)
				);
			}
		}

		return $itemsToUpdate;
	}

	protected function _findBulkDiscount($discounts, $productId, $variants) {
		$productQuantities = $this->_getProductQuantities($discounts, $productId);
		$totalProductQuantity = $this->_getProductTotalQuantity($variants);

		if (empty($productQuantities) || empty($totalProductQuantity)) {
			return [];
		}

		foreach ($productQuantities as $quantity) {
			if ($totalProductQuantity >= $quantity && !empty($discounts[$productId][$quantity])) {
				return $discounts[$productId][$quantity];
			}
		}

		return [];
	}

	protected function _calculateBulkDiscount($basket, $basketItemId, $bulkDiscount) {
		$discountedUnitPrice = BulkDiscountLib::calculateDiscount(
			$basket['BasketItem'][$basketItemId]['unit_price'],
			$bulkDiscount['BulkDiscount']['amount']
		);

		$unitTax = $this->_subject->_controller->BasketManager->BasketItem->calculateUnitTax(
			$discountedUnitPrice,
			$basket['BasketItem'][$basketItemId]['tax_rate']
		);

		$discountedUnitPrice = round($discountedUnitPrice, 2);

		return $this->_createUpdateData(
			$basket,
			$basketItemId,
			[
				'unitPrice' => $discountedUnitPrice,
				'unitTax' => $unitTax,
			]
		);
	}

	protected function _getProductQuantities($discounts, $productId) {
		if (empty($discounts[$productId])) {
			return [];
		}

		$quantities = array_keys($discounts[$productId]);
		rsort($quantities);

		return $quantities;
	}

	protected function _getProductTotalQuantity($variants) {
		return array_sum($variants);
	}

	protected function _setOriginalUnitPrice($basket, $basketItemId, &$originalUnitPrices) {
		//Add the current unit price to originalUnitPrices cookie
		$originalUnitPrices[$basketItemId] = $this->_createUpdateData($basket, $basketItemId);
	}

	protected function _resetAllOriginalUnitPrices($basket, $variants, $originalUnitPrices) {
		foreach ($variants as $basketItemId => $quantity) {
			if (empty($originalUnitPrices[$basketItemId])) {
				continue;
			}

			$this->_resetOriginalUnitPrice($basket, $basketItemId, $originalUnitPrices[$basketItemId]);
		}
	}

	protected function _resetOriginalUnitPrice($basket, $basketItemId, $originalUnitPrice) {
		$itemUnitPrice = $basket['BasketItem'][$basketItemId]['unit_price'];
		$itemUnitTax = $basket['BasketItem'][$basketItemId]['unit_tax'];

		if (!empty($originalUnitPrice)) {
			$itemUnitPrice = $originalUnitPrice['unit_price'];
			$itemUnitTax = $originalUnitPrice['unit_tax'];
		}

		$newUpdate = $this->_createUpdateData(
			$basket,
			$basketItemId,
			[
				'unitPrice' => $itemUnitPrice,
				'unitTax' => $itemUnitTax,
			]
		);

		$basket = $this->_subject->updateItem($newUpdate, null, null, 0, false);
	}

	protected function _createUpdateData($basket, $basketItemId, $newData = []) {
		$oldData = $this->_extractItemData($basket, $basketItemId);

		$updateData = array_merge(
			$oldData,
			$newData
		);

		return [$updateData];
	}

	protected function _extractItemData($basket, $basketItemId) {
		if (empty($basket['BasketItem'][$basketItemId])) {
			return [];
		}

		$itemData = [];
		foreach ($basket['BasketItem'][$basketItemId] as $field => $value) {
			if (!in_array($field, $this->_itemFields)) {
				continue;
			}

			if (is_string(array_search($field, $this->_itemFields))) {
				$field = array_search($field, $this->_itemFields);
			}

			$itemData[$field] = $value;
		}

		return $itemData;
	}
}
