<?php

App::uses('CakeEventListener', 'Event');
App::uses('BulkDiscountLib', 'EvBulkDiscount.Lib');

class BulkDiscountBasketListener implements CakeEventListener {

	public function implementedEvents() {
		return array(
			'EvBasket.Component.Basket.itemAdd' => array(
				'callable' => 'bulkDiscount',
				'priority' => 1
			),
			'EvBasket.Component.Basket.itemUpdated' => array(
				'callable' => 'bulkDiscount',
				'priority' => 1
			),
			'EvBasket.Component.Basket.itemDeleted' => array(
				'callable' => 'bulkDiscount',
				'priority' => 1
			)
		);
	}

	/**
	 * rebuild the bulk discount in the basket
	 * $Event->subject() = BasketManager Component
	 */
	public function bulkDiscount(CakeEvent $Event) {
		if (! CakePlugin::loaded('EvBasket')) {
			return false;
		}

		$basket = $Event->subject()->getBasket(true);

		if (empty($basket['BasketItem'])) {
			return;
		}

		// get the product ids from the basket
		$productIds = Hash::extract($basket, 'BasketItem.{s}.Variant.product_id');

		// get any bulk discountg data
		$BulkDiscount = EvClassRegistry::init('EvBulkDiscount.BulkDiscount');
		$discounts = $BulkDiscount->find(
			'all',
			array(
				'conditions' => array(
					'BulkDiscount.model' => 'EvShop.Product',
					'BulkDiscount.model_id' => $productIds
				),
				'order' => 'BulkDiscount.model_id ASC, BulkDiscount.quantity ASC'
			)
		);

		$discounts = Hash::combine(
			$discounts,
			'{n}.BulkDiscount.quantity',
			'{n}',
			'{n}.BulkDiscount.model_id'
		);

		if (! empty($basket['BasketItem'])) {
			$basketQuantities = Hash::combine(
				$basket['BasketItem'],
				array(
					'%s.%s',
					'{s}.model',
					'{s}.model_id'
				),
				'{s}.quantity',
				'{s}.Variant.product_id'
			);

			$originalUnitPrices = $Event->subject()->_controller->BasketCookie->read('originalUnitPrices');

			$toUpdate = array();

			foreach ($basketQuantities as $productId => $variants) {

				if (empty($discounts[$productId])) {
					continue;
				}

				$quantities = array_keys($discounts[$productId]);
				sort($quantities);

				$totalQuantity = array_sum($variants);

				$matched = 0;
				foreach ($quantities as $quantity) {
					if ($totalQuantity >= $quantity) {
						$matched = $quantity;
					}
				}

				if (! empty($discounts[$productId][$matched])) {
					$discount = $discounts[$productId][$matched];

					foreach ($variants as $basketItemId => $quantity) {
						$canDiscount = true;

						$itemPrice = $basket['BasketItem'][$basketItemId]['unit_price'];
						$itemUnitTax = $basket['BasketItem'][$basketItemId]['unit_tax'];

						if (!empty($originalUnitPrices) && isset($originalUnitPrices[$basketItemId])) {
							$originalUnitPrice = $originalUnitPrices[$basketItemId];
							if ($basket['BasketItem'][$basketItemId]['unit_price'] != $originalUnitPrice['unit_price']) {
								//Need to reset the current item to it's original unit price before we can discount it
								$itemPrice = $originalUnitPrice['unit_price'];
								$itemUnitTax = $originalUnitPrice['unit_tax'];

								$newUpdate[] = array(
									'model' => $basket['BasketItem'][$basketItemId]['model'],
									'model_id' => $basket['BasketItem'][$basketItemId]['model_id'],
									'quantity' => $quantity,
									'unitPrice' => $originalUnitPrice['unit_price'],
									'unitTax' => $originalUnitPrice['unit_tax'],
									'taxRate' => $basket['BasketItem'][$basketItemId]['tax_rate']
								);
								$Event->subject()->updateItem($newUpdate, null, null, 0, false);
							}
						} else {
							//Add the current unit price to originalUnitPrices cookie
							$originalUnitPrices[$basketItemId] = [
								'unit_price' => $basket['BasketItem'][$basketItemId]['unit_price'],
								'unit_tax' => $basket['BasketItem'][$basketItemId]['unit_tax'],
								'tax_rate' => $basket['BasketItem'][$basketItemId]['tax_rate'],
								'unit_price_incTax' => $basket['BasketItem'][$basketItemId]['unit_price_incTax'],
								'row_total' => $basket['BasketItem'][$basketItemId]['row_total'],
								'row_total_incTax' => $basket['BasketItem'][$basketItemId]['row_total_incTax'],
							];
						}

						$unitPrice = BulkDiscountLib::calculateDiscount(
							$itemPrice,
							$discount['BulkDiscount']['amount']
						);

						$unitTax = $Event->subject()->_controller->BasketManager->BasketItem->calculateUnitTax($unitPrice, $basket['BasketItem'][$basketItemId]['tax_rate']);

						$unitPrice = round($unitPrice, 2);

						$toUpdate[] = array(
							'model' => $basket['BasketItem'][$basketItemId]['model'],
							'model_id' => $basket['BasketItem'][$basketItemId]['model_id'],
							'quantity' => $quantity,
							'unitPrice' => $unitPrice,
							'unitTax' => $unitTax,
							'taxRate' => $basket['BasketItem'][$basketItemId]['tax_rate']
						);
					}
				} else {
					foreach ($variants as $basketItemId => $quantity) {
						$price = $basket['BasketItem'][$basketItemId]['unit_price'];
						$unitTax = $basket['BasketItem'][$basketItemId]['unit_tax'];

						if (!empty($originalUnitPrices) && isset($originalUnitPrices[$basketItemId])) {
							$originalUnitPrice = $originalUnitPrices[$basketItemId];
							if ($price != $originalUnitPrice['unit_price']) {
								$price = $originalUnitPrice['unit_price'];
								$unitTax = $originalUnitPrice['unit_tax'];
							}
						}
						$toUpdate[] = array(
							'model' => $basket['BasketItem'][$basketItemId]['model'],
							'model_id' => $basket['BasketItem'][$basketItemId]['model_id'],
							'quantity' => $quantity,
							'unitPrice' => $price,
							'unitTax' => ((!empty($unitTax)) ? $unitTax : null),
							'taxRate' => $basket['BasketItem'][$basketItemId]['tax_rate']
						);
					}
				}
			}

			if (! empty($toUpdate)) {
				$Event->subject()->updateItem($toUpdate, null, null, 0, false);
			}

			$Event->subject()->_controller->set('originalUnitPrices', $originalUnitPrices);
			$Event->subject()->_controller->BasketCookie->write(
				'originalUnitPrices',
				$originalUnitPrices
			);

			if (!empty($originalUnitPrices)) {
				$Event->subject()->_controller->BasketManager->setBasketData('original_price', json_encode($originalUnitPrices), false);
			}
		}
	}
}
