<?php
App::uses('EvBlogAppModel', 'EvBlog.Model');

class BlogPost extends EvBlogAppModel {

	public $actsAs = array(
		'EvTemplates.Template' => array(
			'formInject' => true
		),
		'Routable.Routable' => array(
			'alias' => 'blog/:displayField',
			'config' => 'EvBlog'
		),
		'MetaData.Meta',
	);

	public $displayField = 'name';

	public $imageSlots = array(
		'listing' => array(
			'slots' => 1,
			'fields' => array(
				'alt'
			)
		),
		'main' => array(
			'slots' => -1,
			'fields' => array(
				'alt'
			)
		),
	);

	public $hasAndBelongsToMany = array(
		'BlogCategory' => array(
			'className' => 'EvBlog.BlogCategory',
			'joinTable' => 'ev_blog_blog_categories_blog_posts'
		)
	);

	public $hasMany = array(
		'BlogTag' => array(
			'className' => 'EvBlog.BlogTag',
			'cascade' => true,
			'dependent' => true
		),
		'BlogComment' => array(
			'className' => 'EvBlog.BlogComment',
			'cascade' => true,
			'dependent' => true
		)
	);

	public $order = array(
		'BlogPost.posted' => 'DESC'
	);

	public $validate = array(
		'name' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'Please add a name'
			),
		),
		'posted' => array(
			'datetime' => array(
				'rule' => array('datetime'),
				'message' => 'Please add a posted date and time'
			),
		),
		'body' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'Please add body text'
			),
		),
	);

/**
 * Constructor
 *
 * @param bool|int|string|array $id Set this ID for this model on startup,
 * can also be an array of options, see above.
 * @param string $table Name of database table to use.
 * @param string $ds DataSource connection name.
 */
	public function __construct($id = false, $table = null, $ds = null) {
		if (
			Configure::read('EvBlog.enable_inline_images') === true
			&& CakePlugin::loaded('EvTinyMceImage') === true
		) {
			$this->imageSlots['inline_content'] = [
				'slots' => -1,
			];
		}
		parent::__construct($id, $table, $ds);
	}

/**
 * Invalidate a blog post if it doesn't have at least one blog category.
 *
 * {@inheritDoc}
 */
	public function beforeValidate($options = array()) {
		// Force a category to be selected.
		if (isset($this->data['BlogCategory']['BlogCategory']) && empty($this->data['BlogCategory']['BlogCategory'])) {
			$this->BlogCategory->invalidate('BlogCategory', 'Please select at least one category to publish this post under.');
		}
		return parent::beforeValidate($options);
	}

/**
 * Executes query based on supplied params and returns the found data
 *
 * @param int $id The unique id of the requested item
 * @param array $query Contains additional parameters to insert into the find call
 * @return array Containins the resulting data
 */
	public function readForView($id, $query = array()) {
		$query['contain'][] = 'BlogTag';

		$query['contain']['BlogComment'] = array(
			'conditions' => array(
				'BlogComment.is_active' => 1
			),
			'order' => 'BlogComment.created ASC',
		);

		if (empty($this->disableCategories)) {
			$query['contain'][] = 'BlogCategory';
		}

		return parent::readForView($id, $query);
	}

/**
 * Contain blog tags and blog comments (ordered by their created date).
 * Blog categories are contained if they haven't been disabled in the config.
 *
 * {@inheritDoc}
 */
	public function readForEdit($id, $query = array()) {
		$query['contain'][] = 'BlogTag';
		$query['contain']['BlogComment'] = array(
			'order' => 'BlogComment.created ASC'
		);
		if (empty($this->disableCategories)) {
			$query['contain'][] = 'BlogCategory';
		}

		return parent::readForEdit($id, $query);
	}

/**
 * Set the blog category to null if no blog category id has been provided.
 * Format the posted date to be the server time so all blog posts posted dates
 * are in sync.
 *
 * {@inheritDoc}
 */
	public function beforeSave($options = array()) {
		if (isset($this->data[$this->alias]['blog_category_id']) && $this->data[$this->alias]['blog_category_id'] == 0) {
			$this->data[$this->alias]['blog_category_id'] = null;
		}

		if (!empty($this->data[$this->alias]['posted']) && !empty($this->data[$this->alias]['posted_gmt'])) {
			// Convert date to GMT.
			$this->data[$this->alias]['posted'] = CakeTime::toServer($this->data[$this->alias]['posted'], Configure::read('Config.timezone'));
		}

		return parent::beforeSave($options);
	}

/**
 * Generate a snippet for each found result, assign it an empty blog category id
 * if it is missing one and process the tags so that they are in a comma separated
 * string.
 *
 * {@inheritDoc}
 */
	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);
		$length = Configure::read('EvBlog.snippet_length');

		foreach ($results as $k => $blog) {
			if (isset($blog[$this->alias])) {

				if (isset($blog[$this->alias]['body']) && !empty($blog[$this->alias]['body'])) {
					$snippet = strip_tags($blog[$this->alias]['body']);

					if (strlen($snippet) > $length) {
						$snippet = substr($snippet, 0, $length) . ' &hellip;';
					}

					$results[$k][$this->alias]['snippet'] = $snippet;
				}
			}

			if (! isset($blog[$this->alias]['blog_category_id']) || empty($blog[$this->alias]['blog_category_id'])) {
				$results[$k][$this->alias]['blog_category_id'] = 0;
			}

			// implode all the tags into one string
			if (isset($blog['BlogTag']) && ! empty($blog['BlogTag'])) {
				// sort the tags
				$tags = array();
				foreach ($blog['BlogTag'] as $tag) {
					$tags[] = !empty($tag['tag']) ? $tag['tag'] : $tag;
				}

				$results[$k][$this->alias]['tags'] = implode(', ', $tags);

			} else {
				$results[$k][$this->alias]['tags'] = '';
			}
		}

		return $results;
	}

/**
 * Process blog tags written as a comma separated string into separate tags.
 *
 * {@inheritDoc}
 */
	public function afterSave($created, $options = array()) {
		$this->BlogTag = EvClassRegistry::init('EvBlog.BlogTag');

		if ($created === false) {
			$this->BlogTag->deleteAll(array(
				'BlogTag.blog_post_id' => $this->id
			));
		}

		if (isset($this->data[$this->alias]['tags']) && ! empty($this->data[$this->alias]['tags'])) {
			$tags = explode(',', $this->data[$this->alias]['tags']);
			$blogTags = array();

			foreach ($tags as $tag) {

				$blogTags[] = array(
					'BlogTag' => array(
						'tag' => trim($tag),
						'blog_post_id' => $this->id
					)
				);
			}

			unset($this->data[$this->alias]['tags']);
			$this->BlogTag->saveMany($blogTags);
		}

		return parent::afterSave($created, $options);
	}

}
