<?php
App::uses('EvBlogAppController', 'EvBlog.Controller');
App::uses('CakeTime', 'Utility');
App::uses('CakeText', 'Utility');

class BlogPostsController extends EvBlogAppController {

	public $components = array('RequestHandler');

/**
 * Constructor.
 *
 * @param CakeRequest $request Request object for this controller. Can be null for testing,
 *  but expect that features that use the request parameters will not work.
 * @param CakeResponse $response Response object for this controller.
 */
	public function __construct($request = null, $response = null) {
		parent::__construct($request, $response);

		/*
		 * Add the config for the EvTinyMceImage component now. The component won't be loaded until later but config
		 * settings can be modified before the component is loaded.
		 */
		if (
			Configure::read('EvBlog.enable_inline_images') === true
			&& CakePlugin::loaded('EvTinyMceImage') === true
		) {
			$this->components['EvTinyMceImage.TinyMCE'] = [];
		}
	}

/**
 * Called before the controller action. You can use this method to configure and customize components
 * or perform logic that needs to happen before each controller action.
 *
 * @return void
 * @link https://book.cakephp.org/2.0/en/controllers.html#request-life-cycle-callbacks
 */
	public function beforeFilter() {
		if (isset($this->components['EvTinyMceImage.TinyMCE'])) {
			//Set the EvTinyMceImage component to have the correct model
			$this->components['EvTinyMceImage.TinyMCE']['model_alias'] = $this->{$this->modelClass}->alias;

			//Load the EvTinyMCEImage component with the correct config.
			$this->TinyMCE = $this->loadComponent(
				'EvTinyMceImage.TinyMCE',
				$this->components['EvTinyMceImage.TinyMCE']
			);
		}

		parent::beforeFilter();

		$this->Auth->allow(array(
			'index',
			'view',
			'archives'
		));
	}

	protected function _adminFilterFields() {
		$filterFields = array();

		$Model = $this->{$this->modelClass};

		$filterFields[$Model->alias . '.name'] = array(
			'label' => 'Name',
			'type' => 'string',
			'compare' => array(
				$Model->alias . '.name LIKE' => '%%%s%%'
			)
		);

		// include an author filter field when authors are enabled in the config
		if (! isset($this->disablePostAuthor) || $this->disablePostAuthor === false) {
			$filterFields[$Model->alias . '.author'] = array(
				'label' => 'Author',
				'type' => 'string',
				'compare' => array(
					$Model->alias . '.author LIKE' => '%%%s%%'
				)
			);
		}

		$filterFields[$Model->alias . '.posted'] = array(
			'label' => 'Posted',
			'type' => 'date',
			'compare' => array($Model->alias . ".posted" => "%s")
		);

		return $filterFields;
	}

/**
 * override the admin index columns
 */
	protected function _adminIndexColumns() {
		$columns = parent::_adminIndexColumns();

		$Model = $this->{$this->modelClass};

		// remove the author column when requested to from the config
		if (
			(isset($this->disablePostAuthor) && $this->disablePostAuthor === true) &&
			! empty($columns[$Model->alias . '.author'])
		) {
			unset($columns[$Model->alias . '.author']);
		}

		// free up some space in the index by removing the created column
		unset($columns[$Model->alias . '.created']);

		$additionalColumns = array();

		// add concatinated blog category column
		if (empty($this->disableCategories)) {
			$additionalColumns[$Model->alias . '.blog_category_name_concat'] = array(
				'type' => 'string_full',
				'null' => true,
				'default' => null,
				'length' => (int)100,
				'collate' => 'utf8_general_ci',
				'charset' => 'utf8',
				'label' => 'Categories'
			);
		}

		// add posted column
		$additionalColumns[$Model->alias . '.posted'] = array(
			'type' => 'datetime',
			'null' => true,
			'default' => null,
			'length' => null,
			'label' => 'Posted On'
		);

		$addAdditionalColumnsAfter = isset($columns[$Model->alias . '.author']) ?
			$Model->alias . '.author':
			$Model->alias . '.name';

		return ArrayUtil::addAfter($columns, $addAdditionalColumnsAfter, $additionalColumns);
	}

/**
 * Admin index paginate
 *
 * @return array
 */
	protected function _adminIndexPaginate() {
		$paginate = parent::_adminIndexPaginate();
		$defaults = [];
		if (empty($this->disableCategories)) {
			$defaults = array(
				'contain' => array(
					'BlogCategory'
				)
			);
		}

		return Hash::merge($defaults, $paginate);
	}

	protected function _adminFormFields() {
		$fields = parent::_adminFormFields();

		$Model = $this->{$this->modelClass};

		$additionalFields = array();

		// move the author field when disabled through the config
		if (isset($this->disablePostAuthor) && $this->disablePostAuthor === true) {
			unset($fields[$Model->alias . '.author']);
		}

		//	Don't want to show the hide from search engine flag if the app has disabled this functionality
		if (empty(Configure::read('EvBlog.enable_hide_from_search_engines'))) {
			unset($fields[$Model->alias . '.hide_from_search_engines']);
		}

		// add blog category multicheck field
		if (empty($this->disableCategories)) {
			$additionalFields['BlogCategory'] = array(
				'type' => 'multicheck',
				'label' => 'Category'
			);
		}

		// we only want to show the tags elements when tags are enable in the config
		if (! isset($this->disableTags) || $this->disableTags === false) {
			$additionalFields[$Model->alias . '.tags'] = array(
				'type' => 'string',
				'after' => '&nbsp;&nbsp;Seperate multiple tags with a comma.'
			);
		}

		return ArrayUtil::addAfter($fields, $Model->alias . '.body', $additionalFields);
	}

	protected function _adminPopulateLookups() {
		$this->loadModel('EvBlog.BlogCategory');
		$blogCategories = $this->BlogCategory->find('list');
		$this->set('blogCategories', $blogCategories);
	}

/**
 * Manages a default admin index view of this model
 * Includes a filter form and paginated results
 *
 * Override in your controller to customise
 *
 * @return void
 */
	public function admin_index() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		parent::admin_index();

		// re-fetch paginate data so we can handle the displaying of concatinated blog categories
		$data = $this->Paginator->paginate();

		// concatinate blog category names for each item to show on th admin index
		if (! empty($data)) {
			foreach ($data as $key => $item) {
				if (! empty($item['BlogCategory'])) {
					$blogCategories = Hash::combine($item, 'BlogCategory.{n}.id', 'BlogCategory.{n}.name');
					if (version_compare(PHP_VERSION, '8.0.0', '<')) {
						$data[$key][$modelAlias]['blog_category_name_concat'] = implode($blogCategories, ', ');
					} else {
						$data[$key][$modelAlias]['blog_category_name_concat'] = implode(', ', $blogCategories);
					}
				}
			}
		}

		$this->set(compact('data'));

		return;
	}

/**
 * Manages an admin edit form for this model
 *
 * @param integer $id ID of row to edit
 * @return void
 */
	public function admin_edit($id = null) {
		parent::admin_edit($id);
		$this->view = 'EvBlog.BlogPosts/admin_edit';
	}

/**
 * blog index
 *
 * display blog listing page
 */
	public function index() {
		$Model = $this->{$this->modelClass};

		$this->view = 'EvBlog.BlogPosts/index';

		// Fall back incase no page is found
		$this->set('title_for_layout', "Blog");

		$this->loadModel('EvCore.Page');

		// see if there is a linked content page
		$pageId = Configure::read('EvBlog.page_id');

		if (! empty($pageId)) {
			$pageData = $this->assignPage($pageId, 'page');

			if (!empty(Configure::read('EvBlog.use_page_template')) && !empty($pageData['Page']['template_id'])) {
				$this->view = $this->Tpl->getTemplate($pageData, 'Page');
			}
		}

		if ($this->RequestHandler->isRss()) {
			$posts = $Model->find('all', array(
				'conditions' => array(
					'BlogPost.is_active' => 1,
					'BlogPost.posted <' => date('Y-m-d H:i:s')
				),
				'order' => 'BlogPost.posted DESC',
				'limit' => Configure::read('EvBlog.rss_limit')
			));

			$description = ! empty($page['Page']['body']) ? strip_tags($page['Page']['body']) : null;

			$descLen = strlen($description);

			if ($descLen > 0 && $descLen >= 250) {
				$description = CakeText::truncate($description, 250);
			} elseif ($descLen == 0) {
				$description = ! empty($page['Page']['title']) ? strip_tags($page['Page']['title']) : null;
			}

			$channel = array();

			if (! empty($page)) {
				$channel = array(
					'title' => $page['Page']['title'],
					'link' => array(
						'plugin' => 'ev_blog',
						'controller' => 'blog_posts',
						'action' => 'index'
					),
					'description' => $description,
					'language' => 'en-us'
				);
			}

			return $this->set(compact('channel', 'posts'));
		}

		$this->paginate = $this->_getPaginateQuery();

		$this->set('data', $this->paginate());
		$this->setWidgets();
	}

/**
 * The query used to fetch blog posts for the listing
 *
 * @return array Pagination query
 */
	protected function _getPaginateQuery() {
		return [
			'conditions' => [
				'BlogPost.is_active' => 1,
				'BlogPost.posted <' => date('Y-m-d H:i:s')
			],
			'order' => 'BlogPost.posted DESC',
			'contain' => [
				'ListingImage',
				'BlogComment' => [
					'conditions' => [
						'BlogComment.is_active' => 1
					],
					'order' => 'BlogComment.created ASC'
				],
				'BlogCategory'
			],
			'limit' => Configure::read('EvBlog.listing_limit')
		];
	}

/**
 * View individual blog post
 *
 * @param int id Id of blog post to view
 *
 * @throws NotFoundException If the blog post could not be found
 *
 * @return void
 */
	public function view($id) {
		$Model = $this->{$this->modelClass};
		$alias = $Model->alias;

		$data = $Model->readForViewOrFail($id);
		$this->Meta->set($data);

		$this->set('data', $data);
		$this->setWidgets();

		// Format the list of categories this post is assigned to. We just need to group them up
		// by the category name and url.
		$categories = array();

		if (! empty($data['BlogCategory'])) {
			foreach ($data['BlogCategory'] as $category) {

				$categories[$category['name']] = array(
					'plugin' => 'ev_blog',
					'controller' => 'blog_categories',
					'method' => 'view',
					$category['id']
				);
			}
		}

		$this->set('categories', $categories);

		//Preparing plugin for templates.
		$this->view = 'EvBlog.BlogPosts/view';
		if ($Model->hasBehavior('EvTemplates.Template')) {
			$template = $this->Tpl->getTemplate($data, $this->{$this->modelClass}->alias);
			if ($template) {
				$this->view = $template;
			}
		}
	}

/**
 * Blog archive, listing for blogfs from a certian month
 *
 * @param string $date given month to get posts from
 */
	public function archives($date = false) {
		// Fall back incase no page is found
		$this->set('title_for_layout', "Blog");

		if (! $date) {
			$date = CakeTime::format(time(), '%y-%m-%d');
		}

		$dateBits = explode('-', $date);

		$dateShort = $dateBits['0'] . '-' . $dateBits['1'];

		$category = array(
			'BlogArchive' => array(
				'name' => 'Blog Archive: ' . CakeTime::format($date, '%d/%m/%y')
			)
		);

		$this->Meta->set($category, 'BlogArchive');
		$this->set('category', $category);

		$this->paginate = array(
			'conditions' => array(
				'BlogPost.is_active' => 1,
				'BlogPost.posted >=' => $dateShort . '-01 00:00:00',
				'BlogPost.posted <=' => $date . ' 23:59:59',
				'BlogPost.posted <' => date('Y-m-d H:i:s')
			),
			'order' => 'BlogPost.posted DESC',
			'contain' => array(
				'ListingImage',
				'BlogComment' => array(
					'conditions' => array(
						'BlogComment.is_active' => 1
					)
				),
				'BlogCategory'
			),
			'limit' => Configure::read('EvBlog.listing_limit'),
		);

		$this->loadModel('EvBlog.BlogPost');
		$this->set('data', $this->paginate('BlogPost'));
		$this->setWidgets();

		$this->view = 'EvBlog.BlogPosts/archives';
	}

/**
 * Set widgets
 *
 * @return void
 */
	protected function setWidgets() {
		if (Configure::read('EvBlog.disable_blog_widgets') === true) {
			return;
		}

		$this->widgetLatestPosts();
		$this->widgetCategories();
		$this->widgetArchives();
	}
}
