<?php
App::uses('EvBlogAppController', 'EvBlog.Controller');
App::uses('CakeTime', 'Utility');

class BlogPostsController extends EvBlogAppController {

	public $components = array('RequestHandler');

	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(array(
			'index',
			'view',
			'archives'
		));
	}

	protected function _adminFilterFields() {
		return array(
			'BlogPost.name' => array(
				'label' => 'Name',
				'type' => 'string',
				'compare' => array(
					'BlogPost.name LIKE' => '%%%s%%'
				)
			),
			'BlogPost.author' => array(
				'label' => 'Author',
				'type' => 'string',
				'compare' => array(
					'BlogPost.author LIKE' => '%%%s%%'
				)
			),
			'BlogPost.posted' => array(
				'label' => 'Posted',
				'type' => 'date',
				'compare' => array("BlogPost.posted" => "%s")
			)
		);
	}

/**
 * override the admin index columns
 */
	protected function _adminIndexColumns() {
		$columns = array(
			'BlogPost.name' => array(
				'type' => 'string_full',
				'null' => true,
				'default' => null,
				'length' => (int)100,
				'collate' => 'utf8_general_ci',
				'charset' => 'utf8',
				'label' => 'Name'
			),
			'BlogPost.author' => array(
				'type' => 'string',
				'null' => true,
				'default' => null,
				'length' => (int)100,
				'collate' => 'utf8_general_ci',
				'charset' => 'utf8',
				'label' => 'Author'
			),
			'BlogPost.blog_category_name_concat' => array(
				'type' => 'string_full',
				'null' => true,
				'default' => null,
				'length' => (int)100,
				'collate' => 'utf8_general_ci',
				'charset' => 'utf8',
				'label' => 'Categories'
			),
			'BlogPost.posted' => array(
				'type' => 'datetime',
				'null' => true,
				'default' => null,
				'length' => null,
				'label' => 'Posted On'
			),
			'BlogPost.is_active' => array(
				'type' => 'boolean',
				'null' => true,
				'default' => null,
				'length' => (int)1,
				'label' => 'Active'
			)
		);

		return $columns;
	}

	protected function _adminIndexPaginate() {
		$conditions = $this->_processFilter();

		$paginate = array(
			'conditions' => $conditions,
			'contain' => array(
				'BlogCategory'
			)
		);

		return $paginate;
	}

	protected function _adminFormFields() {
		$fields = parent::_adminFormFields();
		$fields['BlogCategory'] = array(
			'type' => 'multicheck',
			'label' => 'Category'
		);
		$fields['BlogPost.tags'] = array(
			'type' => 'string',
			'after' => '&nbsp;&nbsp;Seperate multiple tags with a comma.'
		);
		return $fields;
	}

	protected function _adminPopulateLookups() {
		$this->loadModel('EvBlog.BlogCategory');
		$blogCategories = $this->BlogCategory->find('list');
		$this->set('blogCategories', $blogCategories);
	}

/**
 * Manages a default admin index view of this model
 * Includes a filter form and paginated results
 *
 * Override in your controller to customise
 *
 * @return void
 */
	public function admin_index() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		parent::admin_index();

		// re-fetch paginate data so we can handle the displaying of concatinated blog categories
		$data = $this->Paginator->paginate();

		// concatinate blog category names for each item to show on th admin index
		if (! empty($data)) {
			foreach ($data as $key => $item) {
				if (! empty($item['BlogCategory'])) {
					$blogCategories = Hash::combine($item, 'BlogCategory.{n}.id', 'BlogCategory.{n}.name');
					$data[$key][$modelAlias]['blog_category_name_concat'] = implode($blogCategories, ', ');
				}
			}
		}

		$this->set(compact('data'));

		return;
	}

/**
 * Manages an admin edit form for this model
 *
 * @param integer $id ID of row to edit
 * @return void
 */
	public function admin_edit($id = null) {
		parent::admin_edit($id);
		$this->view = 'EvBlog.BlogPosts/admin_edit';
	}

/**
 * blog index
 *
 * display blog listing page
 */
	public function index() {
		$Model = $this->{$this->modelClass};

		// Fall back incase no page is found
		$this->set('title_for_layout', "Blog");

		$this->loadModel('Page');

		// see if there is a linked content page
		$pageId = Configure::read('EvBlog.page_id');

		if (! empty($pageId)) {
			$page = $this->Page->readForView($pageId);

			if (! empty($page)) {
				$this->Meta->set($page, 'Page');
				$this->set('page', $page);
			}
		}

		if ($this->RequestHandler->isRss()) {
			$posts = $Model->find('all', array(
				'conditions' => array(
					'BlogPost.is_active' => 1,
					'BlogPost.posted <' => date('Y-m-d H:i:s')
				),
				'order' => 'BlogPost.posted DESC',
				'limit' => Configure::read('EvBlog.rss_limit')
			));

			$description = ! empty($page['Page']['body']) ? strip_tags($page['Page']['body']) : null;

			$descLen = strlen($description);

			if ($descLen > 0 && $descLen >= 250) {
				$description = String::truncate($description, 250);
			} elseif ($descLen == 0) {
				$description = ! empty($page['Page']['title']) ? strip_tags($page['Page']['title']) : null;
			}

			$channel = array();

			if (! empty($page)) {
				$channel = array(
					'title' => $page['Page']['title'],
					'link' => array(
						'plugin' => 'ev_blog',
						'controller' => 'blog_posts',
						'action' => 'index'
					),
					'description' => $description,
					'language' => 'en-us'
				);
			}

			return $this->set(compact('channel', 'posts'));
		}

		$this->paginate = array(
			'conditions' => array(
				'BlogPost.is_active' => 1,
				'BlogPost.posted <' => date('Y-m-d H:i:s')
			),
			'order' => 'BlogPost.posted DESC',
			'contain' => array(
				'ListingImage',
				'BlogComment' => array(
					'conditions' => array(
						'BlogComment.is_active' => 1
					),
					'order' => 'BlogComment.created ASC'
				),
				'BlogCategory'
			),
			'limit' => Configure::read('EvBlog.listing_limit')
		);

		$this->set('data', $this->paginate());

		// call the widgets
		$this->widgetLatestPosts();
		$this->widgetCategories();
		$this->widgetArchives();

		$this->view = 'EvBlog.BlogPosts/index';
	}

/**
 * view individual blog post
 */
	public function view($id) {
		$Model = $this->{$this->modelClass};
		$alias = $Model->alias;

		$data = $Model->readForView($id);
		$this->Meta->set($data);

		$this->set('data', $data);

		// call the widgets
		$this->widgetLatestPosts();
		$this->widgetCategories();
		$this->widgetArchives();

		// Format the list of categories this post is assigned to. We just need to group them up
		// by the category name and url.
		$categories = array();

		foreach ($data[$alias] as $category) {

			$categories[$category['name']] = array(
				'plugin' => 'ev_blog',
				'controller' => 'blog_categories',
				'method' => 'view',
				$category['id']
			);
		}

		$this->set('categories', $categories);

		//Preparing plugin for templates.
		if ($Model->hasBehavior('EvTemplates.Template')) {
			$this->view = $this->Tpl->getTemplate($data);
		}
		if (!$this->view) {
			$this->view = 'EvTeam.BlogPosts/view';
		}
	}

/**
 * Blog archive, listing for blogfs from a certian month
 *
 * @param string $date given month to get posts from
 */
	public function archives($date = false) {
		// Fall back incase no page is found
		$this->set('title_for_layout', "Blog");

		if (! $date) {
			$date = CakeTime::format(time(), '%y-%m-%d');
		}

		$dateBits = explode('-', $date);

		$dateShort = $dateBits['0'] . '-' . $dateBits['1'];

		$category = array(
			'BlogArchive' => array(
				'name' => 'Blog Archive: ' . CakeTime::format($date, '%d/%m/%y')
			)
		);

		$this->Meta->set($category, 'BlogArchive');
		$this->set('category', $category);

		$this->paginate = array(
			'conditions' => array(
				'BlogPost.is_active' => 1,
				'BlogPost.posted >=' => $dateShort . '-01 00:00:00',
				'BlogPost.posted <=' => $date . ' 23:59:59',
				'BlogPost.posted <' => date('Y-m-d H:i:s')
			),
			'order' => 'BlogPost.posted DESC',
			'contain' => array(
				'ListingImage',
				'BlogComment' => array(
					'conditions' => array(
						'BlogComment.is_active' => 1
					)
				),
				'BlogCategory'
			),
			'limit' => 10
		);

		$this->loadModel('EvBlog.BlogPost');
		$this->set('data', $this->paginate('BlogPost'));

		// call the widgets
		$this->widgetLatestPosts();
		$this->widgetCategories();
		$this->widgetArchives();

		$this->view = 'EvBlog.BlogPosts/archives';
	}

}
