<?php
App::uses('EvBlogAppController', 'EvBlog.Controller');
App::uses('CakeTime', 'Utility');

class BlogPostsController extends EvBlogAppController {

	public $components = array('RequestHandler');

	public function beforeFilter() {

		parent::beforeFilter();

		$this->Auth->allow(array(
			'index',
			'view',
			'archives'
		));

	}

	protected function _adminFilterFields() {

		return array(
			'BlogPost.name' => array(
				'label' => 'Name',
				'type' => 'string',
				'compare' => array(
					'BlogPost.name LIKE' => '%%%s%%'
				)
			),
			'BlogPost.author' => array(
				'label' => 'Author',
				'type' => 'string',
				'compare' => array(
					'BlogPost.author LIKE' => '%%%s%%'
				)
			),
			'BlogPost.posted' => array(
				'label' => 'Posted',
				'type' => 'date',
				'compare' => array("BlogPost.posted" => "%s")
			)
		);
	}

/**
 * override the admin index columns
 */
	protected function _adminIndexColumns()	{
		$columns = array(
			'BlogPost.name' => array(
				'type' => 'string_full',
				'null' => true,
				'default' => null,
				'length' => (int) 100,
				'collate' => 'utf8_general_ci',
				'charset' => 'utf8',
				'label' => 'Name'
			),
			'BlogPost.author' => array(
				'type' => 'string',
				'null' => true,
				'default' => null,
				'length' => (int) 100,
				'collate' => 'utf8_general_ci',
				'charset' => 'utf8',
				'label' => 'Author'
			),
			'BlogCategory.name' => array(
				'type' => 'string_full',
				'null' => true,
				'default' => null,
				'length' => (int) 100,
				'collate' => 'utf8_general_ci',
				'charset' => 'utf8',
				'label' => 'Category'
			),
			'BlogPost.posted' => array(
				'type' => 'datetime',
				'null' => true,
				'default' => null,
				'length' => null,
				'label' => 'Posted On'
			),
			'BlogPost.is_active' => array(
				'type' => 'boolean',
				'null' => true,
				'default' => null,
				'length' => (int) 1,
				'label' => 'Active'
			)
		);

		return $columns;
	}

	protected function _adminIndexPaginate() {

		$conditions = $this->_processFilter();

		$paginate = array(
			'conditions'=>$conditions,
			'contain' => array(
				'BlogCategory'
			)
		);

		return $paginate;
	}


	protected function _adminFormFields() {

		return array(
			'BlogPost.id' => array(
				'type' => 'integer',
				'null' => false,
				'default' => null,
				'length' => (int) 11,
				'key' => 'primary'
			),
			'BlogPost.created' => array(
				'type' => 'datetime',
				'null' => true,
				'default' => null,
				'length' => null
			),
			'BlogPost.modified' => array(
				'type' => 'datetime',
				'null' => true,
				'default' => null,
				'length' => null
			),
			'BlogPost.name' => array(
				'type' => 'string',
				'null' => true,
				'default' => null,
				'length' => (int) 100,
				'collate' => 'utf8_general_ci',
				'charset' => 'utf8'
			),
			'BlogPost.author' => array(
				'type' => 'string',
				'null' => true,
				'default' => null,
				'length' => (int) 100,
				'collate' => 'utf8_general_ci',
				'charset' => 'utf8'
			),
			'BlogPost.posted' => array(
				'type' => 'datetime',
				'null' => true,
				'default' => null,
				'length' => null
			),
			'BlogPost.body' => array(
				'type' => 'html',
				'null' => true,
				'default' => null,
				'length' => null,
				'collate' => 'utf8_general_ci',
				'charset' => 'utf8'
			),
			'BlogPost.blog_category_id' => array(
				'type' => 'select',
				'label' => 'Category'
			),
			'BlogPost.tags' => array(
				'type' => 'string',
				'after' => '&nbsp;&nbsp;Seperate multiple tags with a comma.'
			),
			'BlogPost.is_active' => array(
				'type' => 'boolean',
				'null' => false,
				'default' => '0',
				'length' => (int) 1
			),
			'Route.id' => array(
				'type' => 'hidden'
			),
			'Route.alias' => array(
				'type' => 'hidden',
				'length' => (int) 128
			)
		);
	}

	public function _adminPopulateLookups()	{

		$this->loadModel('EvBlog.BlogCategory');
		$blogCategories = $this->BlogCategory->find('list');

		$this->set('blogCategories', $blogCategories);
	}

	public function admin_edit($id = null) {
		parent::admin_edit($id);
		$this->view = 'EvBlog.BlogPosts/admin_edit';
	}


/**
 * blog index
 *
 * display blog listing page
 */
	public function index() {

		$Model = $this->{$this->modelClass};

		// Fall back incase no page is found
		$this->set('title_for_layout', "Blog");

		$this->loadModel('Page');

		// see if there is a linked content page
		$page_id = Configure::read('EvBlog.page_id');

		if (! empty($page_id)) {
			$page = $this->Page->readForView($page_id);

			if (! empty($page)) {

				$this->Meta->set($page, 'Page');

				$this->set('page', $page);
			}
		}

		if ($this->RequestHandler->isRss() ) {

			$posts = $Model->find('all', array(
				'conditions' => array(
					'BlogPost.is_active' => 1,
					'BlogPost.posted <' => date('Y-m-d H:i:s')
				),
				'order' => 'BlogPost.posted DESC',
				'limit' => Configure::read('EvBlog.rss_limit')
			));

			$description = !empty($page['Page']['body']) ? strip_tags($page['Page']['body']) : null;
			$desc_len = strlen($description);
			if ($desc_len > 0 && $desc_len >= 250) {

				$description = String::truncate($description, 250);

			} elseif ($desc_len == 0) {

				$description = !empty($page['Page']['title']) ? strip_tags($page['Page']['title']) : null;
			}

			$channel = array();

			if (! empty($page)) {

				$channel = array(
					'title' => $page['Page']['title'],
					'link' => array(
						'plugin' => 'ev_blog',
						'controller' => 'blog_posts',
						'action' => 'index'
					),
					'description' => $description,
					'language' => 'en-us'
				);

			}

			return $this->set(compact('channel', 'posts'));
		}

		$this->paginate = array(
			'conditions' => array(
				'BlogPost.is_active' => 1,
				'BlogPost.posted <' => date('Y-m-d H:i:s')
			),
			'order' => 'BlogPost.posted DESC',
			'contain' => array(
				'ListingImage',
				'BlogComment' => array(
					'conditions' => array(
						'BlogComment.is_active' => 1
					),
					'order' => 'BlogComment.created ASC'
				),
				'BlogCategory'
			),
			'limit' => Configure::read('EvBlog.listing_limit')
		);

		$this->set('data', $this->paginate());

		// call the widgets
		$this->widgetLatestPosts();
		$this->widgetCategories();
		$this->widgetArchives();

		$this->view = 'EvBlog.BlogPosts/index';
	}


/**
 * view individual blog post
 */
	public function view($id) {

		$Model = $this->{$this->modelClass};

		$data = $Model->readForView($id);
		$this->Meta->set($data);

		$this->set('data', $data);

		// call the widgets
		$this->widgetLatestPosts();
		$this->widgetCategories();
		$this->widgetArchives();

		// Format the list of categories this post is assigned to. We just need to group them up
		// by the category name and url.
		$categories = array();

		foreach ($data['BlogCategory'] as $category) {

			$categories[$category['name']] = array(
				'plugin' => 'ev_blog',
				'controller' => 'blog_categories',
				'method' => 'view',
				$category['id']
			);
		}

		$this->set('categories', $categories);

		$this->view = 'EvBlog.BlogPosts/view';
	}

/**
 * Blog archive, listing for blogfs from a certian month
 *
 * @param string $date given month to get posts from
 */
	public function archives($date = false) {
		// Fall back incase no page is found
		$this->set('title_for_layout', "Blog");

		if (!$date) {
            $date = CakeTime::format(time(), '%y-%m-%d');
        }

        $date_bits = explode('-', $date);
        $date_short = $date_bits['0'] . '-' . $date_bits['1'];

		$category = array(
			'BlogArchive' => array(
				'name' => 'Blog Archive: ' .CakeTime::format($date, '%d/%m/%y')
			)
		);

		$this->Meta->set($category, 'BlogArchive');
		$this->set('category', $category);

		$this->paginate = array(
			'conditions' => array(
				'BlogPost.is_active' => 1,
				'BlogPost.posted >=' => $date_short . '-01 00:00:00',
				'BlogPost.posted <=' => $date . ' 23:59:59',
				'BlogPost.posted <' => date('Y-m-d H:i:s')
			),
			'order' => 'BlogPost.posted DESC',
			'contain' => array(
				'ListingImage',
				'BlogComment' => array(
					'conditions' => array(
						'BlogComment.is_active' => 1
					)
				),
				'BlogCategory'
			),
			'limit' => 10
		);

		$this->loadModel('EvBlog.BlogPost');
		$this->set('data', $this->paginate('BlogPost'));

		// call the widgets
		$this->widgetLatestPosts();
		$this->widgetCategories();
		$this->widgetArchives();

		$this->view = 'EvBlog.BlogPosts/archives';
	}

}
