<?php
App::uses('EvBlogAppModel', 'EvBlog.Model');

class BlogPost extends EvBlogAppModel {

	public $actsAs = array(
		'Routable.Routable' => array(
			'alias' => 'blog/:displayField',
			'config' => 'EvBlog'
		),
		'MetaData.Meta'
	);

	public $displayField = 'name';

	public $imageSlots = array(
		'listing' => array(
			'slots' => 1,
			'fields' => array(
				'alt'
			)
		),
		'main' => array(
			'slots' => -1,
			'fields' => array(
				'alt'
			)
		),
	);

	public $belongsTo = array(
		'BlogCategory' => array(
			'className' => 'EvBlog.BlogCategory'
		)
	);

	public $hasMany = array(
		'BlogTag' => array(
			'className' => 'EvBlog.BlogTag',
			'cascade' => true,
			'dependent' => true
		),
		'BlogComment' => array(
			'className' => 'EvBlog.BlogComment',
			'cascade' => true,
			'dependent' => true
		)
	);

	public $order = array(
		'BlogPost.posted' => 'DESC'
	);

	public $validate = array(
		'name' => array(
			'notempty' => array(
				'rule' => array('notempty')
			),
		),
		'posted' => array(
			'datetime' => array(
				'rule' => array('datetime')
			),
		),
		'body' => array(
			'notempty' => array(
				'rule' => array('notempty')
			),
		),
	);

	public function readForView($id, $query = array()) {

		$query['contain'][] = 'BlogTag';
		$query['contain']['BlogComment'] = array(
			'conditions' => array(
				'BlogComment.is_active' => 1
			),
			'order' => 'BlogComment.created ASC',
		);
		$query['contain'][] = 'BlogCategory';

		return parent::readForEdit($id, $query);
	}

	public function readForEdit($id, $query = array()) {

		$query['contain'][] = 'BlogTag';
		$query['contain']['BlogComment'] = array(
			'order' => 'BlogComment.created ASC'
		);
		$query['contain'][] = 'BlogCategory';

		return parent::readForEdit($id, $query);
	}


	public function beforeSave($options = array()) {

		if (isset($this->data['BlogPost']['blog_category_id']) && $this->data['BlogPost']['blog_category_id'] == 0) {

			$this->data['BlogPost']['blog_category_id'] = null;
		}

		if (!empty($this->data['BlogPost']['posted']) && !empty($this->data['Event']['posted_gmt'])) {
			// Convert date to GMT.
			$this->data['BlogPost']['posted'] = CakeTime::toServer($this->data['BlogPost']['posted'], Configure::read('Config.timezone'));
		}

		return parent::beforeSave($options);
	}


	public function afterFind($results, $primary = false) {

		$results = parent::afterFind($results, $primary);
		$length = Configure::read('EvBlog.snippet_length');

		foreach ($results as $k => $blog) {

			if (isset($blog['BlogPost'])) {

				$snippet = strip_tags($blog['BlogPost']['body']);
				if (strlen($snippet) > $length) {

					$snippet = substr($snippet, 0, $length). ' &hellip;';
				}

				$results[$k]['BlogPost']['snippet'] = $snippet;
			}

			if (! isset($blog['BlogPost']['blog_category_id']) || empty($blog['BlogPost']['blog_category_id'])) {

				$results[$k]['BlogPost']['blog_category_id'] = 0;
			}

			// implode all the tags into one string
			if (isset($blog['BlogTag']) && ! empty($blog['BlogTag'])) {

				// sort the tags
				$tags = array();
				foreach ($blog['BlogTag'] as $tag) {

					$tags[] = $tag['tag'];
				}

				$results[$k]['BlogPost']['tags'] = implode(', ', $tags);

			} else {

				$results[$k]['BlogPost']['tags'] = '';
			}
		}

		return $results;
	}


	public function afterSave($created, $options = array())
	{
		$this->BlogTag = ClassRegistry::init('EvBlog.BlogTag');

		if ($created === false) {

			$this->BlogTag->deleteAll(array(
				'BlogTag.blog_post_id' => $this->id
			));
		}

		if (isset($this->data['BlogPost']['tags']) && ! empty($this->data['BlogPost']['tags'])) {

			$tags = explode(',', $this->data['BlogPost']['tags']);
			$blogTags = array();

			foreach ($tags as $tag) {

				$blogTags[] = array(
					'BlogTag' => array(
						'tag' => trim($tag),
						'blog_post_id' => $this->id
					)
				);
			}

			unset($this->data['BlogPost']['tags']);
			$this->BlogTag->saveMany($blogTags);
		}

		return parent::afterSave($created, $options);
	}

}
