# EvBasket

## Installation

Simply add EvBasket to the `install.php` file within the Config directory and run `Console/cake installer core` once you have added all the other plugins you will need. EvBasket has a dependency of EvCurrency, this also needs adding to the `install.php` array before the installer command is executed.

If this has already been run, simply run `Console/cake installer plugin EvBasket`. Additionally, `Console/cake installer plugin EvCurrency` should also be executed to prevent runtime errors when interacting with EvBasket.

The installer will automatically setup the database tables, create the template row for the basket and create a content page and write an `ev_basket.php` config file with the ID set.

## How it works

The basket works on a dynamic basis, dealing with model names and model IDs to defne what's in the basket. A BasketManager component is bundled that handles most of the heavy lifting of modifying the basket and rebuidling the totals.

The basket totals are also built to handle any number of rows and row types. The order totals are simply "summed". This allows you define a custom label and just add either a postive number or negative number to make the row either add to total or remove from to provide a discount.

The plugin comes with setup controller for handling the processing and loading of the component for the 4 CRUD actions of the basket, activating this by simply POSTing to the correct route and letting the controller / component do the rest.

A cookie is used to store a Hash string which is stored against the main Basket row, this is used to allow the custom to return to their basket. The cookie will become invalid and regenerate **after 1 week**.

The basket plugin can also handle a placeholder shipping value and also tax this value should it be required. This could be because you wish to give customers a rough guide on how much their order will be in total before shipping is actually calculated.

## Setup

Once the installer is run, the config file will need to be opened as numerous attributes will need to be set. These are defined to enable the basket item rows display what it actually contains.

The first attribute that needs to be defined is an array containing the belongsTo relationships that each `BasketItem` row will belong to. This should be a full array of all the possible different model items that could be added to the basket.

The example below is taken from Brighton Tools, upon loading this array is looped and each relationship passed to the `bindModel()` method available to all models.

	'BasketItemBelongsTo' => array(
		'Variant' => array(
			'className' => 'EvShop.Variant',
			'foreignKey' => 'model_id',
			'conditions' => array(
				"BasketItem.model = 'EvShop.Variant'"
			)
		)
	),

The second attribute is an array which is to be passed to the "contain" parameter when loading `BasketItem`. This will allow us to then loading in all the data needed to visually display each item in the basket.

The below example is taken from Brighton Tools and shows the Variant and Product being loaded alongside the Products Custom Field values and the Listing Image for the product so we can show, not only the product name / variant in the basket but a small image preview.

	'BasketItemContains' => array(
		'Variant' => array(
			'Product' => array(
				'ListingImage',
				'CustomFields' => array(
					'Field'
				)
			)
		)
	),

The third attribute is a definition of multiple array values which are essential Hash syntax paths and a separator element, defining how to separate multiple paths if defined. These are all grouped by the model name that is attached to the Basket Item database row. This is so when looking up the `BasketItem` row, the model column can be then used to work out what sort of model this `BasketItem` row is attached to and therefore get the display name data.

The following code is an example from Brighton Tools, and shows the full example. The first parameter is a assocative element with the key of 'separator' so the code knows to use that value to separate the rest of the array elements (**default is a single space**), the next two elements are a Hash path passed to `Hash::get` to retrieve the product name and then the variant name.

	'BasketItemNamePath' => array(
		'EvShop.Variant' => array(
			'separator' => ': ',
			'Variant.Product.name',
			'Variant.name'
		)
	),

###Advanced basket name rows
From version 2.2.3.0 you can take better control of the text within the basket rows. Whilst the above config method is fine for products, for more advanced baskets you may be linking to tables that don't have any sufficient information to display. Priort to 2.2.3.0 your basket would show 'Unknown' as the item text in these sorts of situations.

This update simply adds a check to see if a method called `basketRowName` exists in the model of the basket item. This is only checked if the config file doesnt contain a `BasketItemNamePath` entry for the model.

If a `basketRowName` method exists in the model, that will be used. The method should allow a single `id` parameter. You can then do whatever you need to format the basket item name text. This should be returned as a string. You can also use HTML with this method.

## Custom Return URL
Sometimes you may wish to return somewhere other than the default/config locations.

```
<?=$this->Form->input('ReturnRedirect', ['type' => 'hidden', 'value' => $this->here])?>
<?=$this->Form->input('ReturnRedirect', ['type' => 'hidden', 'value' => '/my_custom_location/'])?>
```

## Custom Messages
You can provide custom messages if you don't wish to you the standard success/warning/fail messages e.g.

```
// config

'flash' => array(
			'add' => array(
				'success' => array(
					'title' => 'Items added',
					'description' => 'Item has been added to the basket'
				),
				'success_trial' => array(
					'title' => 'Trial Added',
					'description' => 'The trial has been added to your account'
				)
			),
		),
		
// form

<?=$this->Form->input('CustomSuccess', ['type' => 'hidden', 'value' => 'success_trial'])?>
<?=$this->Form->input('CustomFailure', ['type' => 'hidden', 'value' => 'fail_trial'])?>
<?=$this->Form->input('CustomWarning', ['type' => 'hidden', 'value' => 'warning_trial'])?>
```

## Template Setup

Within the templates, on your product (or the page where you wish to be able to add a product from) there is a View element available, bundled with EvBasket that sets up the add to basket form. Simply echo out the element as normal, just making sure you pass the required information of model / model_id / quantity.

	<?php
		echo $this->element(
			'EvBasket.add-to-basket',
			[
				'item' => [
					'model' => 'PluginName.Model',
					'model_id' => 69,
					'quantity' => 1 // this is used just to prefil the quantity box
				],
				'submit' => []
			]
		);
	?>

If you will be customising some of the basket templates, there is also a `BasketHelper` to help with certain View functions, such as generating the delete link for the main Basket Page and getting the url for the basket.

## Basket Summary Setup

Bundled with the Basket plugin is the ability to show a basket summary. To set this up you will first need to redefine the constructor in a class such as `AppController`, this is to make the basket manage available on every page (assuming the basket summary wants to be available on every page).

	public function __construct(CakeRequest $Request, CakeResponse $Response) {
		if (empty($this->components['BasketManager'])) {
			$this->components['BasketManager'] = array(
				'className' => 'EvBasket.Basket',
				'Basket' => EvClassRegistry::init('EvBasket.Basket'),
				'BasketItem' => EvClassRegistry::init('EvBasket.BasketItem'),
				'BasketTotal' => EvClassRegistry::init('EvBasket.BasketTotal'),
				'BasketData' => EvClassRegistry::init('EvBasket.BasketData')
			);
		}

		parent::__construct($Request, $Response);
	}

This makes the basket component available at `$this->BasketManager` from within the Controller.

Then using the `beforeRender()` controller callback method we can perform the following:

	public function beforeRender() {
		parent::beforeRender();

		$this->set(
			'basketSummary',
			$this->BasketManager->getBasketSummary()
		);
	}

By calling it in the `beforeRender()`, this ensures all the main controller actions have been run and that any changes made in the current request will be reflected in the basket summary straight away.

In your views, you can simply call the `basket-summary` element to display the summary.

	<?=$this->element('EvBasket.basket-summary')?>

## Addable Items

For any models that are to be "Addable" items to the basket, some extra methods will need to be created.

First is required, this is `getUnitPrice($itemId)`. This method takes the model id we are trying to add to the basket and expects an amount back which is to be used as the Unit Price for that item, this will then be multipled by the quantity.

Full example below from Variant model within EvShop.

	/**
	 * get the unit price of the variant
	 * used by EvBasket
	 *
	 * @param 	int 	$itemId 	itemId
	 * @return 	float|bool
	 */
	public function getUnitPrice($itemId) {
		$Variant = $this->find(
			'first',
			array(
				'conditions' => array(
					'Variant.id' => $itemId
				)
			)
		);

		if (empty($Variant)) {
			return false;
		}

		$salePrice = (float)$Variant['Variant']['sale_price'];
		if (! empty($salePrice)) {
			return $Variant['Variant']['sale_price'];
		}

		return $Variant['Variant']['price'];
	}

The second method is optional, this is `getTaxRate($itemId)`. This is used to get the Tax rate we need to use on the basket item. This method accepts the model id we are trying to add to the basket an expects a percentage returned from the method to be used to calculate tax.

Full example below from Variant model within EvShop.

	/**
	 * get the tax rate for the variant
	 * used by EvBasket
	 *
	 * @param 	int 	$itemId 	item ID
	 * @return 	float
	 */
	public function getTaxRate($itemId) {
		if (! CakePlugin::loaded('EvTax')) {
			return 0;
		}

		$Variant = $this->find(
			'first',
			array(
				'conditions' => array(
					'Variant.id' => $itemId
				),
				'contain' => array(
					'Product'
				)
			)
		);

		if (empty($Variant) || empty($Variant['Product']['tax_level_id'])) {
			return 0;
		}

		$TaxLevel = EvClassRegistry::init('EvTax.TaxLevel');
		$taxRate = $TaxLevel->fieldOnly(
			'rate',
			array(
				'TaxLevel.id' => $Variant['Product']['tax_level_id']
			)
		);

		if (is_null($taxRate)) {
			return 0;
		}

		return $taxRate;
	}

##Multi-Currency Support
As of version 2.2.0.0 EvBasket uses the updated multi-currency shop.

As with most E-Commerce systems, EvBasket does not allow two items from different currencies to be in the basket. To prevent currency conflicts, baskets are purged if the EvCurrency setCurrency component method is called.

###Changing the currency
To change the currency of a basket you can use the `updateCurrency` method in the Basket Manager. What this method does is reads the current basket and takes all the items and data. It then updates the currency and then adds back the items and data so they are added with the correct prices.

There is a small element that can be used that creates a form with a dropdown select with all the available currencies.

##Basket Data
A data table is provided to store generic basket data that can later be passed through to EvTransaction and EvCheckout if you're using them.

The basket data table was created for things like EvDiscount that need to store a dicount code against orders. By adding the discount code to the basket data table, it gets pull ed through to both the transaction and order record.

Data records are automatically added when discount codes or loyalty points are applied to a basket.

###Basket Item Data

Much like Basket Data, Basket Item Data allows for additional information to be stored against each basket item. This is used for discounts and promotions to indicate when a basket item has been added due to a promotion, for example. The promotional item will have the `promotion_id` stored against it to allow us to customise the display of promotional items in the basket.

## Events

### EvBasket.Component.Basket.itemAdd

Called in the Basket Component by addItem right have an item has been successfully added / updated

#### Parameters

* `basketId` - The id of the basket row we have added to
* `items` - An array containing the items successfully updated

### EvBasket.Component.Basket.itemUpdated

Called in the Basket Component by updateItem right have an item has been successfully updated

#### Parameters

* `basketId` - The id of the basket row we have added to
* `items` - An array containing the items successfully updated

### EvBasket.Component.Basket.itemDeleted

Called in the Basket Component by deleteItem right have an item has been successfully deleted

#### Parameters

* `basketId` - The id of the basket row we have added to
* `itemKey` - The item key we have added in model.model_id format
* `BasketItem` - A copy of the BasketItem object that just saved the item

### EvBasket.Component.Basket.totalRowUpdated

Called in the Basket Component when one of the basket totals have been updated (e.g. Delivery)

#### Parameters

* `basketId` - The id of the basket row we have added to
* `data` - An array containing the name and other information about the total that has been updated.

### EvBasket.Component.Basket.postTotalRecompile

Called in the Basket Component after any totals have successfully recompiled

#### Parameters

* `basketId` - The id of the basket row we have added to

## Basket Dropout Emails

Abandonned baskets can now be tracked and an email can be sent out to users encouraging them to finish the checkout along with a link that takes them to the payment screen with their basket setup. To send out the emails there is a shell script which can be setup as a cron to run every day.

* `Console/cake EvBasket.DropoutEmails run`

The email subject/body/link can be configured as well as the redirect location from the email link. See the `basketDropout` key in `config.php`.

## Additional Information

### Non-EU based customers

A check is done to see whether the current customer should be charged VAT based on their location and the `EvCheckout` based `removeVatOutsideEU` config value. When the customer is not due to pay any VAT on their order a template variable is set to allow `EvBasket` based view templates to be handled accordingly.

I.e. Configured to display non tax based pricing when needed.
