<?php
App::uses('Basket', 'EvBasket.Model');

/**
 * Basket Test Case
 */
class BasketTest extends CakeTestCase {

	/**
	 * Fixtures
	 *
	 * @var array
	 */
	public $fixtures = array(
		'plugin.ev_basket.basket',
		'plugin.ev_basket.basket_item',
		'plugin.ev_basket.basket_total',
		'plugin.ev_basket.test_model'
	);

	const CURRENCY_ID = 1;

	/**
	 * setUp method
	 *
	 * @return void
	 */
	public function setUp() {
		parent::setUp();
		Configure::write(
			'EvBasket.BasketItemBelongsTo',
			array(
				'TestModel' => array(
					'className' => 'EvBasket.TestModel',
					'foreignKey' => 'model_id',
					'conditions' => array(
						"BasketItem.model = 'EvBasket.TestModel'"
					)
				)
			)
		);

		$this->Basket = ClassRegistry::init('EvBasket.Basket');
	}

	/**
	 * tearDown method
	 *
	 * @return void
	 */
	public function tearDown() {
		unset($this->Basket);
		return;

		parent::tearDown();
	}

	public function testCheckExistsReturnsTrueWhenValidHashGivenAndFalseWhenNot() {
		$this->assertTrue(
			$this->Basket->checkExists('55d1f4ad-0e40-475f-b4c0-40d7c0a83886', BasketTest::CURRENCY_ID)
		);

		$this->assertFalse(
			$this->Basket->checkExists('68838a0c7d04-0c4b-f574-04e0-da4f1d55', BasketTest::CURRENCY_ID)
		);
	}

	public function testGetFullBasketReturnsFullBasketArray() {
		$this->assertEquals(
			array(
				'Basket' => array(
					'id' => 1,
					'hash' => '55d1f4ad-0e40-475f-b4c0-40d7c0a83886',
					'currency_id' => BasketTest::CURRENCY_ID,
					'created' => '2015-08-17 08:58:06',
					'modified' => '2015-08-17 08:58:06'
				),
				'BasketItem' => array(
					'EvBasket.TestModel.1' => array(
						'id' => '1',
						'basket_id' => '1',
						'model' => 'EvBasket.TestModel',
						'model_id' => '1',
						'unit_price' => 83.33,
						'quantity' => '1',
						'row_total' => '99.990000',
						'tax_rate' => '20.00',
						'unit_tax_amount' => 16.66,
						'row_tax_amount' => 16.66,
						'unit_price_incTax' => 99.99,
						'row_total_incTax' => 99.99,
						'created' => '2015-08-17 08:59:29',
						'modified' => '2015-08-17 08:59:29',
						'TestModel' => array(
							'id' => '1',
							'name' => 'A Test Record',
							'created' => '2015-07-20 16:14:47',
							'modified' => '2015-07-20 16:14:47'
						)
					),
					'EvBasket.TestModel.2' => array(
						'id' => '2',
						'basket_id' => '1',
						'model' => 'EvBasket.TestModel',
						'model_id' => '2',
						'unit_price' => '5.500000',
						'quantity' => '1',
						'row_total' => '5.500000',
						'tax_rate' => 0,
						'unit_price_incTax' => '5.500000',
						'unit_tax_amount' => 0,
						'row_tax_amount' => 0,
						'row_total_incTax' => '5.500000',
						'created' => '2015-08-17 08:59:29',
						'modified' => '2015-08-17 08:59:29',
						'TestModel' => array(
							'id' => '2',
							'name' => 'A Test Record Number two',
							'created' => '2015-07-20 16:14:47',
							'modified' => '2015-07-20 16:14:47'
						)
					)
				),
				'BasketTotal' => array(
					'2' => array(
						'id' => '2',
						'basket_id' => '1',
						'name' => 'Sub Total',
						'amount' => '25.500000',
						'sequence' => '10',
						'is_grand_total' => false,
						'created' => '2015-08-17 09:00:17',
						'modified' => '2015-08-17 09:00:17'
					),
					'3' => array(
						'id' => '3',
						'basket_id' => '1',
						'name' => 'Delivery',
						'amount' => '3.000000',
						'sequence' => '20',
						'is_grand_total' => false,
						'created' => '2015-08-17 09:00:17',
						'modified' => '2015-08-17 09:00:17'
					),
					'1' => array(
						'id' => '1',
						'basket_id' => '1',
						'name' => 'Grand Total',
						'amount' => '28.5000000',
						'sequence' => '30',
						'is_grand_total' => true,
						'created' => '2015-08-17 09:00:17',
						'modified' => '2015-08-17 09:00:17'
					)
				),
				'BasketData' => array()
			),
			$this->Basket->getFullBasket(
				'55d1f4ad-0e40-475f-b4c0-40d7c0a83886',
				array(
					'TestModel'
				),
				BasketTest::CURRENCY_ID
			)
		);
	}

	public function testCreateBasketCreatesAndReturnsTrueAndFalseWhenADuplicateHashGiven() {
		$this->assertTrue(
			$this->Basket->createBasket('68838a0c7d04-0c4b-f574-04e0-da4f1d55', BasketTest::CURRENCY_ID)
		);

		$this->assertFalse(
			$this->Basket->createBasket('68838a0c7d04-0c4b-f574-04e0-da4f1d55', BasketTest::CURRENCY_ID)
		);
	}

	public function testCalculateTax() {
		$basket = $this->Basket->getFullBasket(
			'55d1f4ad-0e40-475f-b4c0-40d7c0a83886',
			array(),
			BasketTest::CURRENCY_ID
		);

		// VAT rate
		$rate = 20;

		// Loop through all prices and check the VAT is calculated correctly
		for ($priceIncVat = 0.01; $priceIncVat < 1000; $priceIncVat += 0.01) {
			$priceExVat = (float)$priceIncVat / 1.2;
			$priceExVat = number_format($priceExVat, 3, '.', '');

			for ($quantity = 1; $quantity <= 100; $quantity += 3) {
				$basketItems = array(
					'Item.1' => array(
						'tax_rate' => 20,
						'unit_price' => $priceExVat,
						'quantity' => $quantity
					)
				);
				$basketItems = $this->Basket->BasketItem->calculateTax($basketItems);

				$this->assertEqual($basketItems['Item.1']['row_total_incTax'], number_format($priceIncVat * $quantity, 2, '.', ''));
			}

		}
	}
}
