<?php

App::uses('EvBasketAppModel', 'EvBasket.Model');

App::uses('BasketLib', 'EvBasket.Lib');

/**
 * BasketItem Model
 *
 * @property Basket $Basket
 * @property Model $Model
 */
class BasketItem extends EvBasketAppModel {

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'basket_id' => array(
			'numeric' => array(
				'rule' => array('numeric'),
				'on' => 'create'
			),
		),
		'model' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'on' => 'create'
			),
		),
		'model_id' => array(
			'numeric' => array(
				'rule' => array('numeric'),
				'on' => 'create'
			),
		),
		'quantity' => array(
			'numeric' => array(
				'rule' => array('numeric')
			),
		)
	);

	// The Associations below have been created with all possible keys, those that are not needed can be removed

/**
 * belongsTo associations
 *
 * @var array
 */
	public $belongsTo = array(
		'Basket' => array(
			'className' => 'EvBasket.Basket'
		)
	);

	public $hasMany = array(
		'BasketItemData' => array(
			'className' => 'EvBasket.BasketItemData',
			'foreignKey' => 'basket_item_id',
			'dependent' => true,
		)
	);

/**
 * constructor - setup the dynamic relationships
 *
 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		$relationships = Configure::read('EvBasket.BasketItemBelongsTo');
		if (! empty($relationships)) {
			$belongsTo = array();
			foreach ($relationships as $key => $value) {
				$belongsTo[$key] = $value;
			}

			if (! empty($belongsTo)) {
				$this->bindModel(
					array(
						'belongsTo' => $belongsTo
					),
					false
				);
			}
		}
	}

	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		if ($primary === true && (isset($results[0]['model']) && isset($results[0]['model_id']))) {
			$results = Hash::combine(
				$results,
				array(
					'%s.%s',
					'{n}.BasketItem.model',
					'{n}.BasketItem.model_id'
				),
				'{n}'
			);
		}

		return $results;
	}

/**
 * calculate the tax for each basket item
 *
 * NOTE TO FUTURE DEVELOPERS: This way of working out tax may seem a little odd but
 * if you work out the tax and subtotals independent of eachother then you end up with
 * penny out issues or not able to get 99.99 as a inc vat price.
 * This code works by taking the ex VAT price that we store in the DB, it then works out the
 * rounded listing price. Then using the rounded listing price it calculates the tax by
 * subtracting them.
 *
 * @param 	array 	$BasketItems 		Array of basket items
 * @return 	arrayh 	$calculatedItems	Array of basket items with calculated tax
 */
	public function calculateTax($basketItems) {
		foreach ($basketItems as $key => $item) {
			if (! empty($item['tax_rate'])) {
				if (empty($item['unit_tax'])) {
					$item['unit_tax'] = $this->calculateUnitTax($item['unit_price'], $item['tax_rate']);

					$item['unit_price'] = round($item['unit_price'], 2);
				}

				//Calculate to total unit amount
				$unitPriceWithTax = $item['unit_tax'] + $item['unit_price'];

				$basketItems[$key]['row_tax_amount'] = $item['unit_tax'] * $item['quantity'];
				$basketItems[$key]['unit_price_incTax'] = $unitPriceWithTax;
				$basketItems[$key]['row_total_incTax'] = $unitPriceWithTax * $item['quantity'];

			} else {
				$basketItems[$key]['unit_price_incTax'] = $item['unit_price'];
				$basketItems[$key]['unit_tax'] = 0;
				$basketItems[$key]['row_tax_amount'] = 0;
				$basketItems[$key]['row_total_incTax'] = $item['row_total'];
			}
		}
		return $basketItems;
	}

/**
 * Provided the unit price and tax rate, calculate the amount of tax an item will have to reach it's total price
 * including tax.
 * @param  decimal $unitPrice The unit price of the item
 * @param  decimal $taxRate   The tax rate of the item
 * @return decimal            The amount of tax a unit item has
 */
	public function calculateUnitTax($unitPrice, $taxRate) {
		$unitPriceRounded = round($unitPrice, 2);

		// Get correct sale price
		$unitPriceWithTaxNonRounded = (1 + ($taxRate / 100)) * $unitPrice; //=99.99

		// Round to 2d.p
		$unitPriceWithTax = round($unitPriceWithTaxNonRounded, 2);

		//Calculate unit tax
		return $unitPriceWithTax - $unitPriceRounded;
	}

/**
 * Add an item to the basket.
 *
 * @param int     $basketId  ID of the basket row.
 * @param array   $item      Array with two elements of model and model_id.
 * @param decimal $unitPrice Unit Price of the item.
 * @param int     $quantity  Number of items to add.
 * @param decimal $taxRate   The tax rate to update.
 * @return bool True if the item has been added, false otherwise.
 */
	public function addItem($basketId, $item, $unitPrice, $quantity, $taxRate = 0) {
		$this->clear();

		$unitTax = $this->calculateUnitTax($unitPrice, $taxRate);
		$unitPrice = round($unitPrice, 2);

		$data = array(
			'BasketItem' => array(
				'basket_id' => $basketId,
				'name' => !empty($item['name']) ? $item['name'] : '',
				'model' => $item['model'],
				'model_id' => $item['model_id'],
				'unit_price' => $unitPrice,
				'unit_tax' => $unitTax,
				'tax_rate' => $taxRate,
				'quantity' => $quantity,
				'row_total' => ($unitPrice * $quantity),
			)
		);

		if (! empty($item['BasketItemData'])) {
			foreach ($item['BasketItemData'] as $key => $value) {
				$data['BasketItem']['BasketItemData'][] = array(
					'name' => $key,
					'item_data' => $value
				);
			}
		}

		$data = $this->_modifyBasketItemData($data, $item);

		return (bool)$this->saveAssociated($data, array('deep' => true));
	}

/**
 * Update an item in the basket
 *
 * @param array   $basketItem  The existing basket item. Used as a fallback if required data is missing from the $item.
 * @param array   $item        The basket item info that the basket item is updated with.
 * @param int     $newQuantity The new quantity of the basket item.
 * @param decimal $unitPrice   The new unit price of the basket item.
 * @param decimal $taxRate     The new tax rate of the basket item.
 * @param decimal $unitTax     The new unit tax of the basket item.
 * @return bool True if the basket item is updated, false otherwise.
 */
	public function updateItem($basketItem, $item, $newQuantity, $unitPrice = null, $taxRate = 0, $unitTax = null) {
		$this->clear();

		$item['id'] = $basketItem['id'];
		$item['quantity'] = $newQuantity;

		if (!empty($unitPrice)) {
			//If we are updating the unit price, calculate the new row total.
			$item['unit_price'] = $unitPrice;
			$item['row_total'] = $item['unit_price'] * $newQuantity;
		} else {
			// If we're not updating the unit price use old one to calculate the new row total.
			$item['unit_price'] = $basketItem['unit_price'];
			$item['row_total'] = $basketItem['unit_price'] * $newQuantity;
		}

		// if we're not updating the tax rate use old one
		if (!empty($taxRate)) {
			$item['tax_rate'] = $taxRate;
		} else {
			$item['tax_rate'] = $basketItem['tax_rate'];
		}

		// if we're not updating the unit tax use old one
		if (!empty($unitTax)) {
			$item['unit_tax'] = $unitTax;
		} else {
			$item['unit_tax'] = $this->calculateUnitTax($item['unit_price'], $item['tax_rate']);
		}

		$data['BasketItem'] = $item;

		if (!empty($item['BasketItemData'])) {
			foreach ($item['BasketItemData'] as $key => $value) {
				$dataItemData = [
					'name' => $key,
					'item_data' => $value,
				];

				//Check if the item data already exists, if it does overwrite it to keep unique
				//name value pairs.
				$overwrite = $this->BasketItemData->checkForExistingData(
					$item['id'],
					$key
				);

				if ($overwrite !== false) {
					$dataItemData['id'] = $overwrite;
				}

				$data['BasketItemData'][] = $dataItemData;
			}
		}

		if (isset($data['BasketItem']['BasketItemData'])) {
			unset($data['BasketItem']['BasketItemData']);
		}

		$data = $this->_modifyBasketItemData($data, $item, $basketItem);

		return (bool)$this->saveAssociated($data);
	}

/**
 * Modify or format basket item data before it is saved. Extend htis method if you want to change the data in anyway
 * before a basket item is added to the database.
 *
 * @param array $itemData     The current basket item data that will be saved.
 * @param array $item         The basket item to be added.
 * @param array $fallbackItem If the basket item is being updated, then this will be the old basket item data.
 * @return array The modified $itemData.
 */
	protected function _modifyBasketItemData($itemData, $item, $fallbackItem = null) {
		return $itemData;
	}

/**
 * delete an item from the basket
 *
 * @param 	array 		$basketItem 	The existing BasketItem Row
 * @return 	bool
 */
	public function deleteItem($basketItem) {
		$this->clear();

		return $this->delete($basketItem['id']);
	}

/**
 * Build the basket item contain array.
 *
 * @param array $itemContain           Custom basket item contains array. If not empty this is used without modification.
 * @param array $itemRelationships     Custom basket item relationships to use to create the contain array if the $itemContain is empty.
 * @param bool  $useContainConfig      If $itemContain is empty, should the config be used. Default is true.
 * @param bool  $useRelationshipConfig If $itemRelationships is empty, should the config be used. Default is true.
 * @return array The basket item contains array.
 */
	public function buildItemContain(
		$itemContain = [],
		$itemRelationships = [],
		$useContainConfig = true,
		$useRelationshipConfig = true
	) {
		if (empty($itemContain) && $useContainConfig) {
			$itemContain = Configure::read('EvBasket.BasketItemContains');
		}

		if (empty($itemRelationships) && $useRelationshipConfig) {
			$itemRelationships = Configure::read('EvBasket.BasketItemBelongsTo');
		}

		if (empty($itemContain) && !empty($itemRelationships)) {
			$itemContain = array();

			foreach ($itemRelationships as $key => $value) {
				if (is_array($value)) {
					$itemContain[] = $key;
				} else {
					$itemContain[] = $value;
				}
			}
		}

		return $itemContain;
	}

/**
 * Format items so that their array keys are the unique basket item keys.
 *
 * @param array $items The basket items to format.
 * @return array.
 */
	public function formatItems($items) {
		foreach ($items as $itemIndex => $basketItem) {
			$basketItemData = $basketItem;
			if (isset($basketItem[$this->alias])) {
				$basketItemData = $basketItem[$this->alias];
			}

			$itemKey = BasketLib::getItemKey($basketItemData, []);

			if (empty($itemKey) || $itemIndex === $itemKey) {
				continue;
			}

			$items[$itemKey] = $basketItem;
			unset($items[$itemIndex]);
		}

		return $items;
	}

/**
 * Merge the items from the secondary basket into the primary basket. If the secondary items are
 * set to be removed then they are deleted.
 *
 * @param int $primaryBasketId The id of the basket to merge items into.
 * @param array $primaryItems The items to be merged into.
 * @param array $secondaryItems The items to be merged.
 * @param bool $removeSecondaryItems
 */
	public function mergeItems($primaryBasketId, $primaryItems, $secondaryItems, $removeSecondaryItems) {
		$dataSource = $this->getDataSource();
		$dataSource->begin();

		$mergedItems = [];
		$mergeKeys = array_intersect(array_keys($primaryItems), array_keys($secondaryItems));

		foreach ($secondaryItems as $itemKey => $item) {
			if (! isset($item['BasketItemData'])) {
				$item['BasketItemData'] = $this->BasketItemData->find(
					'all',
					[
						'conditions' => [
							'basket_item_id' => $item[$this->primaryKey],
						],
					]
				);
			}

			if (! in_array($itemKey, $mergeKeys)) {
				//This item doesn't exist in the primary set of items. Just add it in.
				unset($item['id']);
				$item = Hash::remove($item, 'BasketItemData.{*}.id');
				$item['basket_id'] = $primaryBasketId;

				$mergedItems[$itemKey] = $item;
				continue;
			}

			//Update the quantity and row_total of the existing item
			$existingItem = $primaryItems[$itemKey];

			if (! isset($existingItem['BasketItemData'])) {
				$existingItem['BasketItemData'] = $this->BasketItemData->find(
					'all',
					[
						'conditions' => [
							'basket_item_id' => $existingItem[$this->primaryKey],
						],
					]
				);
			}

			$existingItem['quantity'] += $item['quantity'];
			$existingItem['row_total'] += $item['row_total'];

			$mergedItems[$itemKey] = $existingItem;
		}

		if (empty($mergedItems)) {
			$dataSource->commit();
			return;
		}

		$saved = $this->saveMany($mergedItems);

		if (! $saved) {
			$dataSource->rollback();
			throw new Exception('Failed to save newly merged items');
		}

		if (! $removeSecondaryItems) {
			$dataSource->commit();
			return;
		}

		$deleted = $this->delete(Hash::extract($secondaryItems, '{*}.' . $this->primaryKey));

		if (! $deleted) {
			$dataSource->rollback();
			throw new Exception('Failed to delete secondary merged items');
		}

		$dataSource->commit();

		return;
	}
}
