<?php
App::uses('EvBasketAppModel', 'EvBasket.Model');
/**
 * BasketData Model
 *
 * @property Basket $Basket
 */
class BasketData extends EvBasketAppModel {

/**
 * Display field
 *
 * @var string
 */
	public $displayField = 'name';

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'basket_id' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				//'message' => 'Your custom message here',
				//'allowEmpty' => false,
				//'required' => false,
				//'last' => false, // Stop validation after this rule
				//'on' => 'create', // Limit validation to 'create' or 'update' operations
			),
		),
		'name' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				//'message' => 'Your custom message here',
				//'allowEmpty' => false,
				//'required' => false,
				//'last' => false, // Stop validation after this rule
				//'on' => 'create', // Limit validation to 'create' or 'update' operations
			),
		),
	);

	// The Associations below have been created with all possible keys, those that are not needed can be removed

/**
 * belongsTo associations
 *
 * @var array
 */
	public $belongsTo = array(
		'Basket' => array(
			'className' => 'Basket',
			'foreignKey' => 'basket_id',
			'conditions' => '',
			'fields' => '',
			'order' => ''
		)
	);

/**
 * set a data row for the basket (updates or creates one based on the basket id and name)
 *
 * @param int      $basketId   id of the basket that the record belongs to
 * @param string   $name       name of the data row
 * @param string   $data       value of the data row
 */
	public function setData($basketId, $name, $data, $isVisible = false) {
		$record = $this->checkForExistingData($basketId, $name);

		if ($record !== false) {
			$this->set('id', $record);
		} else {
			$this->clear();
		}

		$this->set(array(
			'basket_id' => $basketId,
			'name' => $name,
			'is_visible' => $isVisible
		));

		$this->set('data', $data);

		return (bool)$this->save();
	}

/**
 * Check if there is any existing data for the basket with the provided name. If there is
 * then return the id of the basket data, otherwise return false.
 *
 * @param int    $basketId The id of the basket to check for data on.
 * @param string $dataName The name of the basket data to check for.
 * @return int|bool The id of the basket data if found, false otherwise.
 */
	public function checkForExistingData($basketId, $dataName) {
		$existingData = $this->find(
			'first',
			[
				'conditions' => [
					$this->alias . '.basket_id' => $basketId,
					$this->alias . '.name' => $dataName,
				]
			]
		);

		if (!empty($existingData[$this->alias]['id'])) {
			return $existingData[$this->alias]['id'];
		}

		return false;
	}

/**
 * delete a single piece of data from the basket
 *
 * @param 	array 		$basketItem 	The existing BasketItem Row
 * @return 	bool
 */
	public function deleteData($basketData) {
		$this->clear();

		return $this->deleteAll(['BasketData.id' => Hash::extract($basketData, '{n}.id')]);
	}

/**
 * Merge the data from the secondary basket into the primary basket. If the secondary data is
 * set to be removed then they are deleted.
 *
 * @param int $primaryBasketId The id of the basket to merge items into.
 * @param array $primaryData The items to be merged into.
 * @param array $secondaryData The items to be merged.
 * @param bool $removeSecondaryData
 */
	public function mergeData($primaryBasketId, $primaryData, $secondaryData, $removeSecondaryData) {
		$dataSource = $this->getDataSource();
		$dataSource->begin();

		$primaryData = Hash::combine($primaryData, '{*}.name', '{*}');
		$secondaryData = Hash::combine($secondaryData, '{*}.name', '{*}');

		$mergedData = [];
		$mergeNames = array_intersect(array_keys($primaryData), array_keys($secondaryData));

		foreach ($secondaryData as $dataName => $data) {
			if (! in_array($dataName, $mergeNames)) {
				//This item doesn't exist in the primary set of items. Just add it in.
				unset($data[$this->primaryKey]);
				$data['basket_id'] = $primaryBasketId;

				$mergedData[$dataName] = $data;
				continue;
			}

			//Overwrite the data of the primary data
			$existingData = $primaryData[$dataName];
			$existingData['data'] = $data['data'];
			$mergedData[$dataName] = $existingData;
		}

		if (empty($mergedData)) {
			$dataSource->commit();
			return;
		}

		$saved = $this->saveMany($mergedData);

		if (! $saved) {
			$dataSource->rollback();
			throw new Exception('Failed to save newly merged data');
		}

		if (! $removeSecondaryData) {
			$dataSource->commit();
			return;
		}

		$deleted = $this->delete(Hash::extract($secondaryData, '{*}.' . $this->primaryKey));

		if (! $deleted) {
			$dataSource->rollback();
			throw new Exception('Failed to delete secondary merged data');
		}

		$dataSource->commit();

		return;
	}
}
