<?php

App::uses('CakeEventListener', 'Event');
App::uses('DiscountLib', 'EvDiscount.Lib');

class EvBasketCheckForDiscountListener implements CakeEventListener {

	protected $_promotionsLoaded = false;

	public function implementedEvents() {
		return array(
			'EvDiscount.Component.DiscountCodes.codeAdded' => 'addDiscount',
			'EvBasket.Component.Basket.itemAdd' => 'rebuildDiscount',
			'EvBasket.Component.Basket.itemUpdated' => 'rebuildDiscount',
			'EvBasket.Component.Basket.itemDeleted' => 'rebuildDiscount',
			'EvLoyaltyPoints.Component.Points.usePoints' => 'addLoyaltyDiscount',
			'EvBasket.Component.Basket.totalRowUpdated' => 'basketTotalRowUpdated',
		);
	}

/**
 * queue up the confirmation emails
 *
 * @param   CakeEvent
 */
	public function addDiscount(CakeEvent $Event, $data = null) {
		$dataCode = $Event->data['data'];

		// Event subject will be the EvDiscount.DiscountCodesComponent
		return $this->rebuildDiscount($Event, $dataCode);
	}

/**
 * Rebuild discount due to basket amends.
 *
 * @param CakeEvent $Event    The event that triggered the rebuild.
 * @param string    $dataCode The discount code applied.
 * @return null|bool.
 */
	public function rebuildDiscount(CakeEvent $Event, $dataCode = null) {
		if (!CakePlugin::loaded('EvDiscount')) {
			return;
		}

		$this->_promotionsLoaded = CakePlugin::loaded('EvPromotion');

		$this->_unapplyDiscounts($Event);

		$this->_addFreeItemsFromDiscounts($Event);

		if ($this->_promotionsLoaded) {
			$promotions = $this->_getPromotions($Event);
			if (!empty($promotions['failed'])) {
				$this->_handleFreeItemsFromPromotions($Event, $promotions['failed']);
			}
			if (!empty($promotions['applicable'])) {
				$this->_addFreeItemsFromPromotions($Event, $promotions['applicable']);
			}
		}

		$EventSubject = $Event->subject();

		$basket = $this->_getBasket($EventSubject);

		$priceModifier = $this->_setupPriceModifier($basket);

		$this->_populatePriceModifierBasket($priceModifier, $basket);

		$this->_populatePriceModifierDiscounts($priceModifier, $EventSubject, $dataCode);
		if ($this->_promotionsLoaded) {
			$this->_populatePriceModifierPromotions($priceModifier, $EventSubject, $promotions['applicable']);
		}

		// Make the price modifier apply the discounts and get the results back
		$priceModifier->applyDiscounts();
		$discountedBasket = $priceModifier->basket()->toArray();

		//Default discount position after subtotal. Discount will always contain what is to betaken off of the subtotal.
		//Tax amounts always contain the discounted value so display this after discount.
		$discountSequence = 15;

		// Loop through the discount breakdown and add up those for vouchers and those for discounts separately
		// Add a basket data record to handle the discount row
		$voucherPreTaxTotal = $voucherIncTaxTotal = $discountPreTaxTotal = $discountIncTaxTotal = 0;
		foreach ($discountedBasket['discountBreakdown'] as $discountCode => $discountAmount) {
			// Voucher codes get prefixed with V_ at the top of this function when adding them
			if (strpos($discountCode, "V_") === 0) {
				$EventSubject->_controller->BasketManager->setBasketData('voucher_code', str_replace("V_", "", $discountCode), true);
				$voucherPreTaxTotal += $discountAmount['amount'];
				$voucherIncTaxTotal += $discountAmount['amount'] + $discountAmount['tax'];
			} elseif (strpos($discountCode, "promotion_") === 0) {
				$EventSubject->_controller->BasketManager->setBasketData('promotion_id', str_replace("promotion_", "", $discountCode), true);
				$discountPreTaxTotal += $discountAmount['amount'];
				$discountIncTaxTotal += $discountAmount['amount'] + $discountAmount['tax'];
			} else {
				$EventSubject->_controller->BasketManager->setBasketData('discount_code', $discountCode, true);
				$discountPreTaxTotal += $discountAmount['amount'];
				$discountIncTaxTotal += $discountAmount['amount'] + $discountAmount['tax'];
			}
		}

		// Add Voucher discount total
		if ($voucherPreTaxTotal > 0) {
			//Don't update the tax as that is dealt with when rebuilding totals
			$EventSubject->_controller->BasketManager->manageTotalRow(
				Configure::read('EvBasket.labels.voucher'),
				-$voucherPreTaxTotal,
				$discountSequence,
				false,
				-$voucherPreTaxTotal,
				-$voucherIncTaxTotal
			);
		} else {
			// No discount is applicable to the basket remove the row
			$EventSubject->_controller->BasketManager->deleteTotalRow(Configure::read('EvBasket.labels.voucher'));
		}

		// Add Discount discount total
		if ($discountPreTaxTotal > 0) {
			//Don't update the tax as that is dealt with when rebuilding totals
			$EventSubject->_controller->BasketManager->manageTotalRow(
				Configure::read('EvBasket.labels.discount'),
				-$discountPreTaxTotal,
				$discountSequence,
				false,
				-$discountPreTaxTotal,
				-$discountIncTaxTotal
			);
		} else {
			// No discount is applicable to the basket remove the row
			$EventSubject->_controller->BasketManager->deleteTotalRow(Configure::read('EvBasket.labels.discount'));
		}

		// Now we have added the discount row we can rebuild the totals
		$Event->subject()->_controller->BasketManager->rebuildTotals();

		return true;
	}

/**
 * Update discounts if a row total changes. This function filters out the discount row change to avoid
 */
	public function basketTotalRowUpdated(CakeEvent $Event) {
		$updateData = $Event->data['data'];

		if ($updateData['name'] == Configure::read('EvBasket.labels.delivery')) {
			return $this->rebuildDiscount($Event);
		} else {
			return true;
		}
	}

	public function addLoyaltyDiscount(CakeEvent $Event) {
		if (!CakePlugin::loaded('EvLoyaltyPoints')) {
			return false;
		}

		$EventSubject = $Event->subject();

		if (!is_object($EventSubject->_controller->BasketManager)) {
			$EventSubject->_controller->BasketManager = $EventSubject->_controller->loadComponent('EvBasket.BasketManager');
		}

		$total = $EventSubject->_controller->BasketManager->findTotalRow(
			Configure::read('EvBasket.labels.subtotal')
		);

		App::uses('PointLib', 'EvLoyaltyPoints.Lib');

		if ($Event->data['pointsToUse'] <= 0) {
			$discount = 0;
		} else {
			$discount = PointLib::calculate($total['amount'], 'EvBasket', 0, $Event->data['pointsToUse']);
		}

		$EventSubject->_controller->BasketManager->manageTotalRow(
			Configure::read('EvBasket.labels.discount'),
			-$discount,
			15
		);

		// Add a basket data record to handle the loyalty disount row
		$EventSubject->_controller->BasketManager->setBasketData('loyalty_points_used', $Event->data['pointsToUse'], true);

		$Event->subject()->_controller->BasketManager->rebuildTotals();

		return true;
	}

/**
 * Get the current basket to apply a discount to.
 *
 * @param obj $EventSubject The class that fired this event.
 * @return array Basket data.
 */
	protected function _getBasket($EventSubject) {
		return $EventSubject->_controller->BasketManager->getBasket();
	}

/**
 * Adds some free items to the basket if other qualifying items are present
 */
	protected function _addFreeItemsFromDiscounts($Event) {
		$EventSubject = $Event->subject();
		// The cookie isn't used any more but there can still be multiple discount codes added to the basket in the
		// database potentially. Current functionality only allows one discount code at a time though.
		if (!is_object($EventSubject->_controller->BasketManager)) {
			$EventSubject->_controller->BasketManager = $EventSubject->_controller->loadComponent('EvBasket.BasketManager');
		}

		// Get discount codes from basket data
		$codes = array();
		$basketData = $EventSubject->_controller->BasketManager->getBasketData();
		if (!empty($basketData)) {
			foreach ($basketData as $key => $value) {
				if ($value['name'] == 'discount_code' || $value['name'] == 'voucher_code') {
					$codes[$value['data']] = $value;
				}
			}
		}

		// If the basket data returns some codes or one is passed through (or both) then
		if (!empty($codes) && is_array($codes)) {
			// Checks with the discount module to see what codes are active and not run out etc.
			// Only returns codes that are still valid
			$DiscountCode = EvClassRegistry::init('EvDiscount.DiscountCode');
			$validCodes = $DiscountCode->validateCodes($codes);

			if (!empty($validCodes)) {

				$basket = $EventSubject->_controller->BasketManager->getBasket();

				// If there were any valid codes...
				foreach ($validCodes as $code => $codeData) {

					if ($codeData['DiscountType']['name'] == 'Free gift') {
						$this->_addFreeProducts($EventSubject->_controller->BasketManager, $codeData['DiscountData']['free_gift_product_id'][0], $codeData['CodeRestriction'], $codeData['DiscountCode']['amount']);
					}
				}
			}
		}
	}

/**
 * Removes all discounts applied to basket
 *
 * @param CakeEvent $Event rebuildDiscount event
 *
 * @return void
 */
	protected function _unapplyDiscounts(CakeEvent $Event) {
		$EventSubject = $Event->subject();
		$promotions = $this->_getPromotions($Event);

		// If there are promotions that will fail this basket, remove all promotions so that we can start fresh
		if (!empty($promotions['failed'])) {
			$EventSubject->_controller->BasketManager->deleteTotalRow(Configure::read('EvBasket.labels.discount'));
			$Event->subject()->_controller->BasketManager->rebuildTotals();
		}
	}

/**
 * Attempts to add the requested free product to the basket, if it has not already been added
 *
 * @param object $BasketManager The basket manager component used
 * @param int $freeProductId The unique id of the required free product
 * @param array $restrictions The product / category / brand restrictions associated with the discount
 * @param int $amountToBuyPerFreeProduct The total number of qualifying products required to qualify
 * @param int $promotionId Contains the unique id of the promotion being applied
 * @param int $quantity The total number of free products to add per promotion
 *
 * @return void
 */
	protected function _addFreeProducts(
		$BasketManager,
		$freeProductId,
		$restrictions,
		$amountToBuyPerFreeProduct,
		$promotionId = null,
		$quantity = null
	) {
		$currentFreebies = 0;

		$totalCount = 0;

		$promotionApplied = false;

		$basket = $BasketManager->getBasket();

		if (!empty($promotionId)) {
			// firstly, loop around the basket and see whether the supplied
			// promotion id has already been applied to the basket and is
			// found against a basket item
			foreach ($basket['BasketItem'] as $item) {
				if (!empty($item['is_promotional']) && !empty($item['BasketItemData'])) {
					// check the BasketItemData array for the promotion id
					foreach ($item['BasketItemData'] as $basketItemData) {
						if (
							$basketItemData['name'] == 'promotion_id' &&
							$basketItemData['item_data'] == $promotionId
						) {
							$promotionApplied = true;
							break;
						}
					}
				}
			}
		}

		if ($promotionApplied !== true) {
			// Check we have the required items in the basket.
			foreach ($basket['BasketItem'] as $basketItem) {

				// Get the product id (depends on the type of basket item)
				if ($basketItem['model'] == 'EvShop.Variant') {
					$productId = $basketItem['Variant']['Product']['id'];
				} else {
					// Not interested in this item
					continue;
				}

				if ($productId == $freeProductId) {
					$currentFreebies += $basketItem['quantity'];
				} else {
					$promotionIsApplicable = true;

					if (!empty($restrictions)) {
						$checkDiscountRestriction = new \CakeEvent('EvBasket.Component.BasketItem.CheckDiscountRestrictions', $basketItem, array(
							'discount' => [
								'code_restrictions' => $restrictions,
							],
							'basket_item' => $basketItem,
						));
						\CakeEventManager::instance()->dispatch($checkDiscountRestriction);

						// If false is returned discount isn't available for this item
						$result = $checkDiscountRestriction->result;

						$promotionIsApplicable = $result['discountValue'];
					}

					if ($promotionIsApplicable) {
						$totalCount += $basketItem['quantity'];
					}
				}
			}

			// should be quantity be assign, allow the requested amount, otherwise
			// calculate how many free items to add to the basket
			if (empty($totalCount)) {
				return null;
			}

			if (empty($quantity)) {
				$requiredFreebies = floor($totalCount / $amountToBuyPerFreeProduct) - $currentFreebies;
			} else {
				$requiredFreebies = $quantity;
			}

			if ($requiredFreebies > 0) {
				// add the free product to the basket
				$this->_addProductToBasket(
					$BasketManager,
					$freeProductId,
					$requiredFreebies,
					$promotionId
				);
			}
		}
	}

/**
 * Adds a product to the basket. Will add the first available variant.
 *
 * @param object $BasketManager The current basket manager component being used
 * @param int $productId The unique id of the product being added
 * @param int $quantity The quantity of items to add to the basket, defaults to 1
 * @param int $promotionId Contains the unique id of the promotion being applied
 *
 * @return void
 *
 */
	protected function _addProductToBasket($BasketManager, $productId, $quantity = 1, $promotionId = null) {
		$useInventory = false;

		$productQuery = [
			'conditions' => [
				'Product.id' => $productId,
			],
			'contain' => [
				'Variant',
			],
		];

		if (CakePlugin::loaded('EvInventory')) {
			// Only contain inventory if the plugin is loaded
			$useInventory = true;
			$productQuery['contain']['Variant'] = 'Inventory';
			// load the lib to check again oos later in the method
			App::uses('InventoryLib', 'EvInventory.Lib');
		}

		$Product = EvClassRegistry::init('EvShop.Product');
		$product = $Product->find('first', $productQuery);

		$totalRemaining = $quantity;
		foreach ($product['Variant'] as $variant) {
			if (
				!$useInventory ||
				($useInventory && InventoryLib::hasEnoughStock($variant, $quantity))
			) {
				// Assign a block of this variant and stop there
				$this->_addVariantToBasket(
					$BasketManager,
					$variant['id'],
					$totalRemaining,
					$promotionId
				);

				$totalRemaining = 0;
			} elseif ($variant['Inventory']['stock'] > 0) {
				// Assign as many as we can of this variant and move on to the next
				$this->_addVariantToBasket(
					$BasketManager,
					$variant['id'],
					$variant['Inventory']['stock'],
					$promotionId
				);

				$totalRemaining -= $variant['Inventory']['stock'];
			}

			if ($totalRemaining <= 0) {
				// We've added all we need to
				break;
			}
		}
	}

/**
 * Adds the supplied product variant to the current basket
 *
 * @param object $BasketManager The current basket manager component being used
 * @param int $variantId The unique id of the product variant being added
 * @param int $quantity The quantity of items to add to the basket, defaults to 1
 *
 * @return void
 */
	protected function _addVariantToBasket($BasketManager, $variantId, $quantity = 1, $promotionId = null) {
		// setup the data
		$item = array(
			'model' => 'EvShop.Variant',
			'model_id' => $variantId,
		);

		// if a promotion id is supplied, add this to the basket item as basket
		// item data to allow us to easily determine promotion items when
		// displaying the backet to customers
		if (!empty($promotionId)) {
			$item['BasketItemData'] = array(
				'promotion_id' => $promotionId,
			);
		}

		$Model = EvClassRegistry::init('EvShop.Variant');
		$unitPrice = $Model->getUnitPrice($variantId);

		// if there's a method to get the taxRate, do so
		$taxRate = 0;
		if (method_exists($Model, 'getTaxRate')) {
			$taxRate = $Model->getTaxRate($item['model_id']);
		}

		if ($unitPrice !== false) {
			// add to the basket but dont dispatch the event or we'll end up going around in circles.
			$BasketManager->addItem($item, $unitPrice, $quantity, $taxRate, false);
		}
	}

/**
 * Get all available promotions.
 *
 * @param CakeEvent $Event assign promotions
 * @return array contains applicable and failed promotions
 */
	protected function _getPromotions($Event) {
		if (!$this->_promotionsLoaded) {
			return [];
		}

		$applicablePromotions = [];
		$failedPromotions = [];

		$EventSubject = $Event->subject();

		// Get discount codes from basket data
		$Promotion = EvClassRegistry::init('EvPromotion.Promotion');
		$activePromotions = $Promotion->findAllActive();
		if (!empty($activePromotions)) {
			// If there were any...
			foreach ($activePromotions as $promoData) {
				// set where it applies to so it's nicer to access
				if ($promoData['DiscountType']['name'] == 'Free gift') {
					if (!empty($promoData['DiscountData']['free_gift_product_id'][0])) {
						$promoData['AppliesTo'] = $promoData['DiscountData']['free_gift_product_id'][0];
					}
				}

				// assign min order values
				$minOrderRequirementsMet = false;
				$minOrderValue = $promoData['Promotion']['min_order'];

				// check it's a valid value in min order
				if (!empty($minOrderValue) && is_numeric($minOrderValue)) {
					// grab the basket totals
					$basket = $EventSubject->_controller->BasketManager->buildTransactionTotals();

					$minOrderRequirementsMet = ($basket['grandtotal'] >= $minOrderValue) ? true : false;
				}

				if ($minOrderRequirementsMet) {
					$applicablePromotions[] = $promoData;
				} else {
					$failedPromotions[] = $promoData;
				}
			}
		}

		return ['applicable' => $applicablePromotions, 'failed' => $failedPromotions];
	}

/**
 * Adds some free items to the basket if other qualifying items are present using the active promotions
 */
	protected function _addFreeItemsFromPromotions($Event, $applicablePromotions) {
		// If there are active promotions
		if (!empty($applicablePromotions)) {
			if (!CakePlugin::loaded('EvPromotion')) {
				// Not using promotions
				return false;
			}

			$EventSubject = $Event->subject();
			if (!is_object($EventSubject->_controller->BasketManager)) {
				$EventSubject->_controller->BasketManager = $EventSubject->_controller->loadComponent('EvBasket.BasketManager');
			}

			$basket = $EventSubject->_controller->BasketManager->getBasket();
			if (empty($basket['BasketItem'])) {
				return false;
			}

			// If there were any...
			foreach ($applicablePromotions as $promoData) {
				if ($promoData['DiscountType']['name'] == 'Free gift') {

					$quantity = 1;

					foreach ($basket['BasketItem'] as $item) {
						$itemKey = $EventSubject->_controller->BasketManager->buildItemKey($item);

						if ($promoData['AppliesTo'] == $item['Variant']['product_id']) {
							if ($EventSubject->_controller->BasketManager->basketItemExists($itemKey)) {
								$EventSubject->_controller->BasketManager->deleteItem(['model' => $item['model'], 'model_id' => $item['model_id']], false);

								$quantity = ($item['quantity'] < $quantity) ? $quantity : $item['quantity'];
							}
						}
					}

					$this->_addFreeProducts(
						$EventSubject->_controller->BasketManager,
						$promoData['AppliesTo'],
						$promoData['CodeRestriction'],
						$promoData['Promotion']['amount'],
						$promoData['Promotion']['id'],
						$quantity // limit to adding one product what adding a freebie
					);
				}
			}
		}

		return true;
	}

	protected function _handleFreeItemsFromPromotions($Event, $failedPromotions) {
		if (!CakePlugin::loaded('EvPromotion')) {
			// Not using promotions
			return false;
		}

		$EventSubject = $Event->subject();
		if (!is_object($EventSubject->_controller->BasketManager)) {
			$EventSubject->_controller->BasketManager = $EventSubject->_controller->loadComponent('EvBasket.BasketManager');
		}

		$basket = $EventSubject->_controller->BasketManager->getBasket();
		if (empty($basket['BasketItem'])) {
			return false;
		}

		// sort basket items
		$promotionalBasketRows = [];
		foreach ($basket['BasketItem'] as $rowId => $row) {
			if (empty($row['is_promotional'])) {
				continue;
			}

			$row['product_id'] = $row['Variant']['product_id'];

			$promotionalBasketRows[$row['id']] = $row;
		}

		// If there are active promotions
		if (!empty($failedPromotions)) {
			// If there were any...
			foreach ($failedPromotions as $promoData) {
				foreach ($promotionalBasketRows as $basketKey => $basketRow) {
					if ($promoData['AppliesTo'] != $basketRow['product_id']) {
						continue;
					}
					// quantity needs to be made dynamic sometime
					$numberOfFreeItems = 1;
					$newQuantity = $basketRow['quantity'] - $numberOfFreeItems;

					if ($newQuantity > 0) {
						$EventSubject->_controller->BasketManager->unflagPromotionalItem($basketKey);
					}

					$EventSubject->_controller->BasketManager->updateItem(['model' => $basketRow['model'], 'model_id' => $basketRow['model_id']], $newQuantity, null, 0, false);
				}

				// delete from basketData if promotion failed
				if (!empty($basket['BasketData']) && is_array($basket['BasketData'])) {
					foreach ($basket['BasketData'] as $basketDatum) {
						if ($basketDatum['name'] == 'promotion_id' && $promoData['Promotion']['id'] == $basketDatum['data']) {
							$EventSubject->_controller->BasketManager->deleteDataWithKey($basketDatum['id'], true, 'id');

						}
					}
				}
				$EventSubject->_controller->BasketManager->getBasket();

			}
		}
	}

/**
 * Sets up the price modifier configuration and returns a setup version.
 *
 * @param array $basket Basket array.
 * @return PriceModifier Initialised price modifier object
 */
	protected function _setupPriceModifier($basket) {
		$priceModifier = new Evoluted\PriceModifier\PriceModifier(
			new Evoluted\PriceModifier\Basket\Basket(new Evoluted\PriceModifier\Storage\Runtime, $basket['Basket']['id'])
		);

		require_once (CakePlugin::path('EvDiscount') . 'Lib' . DS . 'DiscountRestrictions.php');
		$priceModifier->setDiscountRestrictionHandler('Evoluted\EvDiscount\DiscountRestrictions\DiscountRestrictions');

		// In the case of our shop we want vouchers to actually go past the subtotal amount so they apply to delivery as well.
		require_once (CakePlugin::path('EvDiscount') . 'Lib' . DS . 'VoucherDiscountModifier.php');
		$priceModifier->addDiscountModifier('voucher', 'Evoluted\PriceModifier\DiscountModifiers\VoucherDiscountModifier');

		require_once (CakePlugin::path('EvDiscount') . 'Lib' . DS . 'CheapestFreeDiscountModifier.php');
		$priceModifier->addDiscountModifier('Buy X get cheapest free', 'Evoluted\PriceModifier\DiscountModifiers\CheapestFreeDiscountModifier');

		require_once (CakePlugin::path('EvDiscount') . 'Lib' . DS . 'CheapestHalfPriceDiscountModifier.php');
		$priceModifier->addDiscountModifier('Buy X get cheapest half price', 'Evoluted\PriceModifier\DiscountModifiers\CheapestHalfPriceDiscountModifier');

		require_once (CakePlugin::path('EvDiscount') . 'Lib' . DS . 'XForYDiscountModifier.php');
		$priceModifier->addDiscountModifier('X for the price of Y', 'Evoluted\PriceModifier\DiscountModifiers\XForYDiscountModifier');

		require_once (CakePlugin::path('EvDiscount') . 'Lib' . DS . 'FreeDeliveryDiscountModifier.php');
		$priceModifier->addDiscountModifier('Free delivery', 'Evoluted\PriceModifier\DiscountModifiers\FreeDeliveryDiscountModifier');

		require_once (CakePlugin::path('EvDiscount') . 'Lib' . DS . 'FreeGiftDiscountModifier.php');
		$priceModifier->addDiscountModifier('Free gift', 'Evoluted\PriceModifier\DiscountModifiers\FreeGiftDiscountModifier');

		// Set whether the basket is an inc or exvat basket
		$priceModifier->applyDiscountsAfterTax = Configure::read('SiteSetting.ev_basket.discounts_include_tax');

		return $priceModifier;
	}

/**
 * Populates the price modifier basket with basket items. Modifies the price modifier directly
 *
 * @param  PriceModifier &$priceModifier Price modifier object
 * @param  array $basket Basket Array
 *
 * @return null
 */
	protected function _populatePriceModifierBasket(&$priceModifier, $basket) {
		// Add each basket item to the price modifier
		foreach ($basket['BasketItem'] as $basketItem) {
			$data = [
				'id' => $basketItem['id'],
				'unitPrice' => $basketItem['unit_price'],
				'unitTax' => $basketItem['unit_tax'],
				'quantity' => $basketItem['quantity'],
				'taxRate' => $basketItem['tax_rate'],
				'model' => $basketItem['model'],
				'model_id' => $basketItem['model_id'],
			];

			if (!empty($basketItem['Variant'])) {
				$data['Variant'] = $basketItem['Variant'];
			}

			$priceModifier->basket()->insert($data);
		}
	}

/**
 * Adds the dicount rules from the curent dicount codes
 *
 * @param  PriceModifier &$priceModifier Price modifier object
 * @param  array $basket Basket Array
 *
 * @return null
 */
	protected function _populatePriceModifierDiscounts(&$priceModifier, &$EventSubject, $dataCode) {
		// The cookie isn't used any more but there can still be multiple discount codes added to the basket in the
		// database potentially. Current functionality only allows one discount code at a time though.
		if (!is_object($EventSubject->_controller->BasketManager)) {
			$EventSubject->_controller->BasketManager = $EventSubject->_controller->loadComponent('EvBasket.BasketManager');
		}

		// Get the basket data to find the discount codes
		$codes = array();
		if ($dataCode != null) {
			$codes[$dataCode['DiscountCode']['code']] = $dataCode;
		}

		// Add additional discount codes from basket data
		$basketData = $EventSubject->_controller->BasketManager->getBasketData();
		if (!empty($basketData)) {
			foreach ($basketData as $key => $value) {
				if ($value['name'] == 'discount_code' || $value['name'] == 'voucher_code') {
					$codes[$value['data']] = $value;
				}
			}
		}

		$deliveryTotal = $EventSubject->_controller->BasketManager->findTotalRow(Configure::read('EvBasket.labels.delivery'), true);

		// If the basket data returns some codes or one is passed through (or both) then
		if (!empty($codes) && is_array($codes)) {
			// Checks with the discount module to see what codes are active and not run out etc.
			// Only returns codes that are still valid
			$DiscountCode = EvClassRegistry::init('EvDiscount.DiscountCode');
			$validCodes = $DiscountCode->validateCodes($codes);

			if (!empty($validCodes)) {
				// If there were any valid discount modifiers add to the discount rules for the price modifier.
				foreach ($validCodes as $code => $codeData) {
					// Add discount rule
					if ($codeData['DiscountCode']['is_voucher']) {
						$discountType = 'voucher';
					} else {
						$discountType = $codeData['DiscountType']['name'];
					}

					$priceModifier->addDiscountRule([
						'id' => $codeData['DiscountCode']['is_voucher'] ? ("V_" . $code) : $code,
						'discountType' => $discountType,
						// Used for percentage discounts
						'percent' => $codeData['DiscountCode']['amount'],
						// Used for fixed discounts (ignored by percentage so fine to always pass)
						'amount' => $codeData['DiscountCode']['amount'],
						// Percentage discounts are applied per item and fixed discounts are applied at the end
						'applyToItems' => $codeData['DiscountType']['apply_to_items'],
						// Apply discounts before or after applying tax
						'applyDiscountsAfterTax' => Configure::read('SiteSetting.ev_basket.discounts_include_tax'),
						// Additional information passed through for the price modifiers restrictions
						'min_order' => $codeData['DiscountCode']['min_order'],
						'code_restrictions' => $codeData['CodeRestriction'],
						'extra_data' => !empty($codeData['DiscountCode']['extra_data']) ? json_decode($codeData['DiscountCode']['extra_data'], true) : null,

						'delivery' => !empty($deliveryTotal) ? $deliveryTotal['amount'] : 0,
						'deliveryTax' => !empty($deliveryTotal) ? $deliveryTotal['tax'] : 0,
					]);
				}
			}
		}
	}

/**
 * Adds the discount rules from any active promotions
 *
 * @param  PriceModifier &$priceModifier Price modifier object
 * @param  array $basket Basket Array
 *
 * @return null
 */
	protected function _populatePriceModifierPromotions(&$priceModifier, &$EventSubject, $activePromotions) {
		if (!CakePlugin::loaded('EvPromotion')) {
			// Not using promotions
			return false;
		}

		if (!is_object($EventSubject->_controller->BasketManager)) {
			$EventSubject->_controller->BasketManager = $EventSubject->_controller->loadComponent('EvBasket.BasketManager');
		}

		$deliveryTotal = $EventSubject->_controller->BasketManager->findTotalRow(Configure::read('EvBasket.labels.delivery'), true);

		// If there are any active promotions add the price modifiers
		if (!empty($activePromotions)) {
			// If there were any valid discount modifiers add to the discount rules for the price modifier.
			foreach ($activePromotions as $promotion) {
				$priceModifier->addDiscountRule([
					'id' => 'promotion_' . $promotion['Promotion']['id'],
					'discountType' => $promotion['DiscountType']['name'],
					// Used for percentage discounts
					'percent' => $promotion['Promotion']['amount'],
					// Used for fixed discounts (ignored by percentage so fine to always pass)
					'amount' => $promotion['Promotion']['amount'],
					// Percentage discounts are applied per item and fixed discounts are applied at the end
					'applyToItems' => $promotion['DiscountType']['apply_to_items'],
					// Apply discounts before or after applying tax
					'applyDiscountsAfterTax' => Configure::read('SiteSetting.ev_basket.discounts_include_tax'),
					// Additional information passed through for the price modifiers restrictions
					'min_order' => $promotion['Promotion']['min_order'],
					'code_restrictions' => $promotion['CodeRestriction'],

					'extra_data' => !empty($promotion['Promotion']['extra_data']) ? json_decode($promotion['Promotion']['extra_data'], true) : null,

					'delivery' => !empty($deliveryTotal) ? $deliveryTotal['amount'] : 0,
					'deliveryTax' => !empty($deliveryTotal) ? $deliveryTotal['tax'] : 0,
				]);
			}
		}
	}
}
