<?php

App::uses('CakeTime', 'Utility');

/**
 * Static lib to share code between helper and components
 */
class BasketLib {

	/**
	 * given the basket item array row
	 * return the text to show on the basket
	 *
	 * @param 	array 	$BasketItem 	The basket item row from the DB
	 * @return 	string
	 */
	public static function getItemName($BasketItem) {
		// If a name has been set on the basket item use this.
		if (!empty($BasketItem['name'])) {
			return $BasketItem['name'];
		}

		// Try to find the model name
		$modelName = Hash::get($BasketItem, 'model');
		if (empty($modelName)) {
			return 'Unknown';
		}

		// get the config so we know where to find the data for this row
		$config = Configure::read('EvBasket.BasketItemNamePath');
		if (empty($config[$modelName])) {
			// Attempt to get the basket row text from the model (look for a method called basketRowName)
			$itemModel = EvClassRegistry::init($modelName);
			if (method_exists($itemModel, 'basketRowName')) {
				return $itemModel->basketRowName($BasketItem['model_id']);
			}

			return 'Unknown';
		}
		// reposition array to make it easier to work
		$config = $config[$modelName];

		$separator = ' ';
		if (! empty($config['separator'])) {
			$separator = $config['separator'];
			unset($config['separator']);
		}

		if (! is_array($config)) {
			$config = array($config);
		}

		$item = array();
		foreach ($config as $path) {
			$value = Hash::get($BasketItem, $path);

			if (! empty($value) && stripos(implode('', $item), $value) === false) {
				$item[] = $value;
			}
		}

		if (empty($item)) {
			return 'Unknown';
		}

		return implode($separator, $item);
	}

	/**
	 * clear old baskets by checking grand total row for each one
	 *
	 * @param 	int 	$olderThan 			Delete any baskets older than time() - $olderThan
	 * @return 	int 	$numDeletedItems 	Number of items deleted
	 */
	public static function cleanBaskets($olderThan = WEEK) {
		$deleteDate = CakeTime::toServer(
			(time() - $olderThan),
			Configure::read('Config.timezone')
		);

		$Basket = EvClassRegistry::init('EvBasket.Basket');
		$Basket->displayField = 'id';
		$basketIds = $Basket->find(
			'list',
			[
				'joins' => [
					[
						'table' => 'ev_basket_basket_totals',
						'alias' => 'BasketTotal',
						'conditions' => [
							'BasketTotal.basket_id = Basket.id'
						],
						'type' => 'left'
					],
				],
				'conditions' => [
					'Basket.modified <=' => $deleteDate,
					'OR' => [
						'BasketTotal.id IS NULL',
						'BasketTotal.modified <=' => $deleteDate
					]
				],
				'group' => 'Basket.id'
			]
		);

		$deleted = 0;
		if (! empty($basketIds)) {
			foreach ($basketIds as $id) {
				if ($Basket->delete($id)) {
					$deleted++;
				}
			}
		}

		return $deleted;
	}

	/**
	 * delete a basket
	 *
	 * @param 	int 	$basketId 	The basket to remove
	 * @return 	bool
	 */
	public static function deleteBasket($basketId) {
		$Basket = EvClassRegistry::init('EvBasket.Basket');

		return $Basket->delete($basketId);
	}

/**
 * Queries the basket table to find the abandoned ones that need a gentle email reminder to give us ya money
 * @param  array  $passed - any additional params
 * @return Array
 */
	public static function fetchDropoutBaskets($passed = array()) {
		$Basket = EvClassRegistry::init('EvBasket.Basket');

		$now = new DateTime();
		$now = $now->format("Y-m-d H:i:s");

		// Get the minimum inactivity and minimum time since they last ordered interval values
		$minInactivityTime = Configure::check('EvBasket.basketDropout.minInactivityTime') ? Configure::read('EvBasket.basketDropout.minInactivityTime') : "2 hour";
		$minLastOrderedTime = Configure::check('EvBasket.basketDropout.minLastOrderedTime') ? Configure::read('EvBasket.basketDropout.minLastOrderedTime') : "1 week";

		/*
		 * The following query fetches out basket dropouts which meet the following criteria:
		 *  - Basket has an associated user (with an email address too ofc!)
		 * 	- User hasn't completed the checkout process within the last week (based on their email address)
 		 * 	- They haven't been active on the site for more than 2 hours (basket last_activity)
		 */
		$params = array(
			'conditions' => array(
				'Basket.last_activity IS NOT NULL',
				// Last activity was longer than 2 hours ago
				'Basket.last_activity < DATE_SUB("' . $now . '" , INTERVAL ' . $minInactivityTime . ')',
				'Basket.user_id IS NOT NULL',
				'User.email IS NOT NULL',
				// Not already linked with an order (under a week old)
				'Order.id IS NULL',
				// Haven't emailed them yet for this basket
				'BasketDropout.id IS NULL'
				// 'OR' => array(
				// 	'BasketDropout.id IS NULL',
				// 	'BasketDropout.modified < DATE_SUB(NOW(), INTERVAL 1 week)'
				// )
			),
			'fields' => array(
				'Basket.*',
				'User.*',
				'BasketDropout.*',
				'Order.id'
			),
			'joins' => array(
				array(
					'table' => 'ev_basket_basket_dropouts',
					'alias' => 'BasketDropout',
					'type' => 'LEFT',
					'conditions' => array(
						'BasketDropout.basket_hash = Basket.hash'
					)
				),
				array(
					'table' => 'users',
					'alias' => 'User',
					'conditions' => array(
						'Basket.user_id = User.id',
					)
				),
				array(
					'table' => 'ev_checkout_orders',
					'alias' => 'Order',
					'type' => 'LEFT',
					'conditions' => array(
						'Order.email = User.email',
						// Bring out orders linked to this email less than a week old.
						// So if a user created an order over a week ago and is back with a new basket we still want to email them
						// This will bring null if no orders with this email are found from the past week
						'Order.created > DATE_SUB("' . $now . '", INTERVAL ' . $minLastOrderedTime . ')'
					)
				)
			),
			'group' => 'Basket.id'
		);

		$baskets = $Basket->find('all', array_merge_recursive($params, $passed));

		return $baskets;
	}

/**
 * Builds the url to be put inside the emails that creates the basket and redirects them to the checkout screen
 * @param  $dropout - dropout data
 * @return String
 */
	public static function buildDropoutUrl($dropout) {
		$domain = Configure::read('App.fullBaseUrl');

		return $domain . '/ev_basket/basket_dropouts/view/' . $dropout['BasketDropout']['token'] . '?utm_source=basketdropout&utm_medium=email';
	}

/**
 * Find the basket item key from a basket item.
 *
 * @param string|int|array $key Either Basket row ID / compiled basket key string / Array with two elements of model and model_id
 * @param array $basket The basket that contains the basket item if the key array is not passed in.
 * @return string The item key.
 */
	public static function getItemKey($key, $basket) {
		// set the key as the itemKey as a fallback
		$itemKey = $key;

		if (is_array($key)) {
			// It's an array, try and build it
			$itemKey = self::buildItemKey($key);
		} elseif (is_numeric($key) && ! empty($basket)) {
			// it's and id number - try to find the itemKey
			$keysArray = Hash::combine(
				$basket,
				'BasketItem.{s}.id',
				self::_getItemKeyCombination()
			);

			if (empty($keysArray[$key])) {
				return false;
			}

			$itemKey = $keysArray[$key];
		}

		return $itemKey;
	}

/**
 * Get the combination used to find item keys in the basket.
 *
 * @return array The combination of fields used for the basket item key.
 */
	protected static function _getItemKeyCombination() {
		$itemKeyFields = Configure::read('EvBasket.BasketItemKeyFields');

		if (empty($itemKeyFields)) {
			return [];
		}

		//The first array element needs to be a string of %s for each field separated by "."
		$combination = [
			implode('.', array_fill(0, count($itemKeyFields), '%s')),
		];

		foreach ($itemKeyFields as $field) {
			$combination[] = 'BasketItem.{s}.' . $field;
		}

		return $combination;
	}

/**
 * Build the basket item key
 *
 * @param array $item Array with two elements of model and model_id
 * @return string|bool $key Item basket key Model.Model_id format / false on fail
 */
	public static function buildItemKey($item) {
		$itemKeyFields = Configure::read('EvBasket.BasketItemKeyFields');

		if (empty($itemKeyFields)) {
			return false;
		}

		$itemKey = [];
		foreach ($itemKeyFields as $field) {
			if (empty($item[$field])) {
				return false;
			}

			$itemKey[] = $item[$field];
		}

		return implode('.', $itemKey);
	}

/**
 * Find a specified set of data from the basket data using the name of the data.
 *
 * @param array  $basket The basket to find basket data in.
 * @param string $name   The name of the data to find.
 * @return string|array The found data. False if not found.
 */
	public static function findBasketData($basket, $name) {
		if (!empty($basket['BasketData'])) {
			return BasketLib::getBasketData($basket['BasketData'], $name);
		}

		return false;
	}

/**
 * Get a specific item of basket data from the basket data array.
 *
 * @param array $basketData The basket data array as returned as part of an basket.
 * @param array $dataName   The name of the data you want to get.
 * @return string|array The data if found, false if not found and null if no basket data was provided.
 */
	public static function getBasketData($basketData, $dataName) {
		if (empty($basketData) || empty($dataName)) {
			return null;
		}

		return ArrayUtil::findFirst($basketData, 'name', $dataName, 'data');
	}

/**
 * Find a specified set of data from the basket item data using the name of the data.
 *
 * @param array  $basket The basket to find basket item data in.
 * @param array  $item   The item to find basketItemData in.
 * @param string $name   The name of the data to get basket item data of.
 * @return string|array The found data. False if not found.
 */
	public static function findBasketItemData($basket, $item, $name) {
		$itemKey = self::buildItemKey($item);

		if (!empty($basket['BasketItem'][$itemKey]['BasketItemData'])) {
			return self::getBasketItemData($basket['BasketItem'][$itemKey]['BasketItemData'], $name);
		}

		return false;
	}

/**
 * Get a specific item of basket item data from the basket item data array.
 *
 * @param array $basketItemData The basket data array as returned as part of an basket.
 * @param array $itemDataName   The name of the data you want to get.
 * @return string|array The data if found, false if not found and null if no basket item data was provided
 */
	public static function getBasketItemData($basketItemData, $itemDataName) {
		if (empty($basketItemData) || empty($itemDataName)) {
			return null;
		}

		return ArrayUtil::findFirst($basketItemData, 'name', $itemDataName, 'item_data');
	}

}
