<?php

App::uses('AppComponent', 'Controller/Component');
App::uses('CakeText', 'Utility');
App::uses('BasketLib', 'EvBasket.Lib');

class BasketComponent extends AppComponent {

/**
 * the users hash
 */
	public $hash = null;

/**
 * copy of the basket
 */
	public $basketCache = null;

/**
 * Basket model - set via Component Settings
 */
	public $Basket = null;

/**
 * Basket Item model - set via Component Settings
 */
	public $BasketItem = null;

/**
 * Basket Total model - set via Component Settings
 */
	public $BasketTotal = null;

/**
 * Basket Data model - set via Component Settings
 */
	public $BasketData = null;

/**
 * Basket Item Data model - set via Component Settings
 */
	public $BasketItemData = null;

/**
 * setup the cookie component
 *
 */
	public function initialize(Controller $controller) {
		parent::initialize($controller);

		if (empty($this->_controller->BasketCookie)) {
			$this->_controller->BasketCookie = $this->_controller->loadComponent('EvBasket.BasketCookie');
		}

		if (empty($this->Basket)) {
			$this->Basket = EvClassRegistry::init('EvBasket.Basket');
		}

		if (empty($this->BasketItem)) {
			$this->BasketItem = EvClassRegistry::init('EvBasket.BasketItem');
		}

		if (empty($this->BasketTotal)) {
			$this->BasketTotal = EvClassRegistry::init('EvBasket.BasketTotal');
		}

		if (empty($this->BasketData)) {
			$this->BasketData = EvClassRegistry::init('EvBasket.BasketData');
		}

		if (empty($this->BasketItemData)) {
			$this->BasketItemData = EvClassRegistry::init('EvBasket.BasketItemData');
		}
	}

/**
 * setup a hash or retrieve a hash from the cookie
 * @param bool	$readOnly	when true, don't attempt to create a new hash
 * @return string  	Basket hash identifier
 */
	public function setupHash($readOnly = false) {
		if (! empty($this->hash)) {
			return $this->hash;
		}

		$editOrderComponentName = Configure::read('EvBasket.editOrderComponent');
		if (!empty($editOrderComponentName)) {
			// Check for an overridden basket hash if we're editing an order
			$editOrderComponent = $this->_controller->loadComponent($editOrderComponentName);
			$editOrder = $editOrderComponent->checkForEditOrder();
			if (!empty($editOrder)) {
				return $this->hash = $editOrder['basket_hash'];
			}
		}

		$cookieHash = $this->_controller->BasketCookie->read('basketHash');

		$userHash = $this->getBasketHashFromUser();

		// we have a cookie hash and a separate user hash that don't match
		if (! empty($cookieHash)) {
			$associateBasketWithUserEnabled = Configure::read('EvBasket.associateBasketWithUser');
			if ($associateBasketWithUserEnabled === true) {
				$authUserId = $this->_controller->Auth->user('User.id');

				if (! empty($authUserId) && $userHash === false) {
					// associate the current cookie hash with the logged in user
					if ($this->Basket->assignUserToBasket($cookieHash, $authUserId)) {
						// User was successfully assigned to basket
						$this->hash = $cookieHash;
					} else {
						// Could not assign the user to the basket, there was something wrong so
						// need to give the user a fresh basket, this could happen if two users
						// are sharing a computer and cookies have leaked over
						$this->hash = false; // Setting to false creates one further down
					}
				} elseif ($userHash !== false && $userHash !== $cookieHash) {
					// there is a user basket hash that doesn't match the cookie one so we need to merge
					// the baskets, deletes the basket entry not associated with the user
					$this->hash = $this->Basket->mergeBaskets($userHash, $cookieHash, true);
				} else {
					$this->hash = $cookieHash;
				}
			} else {
				// Users cannot be associated with baskets so always use the Cookie hash if set
				$this->hash = $cookieHash;
			}
		} elseif ($userHash !== false) {
			// no cookie hash is assigned but we do have a hash for the user
			$this->hash = $userHash;
		} elseif ($readOnly) {
			// We don't have any sort of basket but don't want to create one either
			return false;
		}

		// We always need a basket Hash so if one of the bits above has
		// dropped out we ensure that we have a new basket if the hash isn't
		// valid
		if (empty($this->hash)) {
			// Generate a new basket
			$this->hash = $this->generateHash();
			$this->setupBasket($this->hash);
		}

		$this->_controller->BasketCookie->write(
			'basketHash',
			$this->hash
		);

		return $this->hash;
	}

/**
 * Checks the logged in user to see if they have a basket hash set.
 * @return  mixed If there is no user logged in it will return false. If there is a hash stored against the user then the hash is returned.
 */
	public function getBasketHashFromUser() {
		// If config is not set to associate users with baskets always return false without checking
		if (Configure::read('EvBasket.associateBasketWithUser') == false) {
			return false;
		}

		$userId = $this->_controller->Auth->user('User.id');

		// No user so no user basket
		if (!$userId) {
			return false;
		}

		$basket = $this->Basket->find('first', [
			'fields' => 'hash',
			'conditions' => [
				'user_id' => $userId
			],
			'order' => [
				'created DESC'
			]
		]);

		if ($basket) {
			return $basket['Basket']['hash'];
		}

		return false;
	}

/**
 * given the hash, setup or return the existing basket
 *
 * @param 	string 	$hash 	Basket Hash
 * @return 	bool 	$result True if a basket was created, false if one already exists
 */
	public function setupBasket($hash) {
		if (! $this->Basket->checkExists($hash)) {
			// Basket doesn't already exist so lets create one
			if (Configure::read('EvBasket.associateBasketWithUser') && $this->_controller->Auth->user()) {
				$userId = $this->_controller->Auth->user('User.id');
			} else {
				$userId = null;
			}

			return $this->Basket->createBasket(
				$hash,
				CakeSession::read('EvCurrency.currencyId'),
				$userId
			);
		}

		return false;
	}

/**
 * build the basket item contain list
 *
 * @param 	array  	$itemContain 	Value from Configure::read('EvBasket.BasketItemContains')
 * @param 	array  	$relationships 	Value from Configure::read('EvBasket.BasketItemBelongsTo')
 * @return 	array
 */
	public function buildItemContain($itemContain, $relationships) {
		return $this->BasketItem->buildItemContain($itemContain, $relationships);
	}

/**
 * generate a hash for the basket
 * extracted into another method for easier unit testing
 *
 * @return string
 */
	public function generateHash() {
		return CakeText::uuid() . '-' . md5($this->_controller->request->clientIp() . rand());
	}

/**
 * Find the basket item key from a basket item.
 *
 * @param string|int|array $key Either Basket row ID / compiled basket key string / Array with two elements of model and model_id
 * @return string The item key.
 */
	public function getItemKey($key) {
		return BasketLib::getItemKey($key, $this->basketCache);
	}

/**
 * Build the basket item key
 *
 * @param array $item Array with two elements of model and model_id
 * @return string|bool $key Item basket key Model.Model_id format / false on fail
 */
	public function buildItemKey($item) {
		return BasketLib::buildItemKey($item);
	}

/**
 * given the item key, check to see if it exists already in the basket
 *
 * @param 	string 		$key 		Item basket key Model.Model_id format
 * @param 	array|null 	$basket 	Optional basket to search in - if null will use $this->basket
 * @return 	bool
 */
	public function basketItemExists($key, $basket = null) {
		if (! empty($basket)) {
			return ! empty($basket['BasketItem'][$key]);
		}

		return ! empty($this->basketCache['BasketItem'][$key]);
	}

/**
 * common method to call out to rebuild the totals
 *
 * @todo Refactor to make it more testable
 * @return bool | array Basket cache
 */
	public function rebuildTotals() {
		$this->getBasket(true);

		// process subtotal
		// process Tax
		$subtotal = 0;
		$tax = 0;
		$BasketItems = $this->basketCache['BasketItem'];

		foreach ($BasketItems as $BasketItem) {
			$subtotal += number_format($BasketItem['row_total'], 2, '.', '');
			$tax += number_format($BasketItem['row_tax_amount'], 2, '.', '');
		}

		//Get subtotal + tax now before tax is changed
		$subtotalIncTax = $subtotal + $tax;

		//Check for discount rows and amend tax for each.
		$tax -= $this->calculateDiscountTaxAmount($subtotal);

		// check for delivery placeholder, only add if we haven't added delivery already
		$deliveryPlaceholder = Configure::read('EvBasket.deliveryPlaceholder');
		$delivery = $this->findTotalRow(Configure::read('EvBasket.labels.delivery'));

		if (! isset($deliveryAmount) && isset($delivery['amount'])) {
			$deliveryAmount = $delivery['amount'];
		}

		// check for delivery taxation
		$delTax = 0;
		$deliveryTax = Configure::read('SiteSetting.ev_basket.delivery_tax_rate');
		if ($deliveryTax !== false && $deliveryTax > 0 && isset($deliveryAmount)) {
			$delTax = (($deliveryTax / 100) * $deliveryAmount);
			$tax += $delTax;
		}

		//If a placeholder exists and the delivery isn't set then set the delivery charge to the placeholder
		if ($deliveryPlaceholder !== false && empty($delivery['id'])) {
			// add / update the delivery row
			$this->manageTotalRow(
				Configure::read('EvBasket.labels.delivery'),
				$deliveryPlaceholder,
				20,
				false,
				$deliveryPlaceholder,
				$deliveryPlaceholder + $delTax
			);

			$deliveryAmount = $deliveryPlaceholder;
		}

		// if at this point tax has worked out to be negative - zero it
		if ($tax < 0) {
			$tax = 0;
		}

		// add / update the subtotal row
		$this->manageTotalRow(
			Configure::read('EvBasket.labels.subtotal'),
			$subtotal,
			10,
			true,
			$subtotal,
			$subtotalIncTax
		);

		// add / update the tax row
		$this->manageTotalRow(
			Configure::read('EvBasket.labels.tax'),
			$tax,
			30,
			false
		);

		// call out to recompile all the basket totals
		$totalResult = $this->BasketTotal->recompileTotals($this->basketCache['Basket']['id']);

		if ($totalResult === true) {
			// dispatch totals recompiled
			$this->_controller->getEventManager()->dispatch(
				new CakeEvent('EvBasket.Component.Basket.postTotalRecompile', $this, [
					'basketId' => $this->basketCache['Basket']['id']
				])
			);

			// rebuild the basket cache
			return $this->getBasket(true);
		}

		return false;
	}

/**
 * Calculate how much to reduce the basket tax by based on the discount totals currently in the basket.
 *
 * @param float $subtotal The current subtotal value of the basket.
 * @return float The amount of tax that has been discounted in the basket.
 */
	public function calculateDiscountTaxAmount($subtotal) {
		$taxReduction = 0;

		$discountRows = Configure::read('EvBasket.discountLabels');
		if (!empty($discountRows)) {
			foreach ($discountRows as $discount) {
				$discountRow = $this->findTotalRow(Configure::read('EvBasket.labels.' . $discount));
				if (!empty($discountRow)) {
					//Ensure all discount values are positive for discount calculations, returned to negative when saving into the basket
					$discountRow['amount'] = abs($discountRow['amount']);
					$discountRow['tax'] = abs($discountRow['tax']);
					$discountRow['display_inc_tax'] = abs($discountRow['display_inc_tax']);
					$discountRow['display_ex_tax'] = abs($discountRow['display_ex_tax']);

					//Update the tax to have the discounted reduction.
					if (
						(float)$subtotal > 0 &&
						! empty($discountRow['amount']) &&
						(float)$discountRow['amount'] > 0
					) {
						$this->manageTotalRow(
							Configure::read('EvBasket.labels.' . $discount),
							-$discountRow['amount'],
							$discountRow['sequence'],
							$discountRow['display_only'],
							-$discountRow['display_ex_tax'],
							-$discountRow['display_inc_tax']
						);

						$taxReduction += $discountRow['tax'];
					}
				}
			}
		}

		return $taxReduction;
	}

/**
 * if we're updating an item - check if we actually want to update multi rows in one go
 *
 * @param 	array 		$item 		Array with two elements of model and model_id / or multidimensional array of model, model_id, quantity, unitPrice
 * @param 	int 		$quantity	Amount of items to add
 * @param 	decimal 	$unitPrice 	Unit price of the item
 * @param 	decimal		$taxRate 	The tax rate to update
 * @return  array 		$toUpdate 	Array of items to update
 */
	public function buildUpdateList($item, $quantity, $unitPrice, $taxRate) {
		if (is_numeric(key($item))) {
			return $item;
		}

		if ($quantity < 0) {
			$quantity = 0;
		}

		$item['quantity'] = $quantity;
		$item['unitPrice'] = $unitPrice;
		$item['taxRate'] = $taxRate;

		return [
			$item
		];
	}

/**
 * when adding an item, check we have all the data needed
 *
 * @param 	array 	$basketItem 	Array of data - array of model, model_id, quantity, unitPrice
 * @return 	bool
 */
	public function checkAddItem($item) {
		$Model = EvClassRegistry::init($item['model']);

		if (
			! empty($item['model']) &&
			! empty($item['model_id']) &&
			! empty($item['quantity']) &&
			$item['quantity'] > 0 &&
			$this->getUnitPriceMethodName($Model) !== false
		) {
			// add in any potential existing item quantities
			$check = $item;
			$basket = $this->getBasket();
			$basketItemKey = $item['model'] . '.' . $item['model_id'];

			if (!empty($basket['BasketItem'][$basketItemKey])) {
				$check['quantity'] += $basket['BasketItem'][$basketItemKey]['quantity'];
			}

			// check we have enough stock
			if (! $this->hasEnoughStock(EvClassRegistry::init($item['model']), $check)) {
				$this->_controller->Flash->fail(
					Configure::read('EvInventory.notEnoughStockMessage')
				);
				return false;
			}

			return $item;
		}

		return false;
	}

/**
 * Check each item that is being added to the basket. If an item can't be added to the basket then it is
 * removed from the items being added and a flash message provided to warn a user that the item hasn't
 * been added.
 *
 * @param array $items An array of basket items to add to the basket.
 * @return array The items that passed the check.
 */
	public function checkAddItems($items) {
		$passedItems = []; //Items that pass the check and can be added to the basket.
		$failedItems = []; //Items that haven't passsed the check and won't be added to the basket.

		foreach ($items as $item) {
			$checkResult = $this->checkAddItem($item);

			if ($checkResult) {
				$passedItems[] = $item;
			} else {
				$failedItems[] = $item;
			}
		}

		return ['passed' => $passedItems, 'failed' => $failedItems];
	}

/**
 * check to see if the item has enough stock (assuming inventories are enabled)
 *
 * @param 	object 	$Model 	The model object of the item we are checking
 * @param 	array 	$item 	The basket item
 * @return 	bool
 */
	public function hasEnoughStock($Model, $item) {
		if (CakePlugin::loaded('EvInventory') && $Model->hasBehavior('EvInventory.Inventories')) {
			App::uses('InventoryLib', 'EvInventory.Lib');

			if (! isset($item['BasketItem'])) {
				$item['BasketItem'] = $item;
			}

			if (! InventoryLib::hasEnoughStock($Model->readForEdit($item['BasketItem']['model_id']), $item['BasketItem']['quantity'])) {
				return false;
			}
		}

		return true;
	}

/**
 * find a total row given the name
 *
 * @param string $totalName Row name for the total
 * @param bool   $rebuild   Rebuild the basket when it is being acquired.
 * @return array|bool $BasketTotal BasketTotal array row
 */
	public function findTotalRow($name, $rebuild = false) {
		$this->getBasket($rebuild);

		if (isset($this->basketCache['BasketTotal'])) {
			$total = ArrayUtil::findFirst($this->basketCache['BasketTotal'], 'name', $name);
			if (!empty($total)) {
				return $total;
			}
		}

		return false;
	}

/**
 * Find a specified set of data from the basket data using the name of the data.
 *
 * @param string $name    The name of the data to find
 * @param bool   $rebuild Rebuild the basket when it is being acquired.
 * @return string|array The found data. False if not found.
 */
	public function findBasketData($name, $rebuild = false) {
		$this->getBasket($rebuild);

		return BasketLib::findBasketData($this->basketCache, $name);
	}

/**
 * Find a specified set of data from the basket item data using the name of the data.
 *
 * @param array  $item    The item to find basketItemData in.
 * @param string $name    The name of the data to get basket item data of.
 * @param bool   $rebuild Rebuild the basket when it is being acquired.
 * @return string|array The found data. False if not found.
 */
	public function findBasketItemData($item, $name, $rebuild = false) {
		$this->getBasket($rebuild);

		return BasketLib::findBasketItemData($this->basketCache, $item, $name);
	}

/**
 * reset a basket - clear the hash and basket
 *
 * @return 	bool
 */
	public function resetBasket() {
		$this->hash = null;
		$this->basketCache = null;

		return true;
	}

/**
 * get a single basket item
 *
 * @param 	string|int|array 	$key 	Either Basket row ID / compiled basket key string / Array with two elements of model and model_id
 * @return  array|bool 			$item 	Item array or bool false
 */
	public function getBasketItem($key) {
		$this->getBasket();

		// set the key as the itemKey as a fallback
		$itemKey = $this->getItemKey($key);

		if ($this->basketItemExists($itemKey)) {
			return $this->basketCache['BasketItem'][$itemKey];
		}

		return false;
	}

/**
 * Get the basket from the current session and save it into the cache. If a basket already exists
 * in the cache then just return that.
 *
 * @param bool $rebuild  Whether to force a rebuild of the basket or not.
 * @param bool $readOnly Read basket if there is one but don't create a new one.
 * @return array $basket Basket Data.
 */
	public function getBasket($rebuild = false, $readOnly = false) {
		if (empty($this->basketCache) || $rebuild) {
			// pass readOnly flag when setting up the basket hash
			$this->setupHash($readOnly);

			if (empty($this->hash)) {
				return false;
			}

			$this->setupBasket($this->hash);
			$this->basketCache = $this->_getFullBasket($this->hash);
		}

		return $this->basketCache;
	}

/**
 * Get the full basket from the basket hash.
 *
 * @param string $basketHash The has of the basket to get.
 * @return array The found basket.
 */
	protected function _getFullBasket($basketHash) {
		$itemContain = $this->BasketItem->buildItemContain();

		return $this->Basket->getFullBasket(
			$basketHash,
			$itemContain
		);
	}

/**
 * Get the full basket from the basket id.
 *
 * @param int $basketId The id of the basket to get.
 * @return array Basket data.
 */
	public function getFullBasketByBasketId($basketId) {
		$basket = $this->Basket->find(
			'first',
			[
				'fields' => [
					'Basket.hash',
				],
				'conditions' => [
					'Basket.id' => $basketId,
				],
			]
		);

		if (!empty($basket)) {
			return $this->_getFullBasket($basket['Basket']['hash']);
		}

		return false;
	}

/**
 * Process new basket items to set up basket item data.
 *
 * @param array $basketItems Basket
 * @return array Array with the new basket items processed.
 */
	public function processNewItem($basketItems) {
		//If only a single item has be passed through then turn it into a multi-dimensional array.
		if (!is_numeric(key($basketItems))) {
			$basketItems = [$basketItems];
		}

		$items = [];
		foreach ($basketItems as $basketItem) {
			$item = [
				'model' => $basketItem['model'],
				'model_id' => $basketItem['model_id']
			];
			$item['quantity'] = $basketItem['quantity'];

			$Model = EvClassRegistry::init($item['model']);
			$item['unitPrice'] = $this->getUnitPriceMethod($Model, $basketItem);

			// if there's a method to get the taxRate, do so
			$taxRate = 0;
			if (method_exists($Model, 'getTaxRate')) {
				$taxRate = $Model->getTaxRate($item['model_id']);
			}
			$item['taxRate'] = $taxRate;

			$itemData = [];
			if (! empty($basketItem['BasketItemData'])) {
				$itemData = $basketItem['BasketItemData'];
			}
			$item['BasketItemData'] = $itemData;

			$item = $this->_processNewSingleItem($basketItem, $Model, $item);

			if ($item['unitPrice'] !== false) {
				$items[] = $item;
			}
		}

		return $items;
	}

/**
 * Modify or add data to a single item while it is being processed to be added to the basket.
 *
 * @param array $basketItem The basket item data from the request.
 * @param Model $Model      The model of the basket item being added.
 * @param array $newItem    The new processed basket item.
 * @return array The modified/formatted new basket item.
 */
	protected function _processNewSingleItem($basketItem, $Model, $newItem) {
		return $newItem;
	}

/**
 * Add an item to the basket.
 *
 * @param array   $items         Array with two elements of model and model_id / or multidimensional array of model, model_id, quantity, unitPrice, taxRate
 * @param decimal $unitPrice     Unit price of the item
 * @param int     $quantity      Amount of items to add
 * @param decimal $taxRate       The tax rate to add
 * @param bool    $dispatchEvent Whether to disaptch the itemAdd event - used to prevent loops
 * @return array|bool The basket if the item was added successfully, false otherwise.
 */
	public function addItem($items, $unitPrice = 0, $quantity = 1, $taxRate = 0, $dispatchEvent = true) {
		$this->getBasket();

		$toUpdate = $this->buildUpdateList($items, $quantity, $unitPrice, $taxRate);

		$result = false;
		$updated = [];

		foreach ($toUpdate as $item) {
			$addedItem = $this->_addSingleItem($item, $unitPrice, $quantity, $taxRate, $dispatchEvent);

			if (!empty($addedItem)) {
				$result = true;
				$updatedItems[] = $addedItem;
			}
		}

		if ($result === true) {
			//Rebuild totals before event fired to make sure that the totals are up to date before
			// any changes are made.
			$returnTotals = $this->rebuildTotals();
			if ($dispatchEvent) {
				// dispatch item added to basket event
				$this->_controller->getEventManager()->dispatch(
					new CakeEvent('EvBasket.Component.Basket.itemAdd', $this, [
						'basketId' => $this->basketCache['Basket']['id'],
						'items' => $updatedItems
					])
				);
			}

			return $returnTotals;
		}

		return false;
	}

/**
 * Add a single item to the basket.
 *
 * @param array   $item          Array with two elements of model and model_id / or multidimensional array of model, model_id, quantity, unitPrice, taxRate
 * @param decimal $unitPrice     Unit price of the item
 * @param int     $quantity      Amount of items to add
 * @param decimal $taxRate       The tax rate to add
 * @param bool    $dispatchEvent Whether to disaptch the itemAdd event - used to prevent loops
 * @return array|bool The item that was added to the basket or false if it failed to be added/updated.
 */
	protected function _addSingleItem($item, $unitPrice, $quantity, $taxRate, $dispatchEvent) {
		// Reformat the basket item data, if there is any.
		$item = $this->_formatAddedItemData($item);

		$item = $this->_addSingleItemExt($item, $unitPrice, $quantity, $taxRate, $dispatchEvent);

		$itemKey = $this->buildItemKey($item);

		// check if we are updating an existing item or creating a new row
		if ($this->basketItemExists($itemKey)) {
			$itemAdded = $this->BasketItem->updateItem(
				$this->basketCache['BasketItem'][$itemKey],
				$item,
				($this->basketCache['BasketItem'][$itemKey]['quantity'] + $item['quantity']),
				$item['unitPrice'],
				$item['taxRate'],
				(isset($item['unitTax'])) ? $item['unitTax'] : null
			);

			//The item already existed so was updated. Mark the item as "updated" so that
			//it can be dealt with accordingly in the success event.
			$item['updated'] = true;
		} else {
			$itemAdded = $this->BasketItem->addItem(
				$this->basketCache['Basket']['id'],
				$item,
				$item['unitPrice'],
				$item['quantity'],
				$item['taxRate']
			);

			//The item didn't exist so was added instead. Mark the item as "added" so that
			//it can be dealt with accordingly in the success event.
			$item['added'] = true;
		}

		if ($itemAdded === false) {
			return false;
		}

		return $item;
	}

/**
 * Format item data that is being added with an item to the basket.
 *
 * @param array $item The basket item being added to the basket.
 * @return array The item with formatted basket item data.
 */
	protected function _formatAddedItemData($item) {
		if (!empty($item['BasketItemData'])) {
			foreach ($item['BasketItemData'] as $key => $itemData) {
				if (is_array($itemData)) {
					// Reformat to a linebreak list.
					$item['BasketItemData'][$key] = implode("\r\n", $itemData);
				}
			}
		}

		return $item;
	}

/**
 * Modify a single item before it is added to the basket. Extend this method if you want to add additional data or
 * modify the current data before the item key is calculated and the item is added or updated.
 *
 * @param array   $item          Array with two elements of model and model_id / or multidimensional array of model, model_id, quantity, unitPrice, taxRate
 * @param decimal $unitPrice     Unit price of the item
 * @param int     $quantity      Amount of items to add
 * @param decimal $taxRate       The tax rate to add
 * @param bool    $dispatchEvent Whether to disaptch the itemAdd event - used to prevent loops
 * @return array Modified item array.
 */
	protected function _addSingleItemExt($item, $unitPrice, $quantity, $taxRate, $dispatchEvent) {
		return $item;
	}
/**
 * Update an item in the basket
 *
 * @param array   $items         Array with two elements of model and model_id / or multidimensional array of model, model_id, quantity, unitPrice, taxRate
 * @param int     $quantity      Amount of items to add
 * @param decimal $unitPrice     Unit price of the item
 * @param decimal $taxRate       The tax rate to update
 * @param bool    $dispatchEvent Whether to disaptch the itemAdd event - used to prevent loops
 * @return bool
 */
	public function updateItem($items, $quantity = null, $unitPrice = null, $taxRate = 0, $dispatchEvent = true) {
		$this->getBasket();

		$toUpdate = $this->buildUpdateList($items, $quantity, $unitPrice, $taxRate);

		$result = false;
		$updated = [];

		foreach ($toUpdate as $item) {

			$itemKey = $this->buildItemKey($item);

			// check if the row actually exists
			if (! $this->basketItemExists($itemKey)) {
				continue;
			}

			// check if we're wanting to delete
			if ($item['quantity'] <= 0) {
				$result = $this->deleteItem($item);

				//deleteItem returns basket so check that it still exists before marking successful.
				if (!empty($result)) {
					$result = true;
					$updatedItems[] = $item;
				}
				continue;
			}

			// check we have enough stock
			if (! $this->hasEnoughStock(EvClassRegistry::init($item['model']), $item)) {
				$this->_controller->Flash->fail(
					Configure::read('EvInventory.notEnoughStockMessage')
				);
				continue;
			}

			$updateItem = $this->BasketItem->updateItem(
				$this->basketCache['BasketItem'][$itemKey],
				$item,
				$item['quantity'],
				(isset($item['unitPrice'])) ? $item['unitPrice'] : null,
				(isset($item['taxRate'])) ? $item['taxRate'] : null,
				(isset($item['unitTax'])) ? $item['unitTax'] : null
			);

			if ($updateItem === true) {
				$result = true;
				$updatedItems[] = $item;
			}
		}

		if ($result === true) {

			//Rebuild totals before event fired to make sure that the totals are up to date before
			// any changes are made.
			$returnTotals = $this->rebuildTotals();
			if ($dispatchEvent) {
				// dispatch item added to basket event
				$this->_controller->getEventManager()->dispatch(
					new CakeEvent('EvBasket.Component.Basket.itemUpdated', $this, [
						'basketId' => $this->basketCache['Basket']['id'],
						'items' => $updatedItems
					])
				);
			}

			return $returnTotals;
		}

		return false;
	}

/**
 * delete and item from the basket
 *
 * @param 	array 		$item 		Array with two elements of model and model_id
 * @param 	bool 		$dispatchEvent 	Whether to disaptch the itemAdd event - used to prevent loops
 * @return 	bool
 */
	public function deleteItem($item, $dispatchEvent = true) {
		$this->getBasket();

		$itemKey = $this->buildItemKey($item);

		// check if the row actually exists
		if (! $this->basketItemExists($itemKey)) {
			return false;
		}

		$result = $this->BasketItem->deleteItem(
			$this->basketCache['BasketItem'][$itemKey]
		);

		if ($result === true) {

			//Rebuild totals before event fired to make sure that the totals are up to date before
			// any changes are made.
			$returnTotals = $this->rebuildTotals();
			if ($dispatchEvent) {
				// dispatch item added to basket event
				$this->_controller->getEventManager()->dispatch(
					new CakeEvent('EvBasket.Component.Basket.itemDeleted', $this, [
						'basketId' => $this->basketCache['Basket']['id'],
						'itemKey' => $itemKey,
						'BasketItem' => $this->BasketItem
					])
				);
			}

			return $returnTotals;
		}

		return false;
	}

	public function deleteTotalRow($name) {
		$this->getBasket();

		if (empty($name)) {
			return false;
		}

		$totalRow = $this->findTotalRow($name);

		$basketTotalId = null;
		if ($totalRow !== false && ! empty($totalRow['id'])) {
			$basketTotalId = $totalRow['id'];
		}

		return $this->BasketTotal->deleteTotalRow(
			$basketTotalId
		);
	}

/**
 * manage a total row
 *
 * @param 	string 	$name 			The name label for the row
 * @param 	decimal $amount 		The amount to add
 * @param 	int 	$sequence 		The sequence order number
 * @param 	bool 	$displayOnly 	Whether it's a displayOnly row (not added n grand total calcs)
 * @param 	bool 	$amountExTax 	The amount not including tax to display
 * @param 	bool 	$amountIncTax 	The amount including tax to display
 * @param   decimal	$tax 			Optional field for adding tax to a row total instead of calculating on the fly
 * @return 	bool
 */
	public function manageTotalRow($name, $amount, $sequence = null, $displayOnly = false, $amountExTax = null, $amountIncTax = null, $dispatchEvent = true) {
		$this->getBasket();

		if (empty($name)) {
			return false;
		}

		$totalRow = $this->findTotalRow($name);

		$data = [
			'name' => $name,
			'amount' => $amount,
			'display_only' => $displayOnly,
			'display_ex_tax' => $amountExTax,
			'display_inc_tax' => $amountIncTax,
			'tax' => $amountIncTax - $amountExTax
		];

		if (!is_null($amountExTax) && !empty($amountExTax)) {
			$data['display_ex_tax'] = $amountExTax;
		}

		if (!is_null($amountIncTax) && !empty($amountIncTax)) {
			$data['display_in_tax'] = $amountIncTax;
		}

		if (is_null($sequence) && ! empty($totalRow['sequence'])) {
			$data['sequence'] = $totalRow['sequence'];
		} elseif (! empty($sequence)) {
			$data['sequence'] = $sequence;
		}

		$basketTotalId = null;
		if ($totalRow !== false && ! empty($totalRow['id'])) {
			$basketTotalId = $totalRow['id'];
		}

		$returnResult = $this->BasketTotal->manageTotalRow(
			$this->basketCache['Basket']['id'],
			$data,
			$basketTotalId
		);

		if ($dispatchEvent) {
			// dispatch item added to basket event
			$this->_controller->getEventManager()->dispatch(
				new CakeEvent('EvBasket.Component.Basket.totalRowUpdated', $this, [
					'basketId' => $this->basketCache['Basket']['id'],
					'data' => $data
				])
			);
		}

		return $returnResult;
	}

/**
 * get basket summary for headers / sidebar
 * @param 	bool 	$readOnly	Read basket if there is one but don't create a new one
 * @return 	array 	$summary 	summary of amount and no. items
 */
	public function getBasketSummary($readOnly = true) {
		$this->getBasket(false, $readOnly);

		$basket = $this->_modifyBasketForSummary($this->basketCache);

		$numberOfItems = 0;
		$total = 0;

		if (isset($basket['BasketItem']) && ! empty($basket['BasketItem'])) {
			foreach ($basket['BasketItem'] as $BasketItem) {
				$numberOfItems += $BasketItem['quantity'];
				$total += $BasketItem['row_total'];
			}
		}

		return [
			'total' => $total,
			'items' => $numberOfItems
		];
	}

/**
 * Modify the basket before it is summarised.
 *
 * @param array $basket The unmodified basket, fresh from the cache
 * @return array The modified basket.
 */
	protected function _modifyBasketForSummary($basket) {
		return $basket;
	}

/**
 * Build transaction items from basket.
 *
 * @return array An array of transaction items built from the basket items.
 */
	public function buildTransactionItems() {
		$this->getBasket();

		$items = [];
		foreach ($this->basketCache['BasketItem'] as $BasketItem) {
			$item = [
				'model' => $BasketItem['model'],
				'model_id' => $BasketItem['model_id'],
				'description' => BasketLib::getItemName($BasketItem),
				'amount' => $BasketItem['row_total'],
				'quantity' => $BasketItem['quantity'],
				'unitPrice' => $BasketItem['unit_price'],
			];

			$item = $this->_buildTransactionItem($this->basketCache, $BasketItem, $item);

			$items[] = $item;
		}

		return $items;
	}

/**
 * Build a transaction item from a basket item. Extend this method to modify/format the transaction item.
 *
 * @param array $basket          The current basket cache.
 * @param array $basketItem      The current basket item that is being built into a transaction item.
 * @param array $transactionItem The transaction item being built.
 * @return array The modified/formatted transaction item.
 */
	protected function _buildTransactionItem($basket, $basketItem, $transactionItem) {
		return $transactionItem;
	}

/**
 * Build the transaction extras array from provided addresses, user and additional data.
 * The totals are found and added to the extras as well.
 * @param  array $addresses The delivery and billing addresses to be used for this transaction
 * @param  array $user      The current user to be used for this transaction
 * @param  array $data      Additional provided data that can be used to add custom extras
 * @return array            The array to be used as transaction extras
 */
	public function buildTransactionExtras($addresses, $user, $data) {
		return [
			'totals' => $this->buildTransactionTotals(),
			'delivery' => (!empty($addresses['delivery'])) ? $addresses['delivery'] : null,
			'billing' => (!empty($addresses['billing'])) ? $addresses['billing'] : null,
			'user' => $user
		];
	}

/**
 * build transaction totals from basket
 *
 * @param array $options Optional flags.  Default values defined at the start of the method
 *
 * @return 	array 	$transactionTotals
 */
	public function buildTransactionTotals($options = []) {
		$options = Hash::merge([
			'ignoreDiscounts' => false // When true, adds discount total row back onto grand total
		], $options);

		$this->getBasket();

		$grandTotal = $this->findTotalRow(Configure::read('EvBasket.labels.grandtotal'));
		$subTotal = $this->findTotalRow(Configure::read('EvBasket.labels.subtotal'));
		$taxTotal = $this->findTotalRow(Configure::read('EvBasket.labels.tax'));
		$deliveryTotal = $this->findTotalRow(Configure::read('EvBasket.labels.delivery'));
		$discount = $this->findTotalRow(Configure::read('EvBasket.labels.discount'));

		$totals = [
			'delivery' => $deliveryTotal['amount'],
			'tax' => $taxTotal['amount'],
			'subtotal' => $subTotal['amount'],
			'grandtotal' => $grandTotal['amount'],
			'discount' => (! empty($discount['amount'])) ? $discount['amount'] : false
		];

		if ($options['ignoreDiscounts']) {
			$totals = $this->_ignoreDiscount($totals);
		}

		return $totals;
	}

/**
 * Removes the discount row from the grand total
 *
 * @param array $totals Basket total rows
 * @return array Array of transaction totals
 */
	protected function _ignoreDiscount($totals) {
		$totals['grandtotal'] -= $totals['discount'];
		$totals['discount'] = false;

		return $totals;
	}

/**
 * set a basket data item
 *
 * @param string   $name          name of the data row
 * @param string   $data          value of the data row
 * @param boolean  $dispatchEvent if true, an event will be dispatched after completing the add/update
 */
	public function setBasketData($name, $data, $isVisible = false, $dispatchEvent = true) {
		$this->getBasket();

		$result = false;

		$dataRecord = $this->BasketData->setData(
			$this->basketCache['Basket']['id'],
			$name,
			$data,
			$isVisible
		);

		if ($dataRecord === true) {
			$result = true;
		}

		if ($result === true) {
			if ($dispatchEvent) {
				// dispatch data record added to basket event
				$this->_controller->getEventManager()->dispatch(
					new CakeEvent('EvBasket.Component.Basket.dataAdd', $this, [
						'basketId' => $this->basketCache['Basket']['id'],
						'data' => $dataRecord
					])
				);
			}

			// call out to recompile all the basket totals
			return $this->rebuildTotals();
		}

		return false;
	}

/**
 * Get the basket data for the current basket
 * @return array $basketData The data for the current basket
 */
	public function getBasketData() {
		$basketData = [];

		$this->getBasket();

		if (! empty($this->basketCache['BasketData'])) {
			foreach ($this->basketCache['BasketData'] as $dataRow) {
				$basketData[] = [
					'name' => $dataRow['name'],
					'data' => $dataRow['data'],
					'is_visible' => $dataRow['is_visible']
				];
			}
		}

		return $basketData;
	}

/**
 * Delete an single piece of data from the current basket. If the data is deleted successfully, the
 * dataDeleted event is dispatched.
 * @param  string $name          The name of the data to be deleted e.g. discount_code
 * @param  bool   $dispatchEvent By default deleting data fires an event, set to false to stop event from firing
 * @return bool                  If the data was deleted or not
 */
	public function deleteData($name, $dispatchEvent = true) {
		return $this->deleteDataWithKey($name, $dispatchEvent, 'name');
	}

	public function deleteDataWithKey($name, $dispatchEvent, $key = null) {
		$this->getBasket();

		// check if the row actually exists
		$dataToDelete = Hash::extract($this->basketCache, 'BasketData.{n}[' . $key . '=' . $name . ']');

		$result = $this->BasketData->deleteData(
			$dataToDelete
		);

		if ($result === true) {
			//Rebuild totals before event fired to make sure that the totals are up to date before
			// any changes are made.
			$returnTotals = $this->rebuildTotals();
			if ($dispatchEvent) {
				// dispatch item added to basket event
				$this->_controller->getEventManager()->dispatch(
					new CakeEvent('EvBasket.Component.Basket.dataDeleted', $this, [
						'basketId' => $this->basketCache['Basket']['id'],
						'dataName' => $name,
						'BasketData' => $this->BasketData
					])
				);
			}

			return $returnTotals;
		}

		return false;
	}

/**
 * Set a basket data item.
 *
 * @param int     $basketItemId  The id of the basket item to add data to.
 * @param string  $name          Name of the data row.
 * @param string  $data          Value of the data row.
 * @param boolean $dispatchEvent If true, an event will be dispatched after completing the add/update.
 * @return array|bool Basket totals if added successfully, false otherwise.
 */
	public function setBasketItemData($basketItemId, $name, $data, $dispatchEvent = true) {
		$this->getBasket();

		$result = $this->BasketItemData->setItemData(
			$basketItemId,
			$name,
			$data
		);

		if (!empty($result)) {
			if ($dispatchEvent) {
				// dispatch data record added to basket event
				$this->_controller->getEventManager()->dispatch(
					new CakeEvent('EvBasket.Component.Basket.itemDataAdd', $this, [
						'basketId' => $this->basketCache['Basket']['id'],
						'basketItemId' => $basketItemId,
						'data' => $result
					])
				);
			}

			// call out to recompile all the basket totals
			return $this->rebuildTotals();
		}

		return false;
	}

/**
 * Deletes an item from the BasketItemData array associated with the supplied
 * basket_item_id and name parameters
 *
 * @param int $basketItemId The unique id of the basket item we're dealing with
 * @param string $name The name of the data item to be deleted e.g. discount_code
 * @return null
 */
	public function deleteBasketItemItemData($basketItemId, $name) {
		$this->getBasket();
		if (! empty($this->basketCache['BasketItem'])) {
			// extract the BasketItemData relating to the supplied $basketItemId and $name
			$basketItemData = Hash::extract(
				$this->basketCache['BasketItem'],
				'{s}.BasketItemData.{n}[basket_item_id=' . $basketItemId . '][name=' . $name . ']'
			);
			// use the extracted $basketItemData array to bring out the BasketItemData id's
			$basketItemDataIds = Hash::extract($basketItemData, '{n}.id');

			if (! empty($basketItemDataIds)) {
				// delete all BasketItemData for the given id's
				$this->BasketItem->BasketItemData->deleteAll(['BasketItemData.id' => $basketItemDataIds]);
			}

			// rebuild the basket cache before returning
			$this->getBasket(true);
		}

		return;
	}

/**
 * Check that the basket in it's current state is acceptable to purchase. At it's most basic, just check that there are
 * currently items in the basket so that a user doesn't purchase an empty basket.
 * @return bool Returns true if the basket is purchaseable, otherwise returns false or a message about the reason why
 *                      the basket isn't currently purchasable.
 */
	public function isBasketPurchasable() {
		$basket = $this->getBasket();

		if (count($basket['BasketItem']) <= 0) {
			return 'Your basket is empty. Please check the contents of your basket before making a purchase.';
		}

		return true;
	}

/**
 * Remove the discount code from a basket and rebuild the total
 *
 * return bool
 */
	public function removeDiscountCode() {
		$this->deleteData('discount_code');
		$this->deleteData('voucher_code');
		$this->deleteTotalRow(Configure::read('EvBasket.labels.discount'));
		$this->deleteTotalRow(Configure::read('EvBasket.labels.voucher'));
		$this->deleteTotalRow(Configure::read('EvBasket.labels.delivery'));
		$this->rebuildTotals();

		return true;
	}

/**
 * Update the currency of the basket. To make sure that the basket is updated correctly, all the items are removed and
 * added again after the currency has been changed. Data is also removed and added again in case the data is dependent
 * on the currency e.g. discounts.
 * @param  int $newCurrencyId The ID of the currency to change the basket too.
 */
	public function updateCurrency($newCurrencyId) {
		$basket = $this->getBasket();
		if ($basket['Basket']['currency_id'] != $newCurrencyId) {
			$basketItems = [];
			foreach ($basket['BasketItem'] as $item) {
				//Get the info needed to add the item back to the basket later
				$saveItem['model'] = $item['model'];
				$saveItem['model_id'] = $item['model_id'];
				$saveItem['quantity'] = $item['quantity'];

				$basketItems[] = $saveItem;

				//Remove the item from the current basket
				//This happens when the currency is changed
			}

			//Get data
			$basketData = [];
			foreach ($basket['BasketData'] as $data) {
				//Get the info needed to add the data back to the basket later
				$saveData['name'] = $data['name'];
				$saveData['data'] = $data['data'];
				$saveData['is_visible'] = $data['is_visible'];

				$basketData[] = $saveData;

				//Remove the data form the current basket
				//This happens when the currency is changed
			}

			//update currency
			$this->_controller->Currencies->setCurrency($newCurrencyId);

			//Add data
			if (!empty($basketData)) {
				foreach ($basketData as $data) {
					//Add the data back to the bask
					if (!$this->setBasketData(
							$data['name'],
							$data['data'],
							$data['is_visible'],
							true
						)
					) {
						//Failed to add data back to basket
						$this->_controller->Flash->fail(
							'Failed to add ' . $data['name'] . 'to the basket.',
							[
								'key' => 'basket-flash'
							]
						);
					}
				}
			}

			//Add items with quantities
			if (!empty($basketItems)) {
				foreach ($basketItems as $item) {
					//Get the new information for the item
					$Model = EvClassRegistry::init($item['model']);

					$unitPrice = $this->getUnitPriceMethod($Model, $item);

					// if there's a method to get the taxRate, do so
					$taxRate = 0;
					if (method_exists($Model, 'getTaxRate')) {
						$taxRate = $Model->getTaxRate($item['model_id']);
					}

					if ($unitPrice !== false) {
						// add the item back to the basket
						if (!$this->addItem($item, $unitPrice, $item['quantity'], $taxRate)) {
							$this->_controller->Flash->fail(
								'Failed to add ' . $item['model'] . ' ' . $item['model_id'] . ' to the basket',
								[
									'key' => 'basket-flash'
								]
							);
						}
					} else {
						//Failed to find a price for the item, couldn't add it back to basket
						$this->_controller->Flash->fail(
							'Failed to add ' . $item['model'] . ' ' . $item['model_id'] . ' to the basket',
							[
								'key' => 'basket-flash'
							]
						);
					}
				}

				//Rebuild
				$this->rebuildTotals();
			}
		}
	}

/**
 * Add the price of delivery to the basket. Checks the site setting for the delivery tax rate and applies it to the
 * delivery rate. Save it to the basket and rebuilds the totals.
 * @param decimal $deliveryRate The price of the delivery to add to the basket.
 */
	public function addDeliveryToBasket($deliveryRate) {
		$deliveryTax = Configure::read('SiteSetting.ev_basket.delivery_tax_rate');

		$deliveryRateIncTax = 0;
		if ($deliveryTax !== false && !empty($deliveryRate)) {
			$deliveryRateIncTax = $deliveryRate * (1 + ($deliveryTax / 100));
		}

		if ($deliveryRate !== false) {
			$this->manageTotalRow(
				Configure::read('EvBasket.labels.delivery'),
				$deliveryRate,
				15,
				false,
				$deliveryRate,
				$deliveryRateIncTax
			);

			return $this->rebuildTotals();
		}

		return false;
	}

/**
 * Removes all is_promotion flags from basket items when the
 * BasketItemData has been removed
 *
 * @param string $itemKey The key of field for the data to delete (e.g. "name" or "data")
 * @return void
 */
	public function unflagPromotionalItem($itemKey) {
		$this->deleteBasketItemItemData($itemKey, 'promotion_id');
	}

/**
 * Given an order, creates a basket to mirror the order items and data.
 * Returns a basket hash that can be used to load the generated basket.
 *
 * @param Order $order An order array
 * @return string The basket hash
 */
	public function createNewBasketFromOrder($order) {
		// Create an empty basket
		$hash = $this->generateHash();
		$this->Basket->createBasket($hash, $order['Order']['currency_id']);

		// Load the new basket
		$basket = $this->Basket->findByHash($hash);

		// Add the items to the new basket
		foreach ($order['OrderItem'] as $orderItem) {

			$basketItem = [
				'name' => $orderItem['name'],
				'model' => $orderItem['model'],
				'model_id' => $orderItem['model_id'],
			];

			if (!empty($orderItem['OrderItemData'])) {
				$basketItem['BasketItemData'] = $orderItem['OrderItemData'];
			}

			$this->BasketItem->addItem(
				$basket['Basket']['id'],
				$basketItem,
				$orderItem['unit_price'],
				$orderItem['quantity'],
				$orderItem['tax_rate']
			);
		}

		// Copy all of the data into the basket data
		foreach ($order['OrderData'] as $orderData) {
			$this->BasketData->setData(
				$basket['Basket']['id'],
				$orderData['name'],
				$orderData['data'],
				$orderData['is_visible']
			);
		}

		// Copy the totals
		foreach ($order['OrderTotal'] as $orderTotal) {
			$this->BasketTotal->manageTotalRow(
				$basket['Basket']['id'],
				[
					'name' => $orderTotal['name'],
					'amount' => $orderTotal['amount'],
					'sequence' => $orderTotal['sequence'],
					'display_only' => false,
					'display_ex_tax' => $orderTotal['display_ex_tax'],
					'display_inc_tax' => $orderTotal['display_inc_tax'],
					'tax' => $orderTotal['display_inc_tax'] - $orderTotal['display_ex_tax'],
					'is_grand_total' => $orderTotal['name'] == Configure::read('EvBasket.labels.grandtotal')
				]
			);
		}

		return $hash;
	}

/**
 * Given a basket item, return the name of the item using the config BasketItemNamePath.
 *
 * @param array $BasketItem The basket item.
 * @return string The name of the basket item.
 */
	public function getItemName($BasketItem) {
		return BasketLib::getItemName($BasketItem);
	}

/**
 * Check if a particular basket is empty. A basket is empty if it can't be found, if it doesn't
 * have any basket totals set up or if it doesn't have any basket items associated with it.
 * If no basket id is passed then the current basket cache is checked.
 *
 * @param int $basketId The id of the basket to check. Current basket if empty.
 * @return bool
 */
	public function isBasketEmpty($basketId = null) {
		if (empty($basketId)) {
			$basket = $this->getBasket(false, true);
		} else {
			$basket = $this->getFullBasketByBasketId($basketId);
		}

		if (empty($basket)) {
			return true;
		}

		if (empty($basket['BasketTotal'])) {
			return true;
		}

		if (empty($basket['BasketItem'])) {
			return true;
		}

		return false;
	}

/**
 * A function to get what method is required
 *
 * @param object $Model The model relating to the basket item
 *
 * @return string|bool
 */
	public function getUnitPriceMethodName($Model) {
		/*
		 e.g. to extend this method with your own get price method.
		if (method_exists($Model, 'myGetUnitPrice')) {
			return 'myGetUnitPrice';
		}
		 */
		return method_exists($Model, 'getUnitPrice') ? 'getUnitPrice' : false;
	}

/**
 * A function that allows a site to use different method for getting a unit price.
 * This can be extended if you don't wish to pass the whole basketItem method.
 *
 * @param object $Model The model relating to the basket item
 * @param array $basketItem Basket Item Data
 *
 * @return float
 */
	public function getUnitPriceMethod($Model, array $basketItem) {
		$methodName = $this->getUnitPriceMethodName($Model);

		if ($methodName === false) {
			return 0;
		}

		return ($methodName === 'getUnitPrice') ? $Model->getUnitPrice($basketItem['model_id']) : $Model->{$methodName}($basketItem);
	}
}
