<?php
App::uses('EvBasketAppModel', 'EvBasket.Model');
/**
 * BasketTotal Model
 *
 * @property Basket $Basket
 */
class BasketTotal extends EvBasketAppModel {

	/**
	 * Display field
	 *
	 * @var string
	 */
	public $displayField = 'name';

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'basket_id' => array(
			'numeric' => array(
				'rule' => array('numeric')
			),
		),
		'name' => array(
			'notBlank' => array(
				'rule' => array('notBlank')
			),
		),
		'amount' => array(
			'numeric' => array(
				'rule' => array('numeric')
			),
		),
	);

	//The Associations below have been created with all possible keys, those that are not needed can be removed

	/**
	 * belongsTo associations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'Basket' => array(
			'className' => 'EvBasket.Basket'
		)
	);

	/**
	 * afterSave - set each items element key to the row ID
	 *
	 */
	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		if ($primary === true) {
			$results = Hash::combine(
				$results,
				'{n}.BasketTotal.id',
				'{n}'
			);
		}

		return $results;
	}

	/**
	 * given an array of totals rows - find the grand total
	 *
	 * @param 	array 		$totalRows 	Array of basket totals
	 * @return 	int|bool  	$key 		Either bool false on fail or the element key it resides in
	 */
	public function findGrandTotalRow($totalRows) {
		$grandTotalArray = Hash::combine($totalRows, '{n}.BasketTotal.id', '{n}.BasketTotal.is_grand_total');
		return array_search(1, $grandTotalArray);
	}

	/**
	 * total up he items and totals
	 *
	 * @param 	array 		$items 		All the basket Items
	 * @param 	array 		$totals 	All the basket Rows
	 * @return 	decimal 	$grandTotal The grand total
	 */
	public function addUpTotals($items, $totals) {
		$grandTotal = 0;

		// loop and add items
		foreach ($items as $BasketItem) {
			$grandTotal += $BasketItem['BasketItem']['row_total'];
		}

		// loop and add totals
		foreach ($totals as $BasketTotal) {
			if (isset($BasketTotal['BasketTotal']['is_grand_total']) && (bool)$BasketTotal['BasketTotal']['is_grand_total'] === true) {
				continue;
			}

			$grandTotal += $BasketTotal['BasketTotal']['amount'];
		}
		
		// this ensures that if the eventual grand total is less than 0, or a figure
		// like 0.0003, we actually return a straight 0
		if (number_format($grandTotal, 2) <= 0) {
			$grandTotal = 0;
		}

		return $grandTotal;
	}

	/**
	 * given the basket ID recalculate the totals
	 *
	 * @param 	int 	$basketId 	Basket Id to recalculate
	 * @return 	bool
	 */
	public function recompileTotals($basketId) {
		if (empty($basketId)) {
			return false;
		}

		// get all the items
		$items = $this->Basket->BasketItem->find(
			'all',
			array(
				'conditions' => array(
					'BasketItem.basket_id' => $basketId
				)
			)
		);

		// get all the displayable totals
		$totals = $this->find(
			'all',
			array(
				'conditions' => array(
					'BasketTotal.basket_id' => $basketId,
					'BasketTotal.display_only' => 0
				),
				'order' => 'BasketTotal.sequence ASC'
			)
		);

		// add the totals together
		$grandTotal = $this->addUpTotals($items, $totals);

		// find me the grand total row
		$grandTotalKey = $this->findGrandTotalRow($totals);

		if ($grandTotalKey !== false) {
			$result = $this->updateGrandTotal(
				$totals[$grandTotalKey]['BasketTotal']['id'],
				$grandTotal
			);
		} else {
			$result = $this->createGrandTotal(
				$basketId,
				$grandTotal
			);
		}

		return $result;
	}

	/**
	 * create a grand total row in the database
	 *
	 * @param 	int 		$basketId 		The basket ID to associate it with
	 * @param 	decimal 	$grandTotal 	The amount
	 * @return 	bool|array
	 */
	public function createGrandTotal($basketId, $grandTotal) {
		$this->clear();

		$this->set(
			array(
				'basket_id' => $basketId,
				'name' => Configure::read('EvBasket.labels.grandtotal'),
				'amount' => $grandTotal,
				'is_grand_total' => 1,
				'sequence' => 50
			)
		);

		return (bool)$this->save();
	}

	/**
	 * update a grand total row in the database
	 *
	 * @param 	int 		$id 			The total row ID
	 * @param 	decimal 	$grandTotal 	The amount
	 * @return 	bool|array
	 */
	public function updateGrandTotal($id, $grandTotal) {
		$this->clear();

		$this->set(
			array(
				'id' => $id,
				'amount' => $grandTotal
			)
		);

		return (bool)$this->save();
	}

	/**
	 * create a total row
	 *
	 * @param 	int 	$basketId 		The basket ID to link it to
	 * @param 	array 	$data 			Data array to enter (name, amount, sequence, display_only)
	 * @param 	int 	$BasketTotalId	Id number for the basket total row (optional)
	 * @return 	bool
	 */
	public function manageTotalRow($basketId, $data, $BasketTotalId = null) {
		$this->clear();

		if (empty($basketId) || empty($data['name']) || ! isset($data['amount'])) {
			return false;
		}

		if (! empty($BasketTotalId)) {
			$data['id'] = $BasketTotalId;
		}

		$data['basket_id'] = $basketId;

		$this->set(
			$data
		);

		return (bool)$this->save();
	}
}
