<?php
App::uses('AppController', 'Controller');
App::uses('CakeRequest', 'Network');
App::uses('CakeResponse', 'Network');
App::uses('ComponentCollection', 'Controller');
App::uses('BasketComponent', 'EvBasket.Controller/Component');
App::uses('Cookie', 'Controller/Component');

// A fake controller to test against
class BasketsControllerTest extends AppController {

}


class BasketComponentTest extends CakeTestCase {

	public $BasketComponent = null;

	public $Controller = null;

	/**
	 * Fixtures
	 *
	 * @var array
	 */
	public $fixtures = array(
		'plugin.ev_basket.basket',
		'plugin.ev_basket.basket_item',
		'plugin.ev_basket.basket_total',
		'plugin.ev_basket.test_model'
	);

	/**
	 * setUp method
	 *
	 * @return void
	 */
	public function setUp() {
		parent::setUp();

		Configure::write(
			'EvBasket.BasketItemBelongsTo',
			array(
				'TestModel' => array(
					'className' => 'EvBasket.TestModel',
					'foreignKey' => 'model_id',
					'conditions' => array(
						"BasketItem.model = 'EvBasket.TestModel'"
					)
				)
			)
		);
		Configure::write('EvBasket.BasketItemContains', array());

		// Setup our component and fake test controller
		$Collection = new ComponentCollection();

		// mock the component to override the generateHash method so we can control it for testing
		$this->BasketComponent = $this->getMockBuilder('BasketComponent')
			->setConstructorArgs(
				array(
					$Collection,
					array(
						'Basket' => EvClassRegistry::init('EvBasket.Basket'),
						'BasketItem' => EvClassRegistry::init('EvBasket.BasketItem'),
						'BasketTotal' => EvClassRegistry::init('EvBasket.BasketTotal')
					)
				)
			)
			->setMethods(array('generateHash'))
			->getMock();

		$this->BasketComponent->expects($this->any())
			->method('generateHash')
			->will($this->returnValue('68838a0c7d04-0c4b-f574-04e0-da4f1d55'));

		$CakeRequest = new CakeRequest();
		$CakeResponse = new CakeResponse();
		$this->Controller = new BasketsControllerTest($CakeRequest, $CakeResponse);
		$this->Controller->constructClasses();

		// mock the cookie class
		$this->Controller->BasketCookie = $this->getMockBuilder('CookieComponent')
			->setMethods(array('write', 'read', 'check'))
			->getMock();

		$this->Controller->BasketCookie->expects($this->any())
			->method('check')
			->will($this->returnValue(false));

		$this->Controller->BasketCookie->expects($this->any())
			->method('read')
			->will($this->returnValue('68838a0c7d04-0c4b-f574-04e0-da4f1d55'));

		$this->Controller->BasketCookie->expects($this->any())
			->method('write')
			->will($this->returnValue(true));

		$this->BasketComponent->initialize($this->Controller);
	}

	/**
	 * tearDown method
	 *
	 * @return void
	 */
	public function tearDown() {
		parent::tearDown();
		// Clean up after we're done
		unset($this->BasketComponent);
		unset($this->Controller);
	}

	public function testSetupHashReturnsAValidHashAndReturnsSameHashWhenCalledAgain() {
		$this->assertEquals(
			'68838a0c7d04-0c4b-f574-04e0-da4f1d55',
			$this->BasketComponent->setupHash()
		);

		$this->assertEquals(
			'68838a0c7d04-0c4b-f574-04e0-da4f1d55',
			$this->BasketComponent->setupHash()
		);
	}

	public function testSetupBasketReturnsTrueAndCreatesBasketIfDoesNotExist() {
		$this->assertTrue(
			$this->BasketComponent->setupBasket('68838a0c7d04-0c4b-f574-04e0-da4f1d55')
		);
	}

	public function testSetupBasketReturnsFalseIfBasketAlreadyExists() {
		$this->assertFalse(
			$this->BasketComponent->setupBasket('55d1f4ad-0e40-475f-b4c0-40d7c0a83886')
		);
	}

	public function testbuildItemContainBuildsCorrectArraysForBasket() {
		$relationships = array(
			'TestModel' => array(
				'className' => 'EvBasket.TestModel',
				'foreignKey' => 'model_id',
				'conditions' => array(
					"BasketItem.model = 'EvBasket.TestModel'"
				)
			),
			'AnotherModel'
		);

		$this->assertEquals(
			array(
				'TestModel' => array(
					'TestModel2'
				),
				'AnotherModel'
			),
			$this->BasketComponent->buildItemContain(
				array(
					'TestModel' => array(
						'TestModel2'
					),
					'AnotherModel'
				),
				$relationships
			)
		);

		$this->assertEquals(
			array(
				'TestModel',
				'AnotherModel'
			),
			$this->BasketComponent->buildItemContain(
				array(),
				$relationships
			)
		);
	}

	public function testbuildItemKeyBuildsTheCorrectKeyWhenGivenCorrectData() {
		$this->assertEquals(
			'EvBasket.TestModel.4',
			$this->BasketComponent->buildItemKey(
				array(
					'model' => 'EvBasket.TestModel',
					'model_id' => 4
				)
			)
		);
	}

	public function testbuildItemKeyReturnsFalseWhenDataMissing() {
		$this->assertFalse(
			$this->BasketComponent->buildItemKey(
				array(
					'model' => 'EvBasket.TestModel'
				)
			)
		);

		$this->assertFalse(
			$this->BasketComponent->buildItemKey(
				array(
					'model_id' => 4
				)
			)
		);
	}

	public function testfindItemKeyReturnsTheCorrectKey() {
		$this->BasketComponent->hash = '55d1f4ad-0e40-475f-b4c0-40d7c0a83886';
		$this->BasketComponent->getBasket();

		$this->assertEquals(
			'EvBasket.TestModel.1',
			$this->BasketComponent->getItemKey('EvBasket.TestModel.1')
		);

		$this->assertEquals(
			'EvBasket.TestModel.1',
			$this->BasketComponent->getItemKey(
				array(
					'model' => 'EvBasket.TestModel',
					'model_id' => 1
				)
			)
		);

		$this->assertEquals(
			'EvBasket.TestModel.1',
			$this->BasketComponent->getItemKey(1)
		);
	}

	public function testFindItemKeyReturnsFalseIfIdIntDoesNotExists() {
		$this->BasketComponent->hash = '55d1f4ad-0e40-475f-b4c0-40d7c0a83886';
		$this->BasketComponent->getBasket();

		$this->assertFalse(
			$this->BasketComponent->getitemKey(100)
		);
	}

	public function testbasketItemExistsReturnsTrueorFalseWhenItemDoesOrDoesntExist() {
		$this->BasketComponent->hash = '55d1f4ad-0e40-475f-b4c0-40d7c0a83886';
		$this->BasketComponent->getBasket();

		$this->assertTrue(
			$this->BasketComponent->basketItemExists('EvBasket.TestModel.1')
		);

		$this->assertFalse(
			$this->BasketComponent->basketItemExists('EvBasket.TestModel.100')
		);
	}

	public function testbasketItemExistsReturnsTrueorFalseWhenItemDoesOrDoesntExistWithPassedBasket() {
		$this->BasketComponent->hash = '55d1f4ad-0e40-475f-b4c0-40d7c0a83886';
		$this->BasketComponent->getBasket();

		$basket = array(
			'Basket' => array(
				'id' => 1,
				'hash' => '55d1f4ad-0e40-475f-b4c0-40d7c0a83886',
				'created' => '2015-08-17 08:58:06',
				'modified' => '2015-08-17 08:58:06'
			),
			'BasketItem' => array(
				'EvBasket.TestModel.100' => array(
					'id' => '1',
					'basket_id' => '1',
					'model' => 'EvBasket.TestModel',
					'model_id' => '100',
					'unit_price' => '10.000000',
					'quantity' => '2',
					'row_total' => '20.000000',
					'created' => '2015-08-17 08:59:29',
					'modified' => '2015-08-17 08:59:29',
					'TestModel' => array(
						'id' => '1',
						'name' => 'A Test Record',
						'created' => '2015-07-20 16:14:47',
						'modified' => '2015-07-20 16:14:47'
					)
				),
				'EvBasket.TestModel.150' => array(
					'id' => '2',
					'basket_id' => '1',
					'model' => 'EvBasket.TestModel',
					'model_id' => '150',
					'unit_price' => '5.500000',
					'quantity' => '1',
					'row_total' => '5.500000',
					'created' => '2015-08-17 08:59:29',
					'modified' => '2015-08-17 08:59:29',
					'TestModel' => array(
						'id' => '2',
						'name' => 'A Test Record Number two',
						'created' => '2015-07-20 16:14:47',
						'modified' => '2015-07-20 16:14:47'
					)
				)
			),
			'BasketTotal' => array(
				'1' => array(
					'id' => '1',
					'basket_id' => '1',
					'name' => 'Grand Total',
					'amount' => '25.5000000',
					'sequence' => '30',
					'is_grand_total' => true,
					'created' => '2015-08-17 09:00:17',
					'modified' => '2015-08-17 09:00:17'
				)
			)
		);

		$this->assertTrue(
			$this->BasketComponent->basketItemExists(
				'EvBasket.TestModel.100',
				$basket
			)
		);

		$this->assertFalse(
			$this->BasketComponent->basketItemExists(
				'EvBasket.TestModel.200',
				$basket
			)
		);
	}

	public function testGetBasketReturnsFullBasketArrayWithSetHash() {
		$this->BasketComponent->hash = '55d1f4ad-0e40-475f-b4c0-40d7c0a83886';

		$this->assertEquals(
			array(
				'Basket' => array(
					'id' => 1,
					'hash' => '55d1f4ad-0e40-475f-b4c0-40d7c0a83886',
					'created' => '2015-08-17 08:58:06',
					'modified' => '2015-08-17 08:58:06'
				),
				'BasketItem' => array(
					'EvBasket.TestModel.1' => array(
						'id' => '1',
						'basket_id' => '1',
						'model' => 'EvBasket.TestModel',
						'model_id' => '1',
						'unit_price' => '10.000000',
						'quantity' => '2',
						'row_total' => '20.000000',
						'created' => '2015-08-17 08:59:29',
						'modified' => '2015-08-17 08:59:29',
						'TestModel' => array(
							'id' => '1',
							'name' => 'A Test Record',
							'created' => '2015-07-20 16:14:47',
							'modified' => '2015-07-20 16:14:47'
						)
					),
					'EvBasket.TestModel.2' => array(
						'id' => '2',
						'basket_id' => '1',
						'model' => 'EvBasket.TestModel',
						'model_id' => '2',
						'unit_price' => '5.500000',
						'quantity' => '1',
						'row_total' => '5.500000',
						'created' => '2015-08-17 08:59:29',
						'modified' => '2015-08-17 08:59:29',
						'TestModel' => array(
							'id' => '2',
							'name' => 'A Test Record Number two',
							'created' => '2015-07-20 16:14:47',
							'modified' => '2015-07-20 16:14:47'
						)
					)
				),
				'BasketTotal' => array(
					'2' => array(
						'id' => '2',
						'basket_id' => '1',
						'name' => 'Sub Total',
						'amount' => '25.500000',
						'sequence' => '10',
						'is_grand_total' => false,
						'created' => '2015-08-17 09:00:17',
						'modified' => '2015-08-17 09:00:17'
					),
					'3' => array(
						'id' => '3',
						'basket_id' => '1',
						'name' => 'Delivery',
						'amount' => '3.000000',
						'sequence' => '20',
						'is_grand_total' => false,
						'created' => '2015-08-17 09:00:17',
						'modified' => '2015-08-17 09:00:17'
					),
					'1' => array(
						'id' => '1',
						'basket_id' => '1',
						'name' => 'Grand Total',
						'amount' => '28.5000000',
						'sequence' => '30',
						'is_grand_total' => true,
						'created' => '2015-08-17 09:00:17',
						'modified' => '2015-08-17 09:00:17'
					)
				)
			),
			$this->BasketComponent->getBasket()
		);
	}

	public function testGetBasketCreatesBasketAndReturnsSameOneAgainFromCache() {
		$this->BasketComponent->getBasket();

		$now = date('Y-m-d H:i:s');
		$this->assertEquals(
			array(
				'Basket' => array(
					'id' => '2',
					'hash' => '68838a0c7d04-0c4b-f574-04e0-da4f1d55',
					'created' => $now,
					'modified' => $now
				),
				'BasketItem' => array(),
				'BasketTotal' => array()
			),
			$this->BasketComponent->getBasket()
		);
	}

	public function testResetBasketResetsHashAndBasket() {
		$this->BasketComponent->getBasket();

		$this->assertEquals(
			$this->BasketComponent->hash,
			'68838a0c7d04-0c4b-f574-04e0-da4f1d55'
		);

		$now = date('Y-m-d H:i:s');
		$this->assertEquals(
			array(
				'Basket' => array(
					'id' => '2',
					'hash' => '68838a0c7d04-0c4b-f574-04e0-da4f1d55',
					'created' => $now,
					'modified' => $now
				),
				'BasketItem' => array(),
				'BasketTotal' => array()
			),
			$this->BasketComponent->basketCache
		);

		$this->assertTrue(
			$this->BasketComponent->resetBasket()
		);

		$this->assertNull(
			$this->BasketComponent->hash
		);

		$this->assertNull(
			$this->BasketComponent->basketCache
		);
	}
}
