<?php

App::uses('AppHelper', 'View/Helper');
App::uses('BasketLib', 'EvBasket.Lib');

class BasketHelper extends AppHelper {

/**
 * Given the basket item array row return the text to show on the basket.
 *
 * @param array $BasketItem The basket item row from the DB
 * @return string The name of the basket item.
 */
	public function getItemName($BasketItem) {
		return BasketLib::getItemName($BasketItem);
	}

/**
 * Helper to just generate the bakset item id and quantity field keep the template neater.
 *
 * @param array $BasketItem The basket item row from the DB.
 * @param array $options    Form Options array for the quantity - to override class and such.
 * @return void.
 */
	public function quantityField($BasketItem, $options = array()) {
		echo $this->_View->Form->addField(
			$BasketItem['id'] . '.BasketItem.id',
			array(
				'type' => 'hidden',
				'value' => $BasketItem['id'],
			)
		);

		echo $this->_View->Form->addField(
			$BasketItem['id'] . '.BasketItem.model',
			array(
				'type' => 'hidden',
				'value' => $BasketItem['model'],
			)
		);

		echo $this->_View->Form->addField(
			$BasketItem['id'] . '.BasketItem.model_id',
			array(
				'type' => 'hidden',
				'value' => $BasketItem['model_id'],
			)
		);

		$options = array_merge(
			array(
				'type' => 'integer',
				'value' => $BasketItem['quantity'],
				'label' => false,
			),
			$options
		);

		echo $this->_View->Form->addField(
			$BasketItem['id'] . '.BasketItem.quantity',
			$options
		);
	}

/**
 * Helper for a delete item link keeps the template neater
 *
 * @param array  $BasketItem The basket item row from the DB.
 * @param string $linkText The text for the link.
 * @param array  $linkOptions Options to be passed to Html->link().
 * @return void.
 */
	public function deleteLink($BasketItem, $linkText, $linkOptions = array()) {
		$route = $this->_View->Route->getItemRoute('EvBasket', 'Basket');
		$route['action'] = 'remove';
		$route[] = $BasketItem['id'];

		$options = array_merge(
			array(
				'confirm' => 'Are you sure?',
			),
			$linkOptions
		);

		echo $this->_View->Html->link(
			$linkText,
			$route,
			$options
		);
	}

/**
 * Helper to return the basket form update url.
 *
 * @param string $type The action to create a route for.
 * @return array Route array for a basket route.
 */
	public function getFormUrl($type) {
		$route = $this->_View->Route->getItemRoute('EvBasket', 'Basket');
		$route['action'] = $type;

		return $route;
	}

/**
 * Get the totals for the basket and adjust the displayed values depending on if VAT is to be shown or not (config setting).
 *
 * @param array $basket  The basket object containing the current totals and items in the basket.
 * @param bool  $showTax An override to display prices including VAT or display tax separately.
 * @return array An array containing the basket totals with any modification applied.
 */
	public function getViewTotals($basket, $showTax = false) {
		$allTotals = $basket['BasketTotal'];
		$totalIds = Hash::combine($allTotals, '{n}.name', '{n}.id');

		$includeTax = Configure::read('EvBasket.siteIncludingTax');

		if (!empty($includeTax) || $showTax) {
			//Need to get the original tax amount
			$originalTaxTotal = 0;
			foreach ($basket['BasketItem'] as $basketItem) {
				$originalTaxTotal += $basketItem['row_tax_amount'];
			}

			$taxAmount = $allTotals[$totalIds[Configure::read('EvBasket.labels.tax')]]['amount'];
			$subTotalAmount = $allTotals[$totalIds[Configure::read('EvBasket.labels.subtotal')]]['amount'];

			if (isset($totalIds[Configure::read('EvBasket.labels.discount')])) {
				$discountAmount = $allTotals[$totalIds[Configure::read('EvBasket.labels.discount')]]['amount'];
			}

			//Get the total price of the basket not including discounts
			$subTotalAmount += $originalTaxTotal;
			$allTotals[$totalIds[Configure::read('EvBasket.labels.subtotal')]]['amount'] = $subTotalAmount;

			//Need to work out the original discount
			if (isset($discountAmount)) {
				//Work out the difference between the original tax and the discounted tax
				$discountedTaxAmount = $originalTaxTotal - $taxAmount;
				$discountAmount -= $discountedTaxAmount;
				$allTotals[$totalIds[Configure::read('EvBasket.labels.discount')]]['amount'] = $discountAmount;
			}
		} else {
			//Unset the tax as we now don't want to display it
			unset($allTotals[$totalIds[Configure::read('EvBasket.labels.tax')]]);
		}

		return $allTotals;
	}

/**
 * Add a custom view total to an already generated set of view totals. If a sequence is provided as part of the custom
 * total then it will be added in order otherwise it will just be added to the end of the array of view totals.
 *
 * @param array $customTotal Array of the new total, containing: Name, Amount and optional sequence.
 * @param array $viewTotals  Array of the modified view totals.
 * @return array The viewTotals with the custom total added.
 */
	public function addCustomViewTotal($customTotal, $viewTotals) {
		$viewTotals[] = $customTotal;

		if (!empty($customTotal['sequence'])) {
			$viewTotals = Hash::sort($viewTotals, '{n}.sequence');
		}

		return $viewTotals;
	}

/**
 * Get a specific total from the basket totals array.
 *
 * @param array $basketTotals The basket totals array as returned as part of an basket.
 * @param array $totalName    The name of the total you want to get.
 * @return string|array The total amount if found, false if not found and null if no basket data was provided.
 */
	public function getBasketTotal($basketTotals, $totalName) {
		if (empty($basketTotals) || empty($totalName)) {
			return null;
		}

		return ArrayUtil::findFirst($basketTotals, 'name', $totalName, 'amount');
	}

/**
 * Get a specific item of basket data from the basket data array.
 *
 * @param array $basketData The basket data array as returned as part of an basket.
 * @param array $dataName   The name of the data you want to get.
 * @return string|array The data if found, false if not found and null if no basket data was provided.
 */
	public function getBasketData($basketData, $dataName) {
		if (empty($basketData) || empty($dataName)) {
			return null;
		}

		return ArrayUtil::findFirst($basketData, 'name', $dataName, 'data');
	}

/**
 * Get a specific item of basket item data from the basket item data array.
 *
 * @param array $basketItemData The basket data array as returned as part of an basket.
 * @param array $dataName       The name of the data you want to get.
 * @return string|array The data if found, false if not found and null if no basket item data was provided
 */
	public function getBasketItemData($basketItemData, $itemDataName) {
		if (empty($basketItemData) || empty($itemDataName)) {
			return null;
		}

		return ArrayUtil::findFirst($basketItemData, 'name', $itemDataName, 'item_data');
	}

/**
 * Get the price to display for an item. The item only needs to be provided if you want to only get the price of the
 * item. If EvBulkDiscount is installed then you can use the additional parameters to get the original price.
 *
 * @param array $item             The array of a single basket item.
 * @param bool  $getOriginalPrice True if you want the original price of an item. False by default.
 * @param array $basketData       Only provide if you want the original price. The basket data array as provided
 *                                as part of an basket.
 * @return string The price to display.
 */
	public function displayBasketItemUnitPrice($item, $getOriginalPrice = false, $basketData = []) {
		//Get the original prices from the basket data
		if ($getOriginalPrice && !empty($item['model']) && !empty($item['model_id']) && !empty($basketData)) {
			$basketItemPath = $item['model'] . '.' . $item['model_id'];

			$originalPriceData = $this->getBasketData($basketData, 'original_price');

			if (!empty($originalPriceData) && !is_array($originalPriceData)) {
				$originalPriceData = json_decode($originalPriceData, true);

				if (!empty($originalPriceData[$basketItemPath]['unit_price_incTax'])) {
					return $originalPriceData[$basketItemPath]['unit_price_incTax'];
				}
			}

			return null;
		}

		//Get the unit price for an item. If the taxed value is available then use that. There are potentially 2 tax
		//fields because the basket incTax field is calculated so doesn't include the underscore whereas the order
		//in_tax field is a database field so is underscored. This helper needs to work with both for compatability.
		$price = $item['unit_price'];

		if (CakeSession::read('EvCheckout.omitOrderVat') !== true) {
			if (!empty($item['unit_price_inc_tax']) && $item['unit_price_inc_tax'] > 0) {
				$price = $item['unit_price_inc_tax'];
			} elseif (!empty($item['unit_price_incTax']) && $item['unit_price_incTax'] > 0) {
				$price = $item['unit_price_incTax'];
			}
		}

		return $price;
	}

/**
 * Get the row total price to display for an item. If the row total inclusive of tax is available then that price will
 * be returned otherwise the exclusive of tax value is returned. Multiple indexes are checked for values inclusive of
 * tax due to ie being able to be compatible with EvCheckout order values.
 *
 * @param array $item The array containing a single basket item.
 * @return string The row total to display.
 */
	public function displayBasketItemRowTotal($item) {
		$price = $item['row_total'];
		if (!empty($item['row_total_inc_tax']) && $item['row_total_inc_tax'] > 0) {
			$price = $item['row_total_inc_tax'];
		} elseif (!empty($item['row_total_incTax']) && $item['row_total_incTax'] > 0) {
			$price = $item['row_total_incTax'];
		}

		return $price;
	}
}
