<?php
App::uses('EvBasketAppModel', 'EvBasket.Model');
/**
 * BasketItem Model
 *
 * @property Basket $Basket
 * @property Model $Model
 */
class BasketItem extends EvBasketAppModel {

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'basket_id' => array(
			'numeric' => array(
				'rule' => array('numeric'),
				'on' => 'create'
			),
		),
		'model' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'on' => 'create'
			),
		),
		'model_id' => array(
			'numeric' => array(
				'rule' => array('numeric'),
				'on' => 'create'
			),
		),
		'quantity' => array(
			'numeric' => array(
				'rule' => array('numeric')
			),
		)
	);

	// The Associations below have been created with all possible keys, those that are not needed can be removed

/**
 * belongsTo associations
 *
 * @var array
 */
	public $belongsTo = array(
		'Basket' => array(
			'className' => 'EvBasket.Basket'
		)
	);

	public $hasMany = array(
		'BasketItemData' => array(
			'className' => 'EvBasket.BasketItemData',
			'foreignKey' => 'basket_item_id'
		)
	);

/**
 * constructor - setup the dynamic relationships
 *
 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		$relationships = Configure::read('EvBasket.BasketItemBelongsTo');
		if (! empty($relationships)) {
			$belongsTo = array();
			foreach ($relationships as $key => $value) {
				$belongsTo[$key] = $value;
			}

			if (! empty($belongsTo)) {
				$this->bindModel(
					array(
						'belongsTo' => $belongsTo
					),
					false
				);
			}
		}
	}

	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		if ($primary === true && (isset($results[0]['model']) && isset($results[0]['model_id']))) {
			$results = Hash::combine(
				$results,
				array(
					'%s.%s',
					'{n}.BasketItem.model',
					'{n}.BasketItem.model_id'
				),
				'{n}'
			);
		}

		return $results;
	}

/**
 * calculate the tax for each basket item
 *
 * NOTE TO FUTURE DEVELOPERS: This way of working out tax may seem a little odd but
 * if you work out the tax and subtotals independent of eachother then you end up with
 * penny out issues or not able to get 99.99 as a inc vat price.
 * This code works by taking the ex VAT price that we store in the DB, it then works out the
 * rounded listing price. Then using the rounded listing price it calculates the tax by
 * subtracting them.
 *
 * @param 	array 	$BasketItems 		Array of basket items
 * @return 	arrayh 	$calculatedItems	Array of basket items with calculated tax
 */
	public function calculateTax($basketItems) {
		foreach ($basketItems as $key => $item) {
			if (! empty($item['tax_rate'])) {
				if (empty($item['unit_tax'])) {
					$item['unit_tax'] = $this->calculateUnitTax($item['unit_price'], $item['tax_rate']);

					$item['unit_price'] = round($item['unit_price'], 2);
				}

				//Calculate to total unit amount
				$unitPriceWithTax = $item['unit_tax'] + $item['unit_price'];

				$basketItems[$key]['row_tax_amount'] = $item['unit_tax'] * $item['quantity'];
				$basketItems[$key]['unit_price_incTax'] = $unitPriceWithTax;
				$basketItems[$key]['row_total_incTax'] = $unitPriceWithTax * $item['quantity'];

			} else {
				$basketItems[$key]['unit_price_incTax'] = $item['unit_price'];
				$basketItems[$key]['unit_tax'] = 0;
				$basketItems[$key]['row_tax_amount'] = 0;
				$basketItems[$key]['row_total_incTax'] = $item['row_total'];
			}
		}
		return $basketItems;
	}

/**
 * Provided the unit price and tax rate, calculate the amount of tax an item will have to reach it's total price
 * including tax.
 * @param  decimal $unitPrice The unit price of the item
 * @param  decimal $taxRate   The tax rate of the item
 * @return decimal            The amount of tax a unit item has
 */
	public function calculateUnitTax($unitPrice, $taxRate) {
		$unitPriceRounded = round($unitPrice, 2);

		// Get correct sale price
		$unitPriceWithTaxNonRounded = (1 + ($taxRate / 100)) * $unitPrice; //=99.99

		// Round to 2d.p
		$unitPriceWithTax = round($unitPriceWithTaxNonRounded, 2);

		//Calculate unit tax
		return $unitPriceWithTax - $unitPriceRounded;
	}

/**
 * add an item to the basket
 *
 * @param 	int 		$basketId 		ID of the basket row
 * @param 	array 		$item 			Array with two elements of model and model_id
 * @param 	decimal 	$unitPrice 		Unit Price of the item
 * @param 	int 		$quantity		Number of items to add
 * @param 	decimal		$taxRate 		The tax rate to update
 * @return 	bool
 */
	public function addItem($basketId, $item, $unitPrice, $quantity, $taxRate = 0) {
		$this->clear();

		$unitTax = $this->calculateUnitTax($unitPrice, $taxRate);
		$unitPrice = round($unitPrice, 2);

		$data = array(
			'BasketItem' => array(
				'basket_id' => $basketId,
				'name' => !empty($item['name']) ? $item['name'] : '',
				'model' => $item['model'],
				'model_id' => $item['model_id'],
				'unit_price' => $unitPrice,
				'unit_tax' => $unitTax,
				'tax_rate' => $taxRate,
				'quantity' => $quantity,
				'row_total' => ($unitPrice * $quantity),
			)
		);

		if (! empty($item['BasketItemData'])) {
			foreach ($item['BasketItemData'] as $key => $value) {
				$data['BasketItem']['BasketItemData'][] = array(
					'name' => $key,
					'item_data' => $value
				);
			}
		}

		return (bool)$this->saveMany($data, array('deep' => true));
	}

/**
 * update an item in the basket
 *
 * @param 	array 		$basketItem 	The existing BasketItem Row
 * @param 	int 		$quantity		Number of items to add
 * @param 	decimal 	$unitPrice 		Unit Price of the item
 * @param 	decimal		$taxRate 		The tax rate to update
 * @param   decimal     $unitTax        The unit tax of the item to update
 * @return 	bool
 */
	public function updateItem($basketItem, $newQuantity, $unitPrice = null, $taxRate = 0, $unitTax = null) {
		$this->clear();

		$data = array(
			'id' => $basketItem['id'],
			'quantity' => $newQuantity,
			'row_total' => $basketItem['unit_price'] * $newQuantity
		);

		// if we're not updating the unit price use old one
		if (!empty($unitPrice)) {
			$data['unit_price'] = $unitPrice;
			$data['row_total'] = $data['unit_price'] * $newQuantity;
		}

		// if we're not updating the unit price use old one
		if (!empty($taxRate)) {
			$data['tax_rate'] = $taxRate;
		}

		// if we're not updating the unit tax use old one
		if (!empty($unitTax)) {
			$data['unit_tax'] = $unitTax;
		}

		if (! empty($basketItem['BasketItemData'])) {
			foreach ($basketItem['BasketItemData'] as $key => $value) {
				$data['BasketItemData'][] = array(
					'name' => $key,
					'item_data' => $value
				);
			}
		}

		$this->set($data);

		return (bool)$this->save();
	}

/**
 * delete an item from the basket
 *
 * @param 	array 		$basketItem 	The existing BasketItem Row
 * @return 	bool
 */
	public function deleteItem($basketItem) {
		$this->clear();

		return $this->delete($basketItem['id']);
	}
}
