<?php
App::uses('EvBasketAppModel', 'EvBasket.Model');
App::uses('CakeSession', 'Model/Datasource');
/**
 * Basket Model
 *
 * @property BasketItem $BasketItem
 * @property BasketTotal $BasketTotal
 */
class Basket extends EvBasketAppModel {

/**
 * The Associations below have been created with all possible keys, those that are not needed can be removed
 */
/**
 * hasMany associations
 *
 * @var array
 */
	public $hasMany = array(
		'BasketItem' => array(
			'className' => 'EvBasket.BasketItem',
			'dependent' => true,
			'cascade' => true
		),
		'BasketTotal' => array(
			'className' => 'EvBasket.BasketTotal',
			'dependent' => true,
			'cascade' => true
		),
		'BasketData' => array(
			'className' => 'EvBasket.BasketData',
			'dependant' => true,
			'cascade' => true
		)
	);

	public $validate = array(
		'hash' => array(
			'unique' => array(
				'rule' => 'isUnique'
			)
		)
	);

/**
 * convience method to get the full basket
 *
 * @param 	string 	$hash 			The hash identifier
 * @param 	array 	$itemContain
 * @return  array|null
 */
	public function getFullBasket($hash, $itemContain, $currencyId = null) {
		if (!$currencyId) {
			$currencyId = CakeSession::read('EvCurrency.currencyId');
		}

		$basket = $this->find(
			'first',
			array(
				'conditions' => array(
					'Basket.hash' => $hash,
					'Basket.currency_id' => $currencyId
				),
				'contain' => array(
					'BasketItem' => $itemContain,
					'BasketTotal' => array(
						'order' => 'BasketTotal.is_grand_total ASC, BasketTotal.sequence ASC'
					),
					'BasketData'
				)
			)
		);

		if (! empty($basket)) {

			$this->updateLastActivity($basket);

			if (
				(bool)strpos(json_encode($itemContain), 'CustomFields') === true &&
				! empty($basket['BasketItem'])
			) {
				$CustomField = EvClassRegistry::init('EvCustomFields.CustomFieldData');

				foreach ($basket['BasketItem'] as $key => $item) {
					$basket['BasketItem'][$key] = $CustomField->formatCfArray($item);
				}
			}

			$basket = $this->formatItemsAndTotals($basket);

			$event = new CakeEvent('EvBasket.Model.Basket.formattedBasket', $this, ['basket' => $basket]);
			$this->getEventManager()->dispatch($event);

			if (!empty($event->result)) {
				$basket = $event->result['basket'];
			}
		}

		return $basket;
	}

/**
 * Update a baskets last_activity field to the current datetime.
 * Called on every request to get a full basket so we can track activity
 * @param  $basket
 * @return bool|array See Model::save() False on failure or an array of model data on success.
 */
	public function updateLastActivity($basket) {
		if (empty($basket['Basket']['id'])) {
			return false;
		}

		$now = date("Y-m-d H:i:s");

		// Only update last_activity if it's been more than a minute since it was last updated.
		// Prevents multiple queries per request etc.
		if (
			empty($basket['Basket']['last_activity']) ||
			strtotime($basket['Basket']['last_activity']) < strtotime($now) - MINUTE
		) {
			$this->id = $basket['Basket']['id'];

			return $this->saveField('last_activity', $now);
		}

		return false;
	}

/**
 * format the items and totlas keys to be more easily accessbile
 *
 * @param 	array 	$basket 	The basket array
 * @return 	array 	$basket 	The formated basket array
 */
	public function formatItemsAndTotals($basket) {
		$basket['BasketItem'] = Hash::combine(
			$basket['BasketItem'],
			array(
				'%s.%s',
				'{n}.model',
				'{n}.model_id'
			),
			'{n}'
		);

		// calculate tax for each row item
		$basket['BasketItem'] = $this->BasketItem->calculateTax($basket['BasketItem']);

		$basket['BasketTotal'] = Hash::combine(
			$basket['BasketTotal'],
			'{n}.id',
			'{n}'
		);

		return $basket;
	}

/**
 * check to see if a basket exists at the given hash
 *
 * @param 	string 	$hash 	hash identifier to look for
 * @return 	bool
 */
	public function checkExists($hash, $currencyId = null) {
		if (!$currencyId) {
			$currencyId = CakeSession::read('EvCurrency.currencyId');
		}

		return (bool)$this->find(
			'count',
			array(
				'conditions' => array(
					'Basket.hash' => $hash,
					'Basket.currency_id' => $currencyId
				),
				'callbacks' => false
			)
		);
	}

/**
 * create a basket row with the given hash
 *
 * @param 	string 	$hash 	hash identifier to save basket ot
 * @return 	bool
 */
	public function createBasket($hash, $currencyId = null, $userId = null) {
		if (empty($currencyId)) {

			$currencyId = CakeSession::read('EvCurrency.currencyId');
			if (empty($currencyId)) {
				$currencyId = CakeSession::read('EvCurrency.id');
			}
		}

		$data = array(
			'Basket' => array(
				'hash' => $hash,
				'currency_id' => $currencyId
			)
		);

		// If set link the user to the basket so the basket can be used across logins
		if ($userId != null) {
			$data['Basket']['user_id'] = $userId;
		}

		$this->clear();

		$result = $this->save($data);

		if ($result !== false) {
			return true;
		}

		return $result;
	}

	/**
	 * Takes two basket hash's, gets the items from the second basket and moves them to the first
	 * @param  string $masterBasketHash    The primary basket that the other basket will be merged into
	 * @param  string $secondaryBasketHash The basket who's items will be merged into the primary basket
	 * @param  bool $removeSecondaryBasket Defaults to false, if set to try it will delete the basket
	 * @return string                      Hash of the new basket
	 */
	public function mergeBaskets($masterBasketHash, $secondaryBasketHash, $removeSecondaryBasket = false) {
		// Get the items in the secordary basket
		$masterBasket = $this->getFullBasket($masterBasketHash, array());
		$secondaryBasket = $this->getFullBasket($secondaryBasketHash, array());

		// If neither basket exists there is nothing to merge
		if (!$masterBasket && !$secondaryBasket) {
			return false;
		} elseif (!$masterBasket) {
			// Just one basket so return that
			return $secondaryBasketHash;
		} elseif (!$secondaryBasket) {
			// Just have one valid basket which is master
			return $masterBasketHash;
		}

		// Got to here then both baskets must be valid baskets, time to merge.

		// If the baskets don't match currency return false
		if ($masterBasket['Basket']['currency_id'] != $secondaryBasket['Basket']['currency_id']) {
			return false;
		}

		$this->BasketItem->updateAll(
			array(
				'BasketItem.basket_id' => $masterBasket['Basket']['id']
			),
			array(
				'BasketItem.basket_id' => $secondaryBasket['Basket']['id']
			)
		);

		if ($removeSecondaryBasket) {
			$this->delete($secondaryBasket['Basket']['id']);
		}

		return $masterBasketHash;
	}

/**
 * Updates the supplied basket to associate it with the requested user
 *
 * @param string $basketHash The hash of the basket row to update
 * @param string $userId The unique id of the user
 * @return bool
 */
	public function assignUserToBasket($basketHash = '', $userId = '') {
		// Check if there is another user associated with the hash, if they are sharing a computer
		// there is a possibility that they have a left over cookie hash that isn't theirs so we
		// check and return false if another user has it
		$basketExists = $this->find('first', [
			'fields' => ['id'],
			'conditions' => [
				'hash' => $basketHash,
				'user_id <>' => '',
				'user_id <>' => $userId
			],
			'callbacks' => false
		]);

		// This hash already belongs to another user
		if (!empty($basketExists['Basket']['id'])) {
			return false;
		}

		// Assign the basket to the user
		return $this->updateAll(
			['user_id' => $userId],
			['hash' => $basketHash, 'user_id' => '']
		);
	}

/**
 * Assign a basket from one user to another.
 *
 * @param int $id            The id of the basket to reassign.
 * @param int $currentUserId The id of the user the basket currently belongs to.
 * @param int $newUserId     The id of the user the basket should be reassigned to.
 * @return bool
 */
	public function reassignBasketToUser($id, $currentUserId, $newUserId) {
		$basket = $this->find(
			'first',
			[
				'fields' => [
					'Basket.id',
				],
				'conditions' => [
					'Basket.id' => $id,
					'Basket.user_id' => $currentUserId
				],
			]
		);

		if (empty($basket)) {
			return false;
		}

		// Remove any baskets with the new user Id to prevent them having two baskets in the database
		$this->deleteAll([
			'user_id' => $newUserId
		]);

		// Reassign the basket to the new user.
		$basket['Basket']['user_id'] = $newUserId;
		if (!$this->save($basket)) {
			return false;
		}

		return $this->updateLastActivity($basket);
	}
}
