<?php

App::uses('CakeTime', 'Utility');

/**
 * Static lib to share code between helper and components
 */
class BasketLib {

	/**
	 * given the basket item array row
	 * return the text to show on the basket
	 *
	 * @param 	array 	$BasketItem 	The basket item row from the DB
	 * @return 	string
	 */
	public static function getItemName($BasketItem) {
		// If a name has been set on the basket item use this.
		if (!empty($BasketItem['name'])) {
			return $BasketItem['name'];
		}

		// Try to find the model name
		$modelName = Hash::get($BasketItem, 'model');
		if (empty($modelName)) {
			return 'Unknown';
		}

		// get the config so we know where to find the data for this row
		$config = Configure::read('EvBasket.BasketItemNamePath');
		if (empty($config[$modelName])) {

			// Attempt to get the basket row text from the model (look for a method called basketRowName)
			$itemModel = EvClassRegistry::init($modelName);
			if (method_exists($itemModel, 'basketRowName')) {
				return $itemModel->basketRowName($BasketItem[$modelName]['id']);
			}

			return 'Unknown';
		}
		// reposition array to make it easier to work
		$config = $config[$modelName];

		$separator = ' ';
		if (! empty($config['separator'])) {
			$separator = $config['separator'];
			unset($config['separator']);
		}

		if (! is_array($config)) {
			$config = array($config);
		}

		$item = array();
		foreach ($config as $path) {
			$value = Hash::get($BasketItem, $path);

			if (! empty($value) && stripos(implode('', $item), $value) === false) {
				$item[] = $value;
			}
		}

		if (empty($item)) {
			return 'Unknown';
		}

		return implode($separator, $item);
	}

	/**
	 * clear old baskets by checking grand total row for each one
	 *
	 * @param 	int 	$olderThan 			Delete any baskets older than time() - $olderThan
	 * @return 	int 	$numDeletedItems 	Number of items deleted
	 */
	public static function cleanBaskets($olderThan = WEEK) {
		$deleteDate = CakeTime::toServer(
			(time() - $olderThan),
			Configure::read('Config.timezone')
		);

		$Basket = EvClassRegistry::init('EvBasket.Basket');
		$Basket->displayField = 'id';
		$basketIds = $Basket->find(
			'list',
			[
				'joins' => [
					[
						'table' => 'ev_basket_basket_totals',
						'alias' => 'BasketTotal',
						'conditions' => [
							'BasketTotal.basket_id = Basket.id'
						],
						'type' => 'left'
					],
				],
				'conditions' => [
					'Basket.modified <=' => $deleteDate,
					'OR' => [
						'BasketTotal.id IS NULL',
						'BasketTotal.modified <=' => $deleteDate
					]
				],
				'group' => 'Basket.id'
			]
		);

		$deleted = 0;
		if (! empty($basketIds)) {
			foreach ($basketIds as $id) {
				if ($Basket->delete($id)) {
					$deleted++;
				}
			}
		}

		return $deleted;
	}

	/**
	 * delete a basket
	 *
	 * @param 	int 	$basketId 	The basket to remove
	 * @return 	bool
	 */
	public static function deleteBasket($basketId) {
		$Basket = EvClassRegistry::init('EvBasket.Basket');

		return $Basket->delete($basketId);
	}

/**
 * Queries the basket table to find the abandoned ones that need a gentle email reminder to give us ya money
 * @param  array  $passed - any additional params
 * @return Array
 */
	public static function fetchDropoutBaskets($passed = array()) {
		$Basket = EvClassRegistry::init('EvBasket.Basket');

		$now = new DateTime();
		$now = $now->format("Y-m-d H:i:s");

		// Get the minimum inactivity and minimum time since they last ordered interval values
		$minInactivityTime = Configure::check('EvBasket.basketDropout.minInactivityTime') ? Configure::read('EvBasket.basketDropout.minInactivityTime') : "2 hour";
		$minLastOrderedTime = Configure::check('EvBasket.basketDropout.minLastOrderedTime') ? Configure::read('EvBasket.basketDropout.minLastOrderedTime') : "1 week";

		/*
		 * The following query fetches out basket dropouts which meet the following criteria:
		 *  - Basket has an associated user (with an email address too ofc!)
		 * 	- User hasn't completed the checkout process within the last week (based on their email address)
 		 * 	- They haven't been active on the site for more than 2 hours (basket last_activity)
		 */
		$params = array(
			'conditions' => array(
				'Basket.last_activity IS NOT NULL',
				// Last activity was longer than 2 hours ago
				'Basket.last_activity < DATE_SUB("' . $now . '" , INTERVAL ' . $minInactivityTime . ')',
				'Basket.user_id IS NOT NULL',
				'User.email IS NOT NULL',
				// Not already linked with an order (under a week old)
				'Order.id IS NULL',
				// Haven't emailed them yet for this basket
				'BasketDropout.id IS NULL'
				// 'OR' => array(
				// 	'BasketDropout.id IS NULL',
				// 	'BasketDropout.modified < DATE_SUB(NOW(), INTERVAL 1 week)'
				// )
			),
			'fields' => array(
				'Basket.*',
				'User.*',
				'BasketDropout.*',
				'Order.id'
			),
			'joins' => array(
				array(
					'table' => 'ev_basket_basket_dropouts',
					'alias' => 'BasketDropout',
					'type' => 'LEFT',
					'conditions' => array(
						'BasketDropout.basket_hash = Basket.hash'
					)
				),
				array(
					'table' => 'users',
					'alias' => 'User',
					'conditions' => array(
						'Basket.user_id = User.id',
					)
				),
				array(
					'table' => 'ev_checkout_orders',
					'alias' => 'Order',
					'type' => 'LEFT',
					'conditions' => array(
						'Order.email = User.email',
						// Bring out orders linked to this email less than a week old.
						// So if a user created an order over a week ago and is back with a new basket we still want to email them
						// This will bring null if no orders with this email are found from the past week
						'Order.created > DATE_SUB("' . $now . '", INTERVAL ' . $minLastOrderedTime . ')'
					)
				)
			),
			'group' => 'Basket.id'
		);

		$baskets = $Basket->find('all', array_merge_recursive($params, $passed));

		return $baskets;
	}

/**
 * Builds the url to be put inside the emails that creates the basket and redirects them to the checkout screen
 * @param  $dropout - dropout data
 * @return String
 */
	public static function buildDropoutUrl($dropout) {
		$domain = Configure::read('App.fullBaseUrl');

		return $domain . '/ev_basket/basket_dropouts/view/' . $dropout['BasketDropout']['token'] . '?utm_source=basketdropout&utm_medium=email';
	}

}
