<?php

App::uses('EvBasketAppController', 'EvBasket.Controller');

class BasketController extends EvBasketAppController {

/**
 * If editing an order allow discounts to be applied if they already exist
 */
	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(
			[
				'index',
				'add',
				'add_multiple',
				'remove',
				'update',
				'test',
			]
		);
	}

/**
 * Called after the controller action is run, but before the view is rendered. You can use this method
 * to perform logic or set view variables that are required on every request.
 *
 * @return void
 * @link http://book.cakephp.org/2.0/en/controllers.html#request-life-cycle-callbacks
 */
	public function beforeRender() {
		parent::beforeRender();

		// if the session is set to omit over VAT then set a view template to
		// show ex VAT pricings on basket templates. The session variable is
		// set in EvCheckout.PreStageComponent::processBasketVATFromAddress
		// once the customer have chosen / entered their delivery address
		if (CakeSession::check('EvCheckout.omitOrderVat')) {
			$omitOrderVat = CakeSession::read('EvCheckout.omitOrderVat');
			$this->set(compact('omitOrderVat'));
		}
	}

/**
 * main basket page
 *
 */
	public function index() {
		$this->view = 'EvBasket./Fallbacks/Basket/index';

		$pageId = Configure::read('EvBasket.pageId');

		// check if we have a specific content page set
		// if so - set the template
		if (! empty($pageId)) {
			$pageData = $this->assignPage($pageId);

			if (! empty($pageData)) {
				$this->view = $this->Tpl->getTemplate($pageData, 'Page');
			}
		}

		$basketData = $this->BasketManager->getBasket();
		if (empty($basketData['BasketItem'])) {
			//Remove the discount completely from the basket
			$this->BasketManager->deleteData('discount_code');
			$this->BasketManager->deleteData('voucher_code');
			$this->BasketManager->deleteTotalRow(Configure::read('EvBasket.labels.discount'));
			$this->BasketManager->deleteTotalRow(Configure::read('EvBasket.labels.voucher'));
		}
		$this->BasketManager->deleteTotalRow(Configure::read('EvBasket.labels.delivery'));
		$this->BasketManager->rebuildTotals();
		$basketData = $this->BasketManager->getBasket();

		$this->set(
			'basket',
			$basketData
		);
	}

/**
 * delete an item from your basket (if valid)
 * and redirect back to previous page
 */
	public function remove($id) {
		// given the BasketItem row ID and find the model data
		$basketItem = $this->BasketManager->getBasketItem($id);

		$itemData = array(
			'model' => $basketItem['model'],
			'model_id' => $basketItem['model_id']
		);

		// call the delete
		if ($this->BasketManager->deleteItem($itemData)) {
			$this->Flash->success(
				Configure::read('EvBasket.flash.remove.success'),
				array(
					'key' => 'basket-flash'
				)
			);
		} else {
			$this->Flash->fail(
				Configure::read('EvBasket.flash.remove.fail'),
				array(
					'key' => 'basket-flash'
				)
			);
		}

		// process redurect
		if (Configure::read('EvBasket.redirects.delete') === 'referer') {
			$redirect = $this->referer();
		} else {
			$redirect = Configure::read('EvBasket.redirects.delete');
		}

		$this->redirect(
			$redirect
		);
	}

/**
 * loop and update the basket items
 *
 */
	public function update() {
		// process redirect
		if (Configure::read('EvBasket.redirects.update') === 'referer') {
			$redirect = $this->referer();
		} else {
			$redirect = Configure::read('EvBasket.redirects.update');
		}

		// check it's a post request and that we have data
		if (! $this->request->is('post') || empty($this->request->data)) {
			$this->redirect($this->referer());
		}

		//format the array and pass to update
		$toUpdate = Hash::combine(
			$this->request->data,
			'{n}.BasketItem.id',
			'{n}.BasketItem'
		);

		if ($this->BasketManager->updateItem($toUpdate)) {
			$this->Flash->success(
				Configure::read('EvBasket.flash.update.success'),
				array(
					'key' => 'basket-flash'
				)
			);
		} else {
			$this->Flash->fail(
				Configure::read('EvBasket.flash.update.fail'),
				array(
					'key' => 'basket-flash'
				)
			);
		}

		// redirect
		$this->redirect(
			$redirect
		);
	}

/**
 * add a new item to the basket
 */
	public function add() {
		// process redirect
		if (Configure::read('EvBasket.redirects.add') === 'referer') {
			$redirect = $this->referer();
		} else {
			$redirect = Configure::read('EvBasket.redirects.add');
		}

		// check it's a post request and that we have data
		if (! $this->request->is('post') || empty($this->request->data['BasketItem'])) {
			$this->redirect($this->referer());
		}

		// check we have all the fields needed to add an item to the basket
		if ($this->BasketManager->checkAddItem($this->request->data['BasketItem']) === false) {
			$this->redirect($this->referer());
		}

		// setup the data
		$item = $this->BasketManager->processNewItem($this->request->data['BasketItem']);

		if (!empty($item)) {
			// add to the basket
			if ($this->BasketManager->addItem($item)) {
				$this->Flash->success(
					Configure::read('EvBasket.flash.add.success'),
					array(
						'key' => 'basket-flash'
					)
				);
			} else {
				$this->Flash->fail(
					Configure::read('EvBasket.flash.add.fail'),
					array(
						'key' => 'basket-flash'
					)
				);
			}

		} else {
			$this->Flash->fail(
				Configure::read('EvBasket.flash.add.fail'),
				array(
					'key' => 'basket-flash'
				)
			);
		}

		// redirect
		$this->redirect(
			$redirect
		);
	}

/**
 * Add multiple new items to the basket.
 */
	public function add_multiple() {
		// process redirect
		if (Configure::read('EvBasket.redirects.add_multiple') === 'referer') {
			$redirect = $this->referer();
		} else {
			$redirect = Configure::read('EvBasket.redirects.add_multiple');
		}

		// check it's a post request and that we have data
		if (! $this->request->is('post') || empty($this->request->data['BasketItem'])) {
			$this->redirect($this->referer());
		}

		$items = $this->BasketManager->processNewItem($this->request->data['BasketItem']);

		$items = $this->BasketManager->checkAddItems($items);
		if (empty($items['passed'])) {
			$this->Flash->fail(
				Configure::read('EvBasket.flash.add_multiple.fail'),
				array(
					'key' => 'basket-flash'
				)
			);
			$this->redirect($this->referer());
		}

		// add to the basket
		if ($this->BasketManager->addItem($items['passed'])) {
			$this->Flash->success(
				Configure::read('EvBasket.flash.add_multiple.success'),
				array(
					'key' => 'basket-flash'
				)
			);

			if (!empty($items['failed'])) {
				$this->Flash->warning(
					Configure::read('EvBasket.flash.add_multiple.warning'),
					array(
						'key' => 'basket-flash'
					)
				);
			}
		} else {
			$this->Flash->fail(
				Configure::read('EvBasket.flash.add_multiple.fail'),
				array(
					'key' => 'basket-flash'
				)
			);
		}

		$this->redirect(
			$redirect
		);
	}
}
