<?php

App::uses('AppShell', 'Console/Command');
App::uses('BasketLib', 'EvBasket.Lib');

/**
 * This shell is used to handle sending out basket dropout emails
 * What defines a basket dropout:
 *  - Basket has an associated user (with an email address too ofc!)
 * 	- User hasn't completed the checkout process within the last week (based on their email address)
 * 	- They haven't been active on the site for more than 2 hours (basket last_activity)
 */

class DropoutEmailsShell extends AppShell {

/**
 * Starts up the Shell and displays the welcome message.
 * Allows for checking and configuring prior to command or main execution
 *
 * @return void
 * @link http://book.cakephp.org/2.0/en/console-and-shells.html#Shell::startup
 */
	public function startup() {
		parent::startup();

		$this->Dropout = EvClassRegistry::init('EvBasket.BasketDropout');
	}

	public function run() {
		$baskets = $this->_fetchBasketDropouts();

		if (empty($baskets)) {
			$this->out('No abandoned baskets have been found.');
			return;
		}

		foreach ($baskets as $basket) {
			$dropout = $this->_saveDropout($basket);

			if (!empty($dropout['BasketDropout'])) {
				$this->_sendEmail($dropout, $basket);
			}
		}
	}

/**
 * Calls a basket library method to get the baskets that have been abandoned.
 * @return Array
 */
	protected function _fetchBasketDropouts() {
		return BasketLib::fetchDropoutBaskets();
	}

/**
 * Gets the dropout url from a basket lib method
 * @param  $dropout - dropout data
 * @return String URL
 */
	protected function _buildDropoutUrl($dropout) {
		return BasketLib::buildDropoutUrl($dropout);
	}

/**
 * Send the email containing the link for rebuilding the basket
 * @param  $dropout - record from the basket_dropouts table
 * @return bool true on email success
 */
	protected function _sendEmail($dropout, $basket) {
		$this->out('sending email..');

		// create the URL
		$domain = Configure::read('App.fullBaseUrl');

		$dropoutUrl = $this->_buildDropoutUrl($dropout);

		if (Configure::check('EvBasket.basketDropout.email_subject')) {
			$subject = Configure::read('EvBasket.basketDropout.email_subject');
		} else {
			$subject = Configure::read('SiteSetting.general.site_title') . ' - Complete Your Order';
		}

		// send the email
		$Email = new CustomEmail();

		$Email->from(Configure::read('SiteSetting.general.admin_email'), Configure::read('SiteSetting.general.site_title'))
			->to($dropout['BasketDropout']['email'])
			->bcc(array(
				'notifications@evoluted.net',
			))
			->subject($subject)
			->template('EvBasket.Dropouts/basketDropout')
			->viewVars(array(
				'body' => Configure::read('EvBasket.basketDropout.email_body'),
				'data' => $dropout,
				'basket' => $basket,
				'site' => $site,
				'url' => $dropoutUrl
			))
			->domain($domain)
			->sendAs = 'html';

		if ($Email->send() !== false) {
			$this->out('email sent successfully');
			return true;
		} else {
			$this->out('email unable to send.');

			// email failed so remove record
			$this->Dropout->delete($dropout['BasketDropout']['id']);

			return false;
		}
	}

/**
 * Log the basket dropout in the database
 * @param $basket
 * @return Array - the new basket_dropout record
 */
	protected function _saveDropout($basket) {
		$this->out('saving dropout..');

		$data = array(
			'user_id' => $basket['User']['id'],
			'email' => $basket['User']['email'],
			'basket_hash' => $basket['Basket']['hash'],
			'token' => $this->Dropout->token()
		);

		if (!empty($basket['BasketDropout']['id'])) {
			// An id may already be available if the user clicked an email basketdropout link but never completed the checkout
			$data['id'] = $basket['BasketDropout']['id'];
		}

		$result = $this->Dropout->save($data);

		return $result;
	}

}
