<?php
App::uses('EvBasketAppModel', 'EvBasket.Model');
/**
 * BasketItem Model
 *
 * @property Basket $Basket
 * @property Model $Model
 */
class BasketItem extends EvBasketAppModel {

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'basket_id' => array(
			'numeric' => array(
				'rule' => array('numeric'),
				'on' => 'create'
			),
		),
		'model' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'on' => 'create'
			),
		),
		'model_id' => array(
			'numeric' => array(
				'rule' => array('numeric'),
				'on' => 'create'
			),
		),
		'quantity' => array(
			'numeric' => array(
				'rule' => array('numeric')
			),
		)
	);

	//The Associations below have been created with all possible keys, those that are not needed can be removed

	/**
	 * belongsTo associations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'Basket' => array(
			'className' => 'EvBasket.Basket'
		)
	);

	/**
	 * constructor - setup the dynamic relationships
	 *
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		$relationships = Configure::read('EvBasket.BasketItemBelongsTo');
		if (! empty($relationships)) {
			$belongsTo = array();
			foreach ($relationships as $key => $value) {
				$belongsTo[$key] = $value;
			}

			if (! empty($belongsTo)) {
				$this->bindModel(
					array(
						'belongsTo' => $belongsTo
					),
					false
				);
			}
		}
	}

	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		if ($primary === true && (isset($results[0]['model']) && isset($results[0]['model_id']))) {
			$results = Hash::combine(
				$results,
				array(
					'%s.%s',
					'{n}.BasketItem.model',
					'{n}.BasketItem.model_id'
				),
				'{n}'
			);
		}

		return $results;
	}

	/**
	 * calculate the tax for each basket item
	 *
	 * @param 	array 	$BasketItems 		Array of basket items
	 * @return 	arrayh 	$calculatedItems	Array of basket items with calculated tax
	 */
	public function calculateTax($basketItems) {
		foreach ($basketItems as $key => $item) {
			if (! empty($item['tax_rate'])) {
				$basketItems[$key]['unit_tax_amount'] = number_format(($item['tax_rate'] / 100) * $item['unit_price'], 2, '.', '');
				$basketItems[$key]['row_tax_amount'] = $basketItems[$key]['unit_tax_amount'] * $item['quantity'];
				$basketItems[$key]['unit_price_incTax'] = round(($item['unit_price'] + $basketItems[$key]['unit_tax_amount']), 2);
				$basketItems[$key]['row_total_incTax'] = $basketItems[$key]['unit_price_incTax'] * $item['quantity'];
			} else {
				$basketItems[$key]['unit_price_incTax'] = $item['unit_price'];
				$basketItems[$key]['unit_tax_amount'] = 0;
				$basketItems[$key]['row_tax_amount'] = 0;
				$basketItems[$key]['row_total_incTax'] = $item['row_total'];
			}
		}

		return $basketItems;
	}

	/**
	 * add an item to the basket
	 *
	 * @param 	int 		$basketId 		ID of the basket row
	 * @param 	array 		$item 			Array with two elements of model and model_id
	 * @param 	decimal 	$unitPrice 		Unit Price of the item
	 * @param 	int 		$quantity		Number of items to add
	 * @param 	decimal		$taxRate 		The tax rate to update
	 * @return 	bool
	 */
	public function addItem($basketId, $item, $unitPrice, $quantity, $taxRate = 0) {
		$this->clear();

		$this->set(
			array(
				'basket_id' => $basketId,
				'model' => $item['model'],
				'model_id' => $item['model_id'],
				'unit_price' => $unitPrice,
				'tax_rate' => $taxRate,
				'quantity' => $quantity,
				'row_total' => ($unitPrice * $quantity)
			)
		);

		return (bool)$this->save();
	}

	/**
	 * update an item in the basket
	 *
	 * @param 	array 		$basketItem 	The existing BasketItem Row
	 * @param 	int 		$quantity		Number of items to add
	 * @param 	decimal 	$unitPrice 		Unit Price of the item
	 * @param 	decimal		$taxRate 		The tax rate to update
	 * @return 	bool
	 */
	public function updateItem($basketItem, $newQuantity, $unitPrice = null, $taxRate = 0) {
		$this->clear();

		// if we're not updating the unit price use old one
		if (empty($unitPrice)) {
			$unitPrice = $basketItem['unit_price'];
		}

		// if we're not updating the unit price use old one
		if (empty($taxRate)) {
			$taxRate = $basketItem['tax_rate'];
		}

		$this->set(
			array(
				'id' => $basketItem['id'],
				'unit_price' => $unitPrice,
				'tax_rate' => $taxRate,
				'quantity' => $newQuantity,
				'row_total' => ($unitPrice * $newQuantity)
			)
		);

		return (bool)$this->save();
	}

	/**
	 * delete an item from the basket
	 *
	 * @param 	array 		$basketItem 	The existing BasketItem Row
	 * @return 	bool
	 */
	public function deleteItem($basketItem) {
		$this->clear();

		return $this->delete($basketItem['id']);
	}
}
