<?php

App::uses('AppComponent', 'Controller/Component');
App::uses('CakeText', 'Utility');
App::uses('BasketLib', 'EvBasket.Lib');

class BasketComponent extends AppComponent {

	/**
	 * the users hash
	 */
	public $hash = null;

	/**
	 * copy of the basket
	 */
	public $basketCache = null;

	/**
	 * Basket model - set via Component Settings
	 */
	public $Basket = null;

	/**
	 * Basket Item model - set via Component Settings
	 */
	public $BasketItem = null;

	/**
	 * Basket Total model - set via Component Settings
	 */
	public $BasketTotal = null;

	/**
	 * setup the cookie component
	 *
	 */
	public function initialize(Controller $controller) {
		parent::initialize($controller);

		if (empty($this->_controller->BasketCookie)) {
			$this->_controller->BasketCookie = $this->_controller->loadComponent('EvBasket.BasketCookie');
		}
	}

	/**
	 * setup a hash or retrieve a hash from the cookie
	 *
	 * @return string  	Basket hash identifier
	 */
	public function setupHash() {
		if (! empty($this->hash)) {
			return $this->hash;
		}

		if ($this->_controller->BasketCookie->check('basketHash')) {
			return $this->hash = $this->_controller->BasketCookie->read('basketHash');
		}

		$this->hash = $this->generateHash();
		$this->_controller->BasketCookie->write(
			'basketHash',
			$this->hash
		);

		return $this->hash;
	}

	/**
	 * given the hash, setup or return the existing basket
	 *
	 * @param 	string 	$hash 	Basket Hash
	 * @return 	bool 	$result True if a basket was created, false if one already exists
	 */
	public function setupBasket($hash) {
		if (! $this->Basket->checkExists($hash)) {
			return $this->Basket->createBasket(
				$hash
			);
		}

		return false;
	}

	/**
	 * build the basket item contain list
	 *
	 * @param 	array  	$itemContain 	Value from Configure::read('EvBasket.BasketItemContains')
	 * @param 	array  	$relationships 	Value from Configure::read('EvBasket.BasketItemBelongsTo')
	 * @return 	array
	 */
	public function buildItemContain($itemContain, $relationships) {
		if (empty($itemContain)) {
			$itemContain = array();

			foreach ($relationships as $key => $value) {
				if (is_array($value)) {
					$itemContain[] = $key;
				} else {
					$itemContain[] = $value;
				}
			}
		}

		return $itemContain;
	}

/**
 * generate a hash for the basket
 * extracted into another method for easier unit testing
 *
 * @return string
 */
	public function generateHash() {
		return CakeText::uuid() . '-' . md5($this->_controller->request->clientIp() . rand());
	}

	/**
	 * build the basket item key
	 *
	 * @param 	array 			$item 	Array with two elements of model and model_id
	 * @return 	string|bool 	$key 	Item basket key Model.Model_id format / false on fail
	 */
	public function buildItemKey($item) {
		if (empty($item['model']) || empty($item['model_id'])) {
			return false;
		}

		return $item['model'] . '.' . $item['model_id'];
	}

	/**
	 * find item key
	 *
	 * @param 	string|int|array 	$key 	Either Basket row ID / compiled basket key string / Array with two elements of model and model_id
	 * @return  string
	 */
	public function getItemKey($key) {
		// set the key as the itemKey as a fallback
		$itemKey = $key;

		if (is_array($key)) {
			// It's an array, try and build it
			$itemKey = $this->buildItemKey($key);
		} elseif (is_numeric($key) && ! empty($this->basketCache)) {
			// it's and id number - try to find the itemKey
			$keysArray = Hash::combine(
				$this->basketCache,
				'BasketItem.{s}.id',
				array(
					'%s.%s',
					'BasketItem.{s}.model',
					'BasketItem.{s}.model_id'
				)
			);

			if (empty($keysArray[$key])) {
				return false;
			}

			$itemKey = $keysArray[$key];
		}

		return $itemKey;
	}

	/**
	 * given the item key, check to see if it exists already in the basket
	 *
	 * @param 	string 		$key 		Item basket key Model.Model_id format
	 * @param 	array|null 	$basket 	Optional basket to search in - if null will use $this->basket
	 * @return 	bool
	 */
	public function basketItemExists($key, $basket = null) {
		if (! empty($basket)) {
			return ! empty($basket['BasketItem'][$key]);
		}

		return ! empty($this->basketCache['BasketItem'][$key]);
	}

	/**
	 * common method to call out to rebuild the totals
	 *
	 * @todo Refactor to make it more testable
	 * @return bool
	 */
	public function rebuildTotals() {
		$this->getBasket(true);

		// process subtotal
		// process Tax
		$subtotal = 0;
		$tax = 0;
		$BasketItems = $this->basketCache['BasketItem'];

		foreach ($BasketItems as $BasketItem) {
			$subtotal += number_format($BasketItem['row_total'], 2, '.', '');
			$tax += number_format($BasketItem['row_tax_amount'], 2, '.', '');
		}

		// check for a discount row and amend tax
		$discount = $this->findTotalRow(Configure::read('EvBasket.labels.discount'));
		$discount['amount'] = -$discount['amount'];

		if (
			(float)$subtotal > 0 &&
			(float)$tax > 0 &&
			! empty($discount['amount']) &&
			(float)$discount['amount'] > 0
		) {
			$basketVatRate = $tax / $subtotal;
			$discountVat = number_format(($discount['amount'] * $basketVatRate), 2, '.', '');
			$tax -= $discountVat;
		}

		// check for delivery placeholder, only add if we haven't added delivery already
		$deliveryPlaceholder = Configure::read('EvBasket.deliveryPlaceholder');
		$delivery = $this->findTotalRow(Configure::read('EvBasket.labels.delivery'));

		if ($deliveryPlaceholder !== false && empty($delivery['id'])) {
			// add / update the delivery row
			$this->manageTotalRow(
				Configure::read('EvBasket.labels.delivery'),
				$deliveryPlaceholder,
				20,
				false
			);

			$deliveryAmount = $deliveryPlaceholder;
		}

		if (! isset($deliveryAmount) && isset($delivery['amount'])) {
			$deliveryAmount = $delivery['amount'];
		}

		// check for delivery taxation
		$deliveryTax = Configure::read('EvBasket.deliveryTax');
		if ($deliveryTax !== false && $deliveryTax > 0 && isset($deliveryAmount)) {
			// check with ash - add 20% tax rate
			$delTax = (($deliveryTax / 100) * $deliveryAmount);
			$tax += $delTax;
		}

		// add / update the subtotal row
		$this->manageTotalRow(
			Configure::read('EvBasket.labels.subtotal'),
			$subtotal,
			10,
			true
		);

		// add / update the tax row
		$this->manageTotalRow(
			Configure::read('EvBasket.labels.tax'),
			$tax,
			30,
			false
		);

		// call out to recompile all the basket totals
		$totalResult = $this->BasketTotal->recompileTotals($this->basketCache['Basket']['id']);

		if ($totalResult === true) {
			// dispatch totals recompiled
			$this->_controller->getEventManager()->dispatch(
				new CakeEvent('EvBasket.Component.Basket.postTotalRecompile', $this, array(
					'basketId' => $this->basketCache['Basket']['id']
				))
			);

			// rebuild the basket cache
			$this->getBasket(true);

			return true;
		}

		return false;
	}

	/**
	 * if we're updating an item - check if we actually want to update multi rows in one go
	 *
	 * @param 	array 		$item 		Array with two elements of model and model_id / or multidimensional array of model, model_id, quantity, unitPrice
	 * @param 	int 		$quantity	Amount of items to add
	 * @param 	decimal 	$unitPrice 	Unit price of the item
	 * @param 	decimal		$taxRate 	The tax rate to update
	 * @return  array 		$toUpdate 	Array of items to update
	 */
	public function buildUpdateList($item, $quantity, $unitPrice, $taxRate) {
		if (is_numeric(key($item))) {
			return $item;
		}

		if ($quantity < 0) {
			$quantity = 0;
		}

		$item['quantity'] = $quantity;
		$item['unitPrice'] = $unitPrice;
		$item['taxRate'] = $taxRate;

		return array(
			$item
		);
	}

	/**
	 * when adding an item, check we have all the data needed
	 *
	 * @param 	array 	$basketItem 	Array of data - array of model, model_id, quantity, unitPrice
	 * @return 	bool
	 */
	public function checkAddItem($item) {
		$Model = EvClassRegistry::init($item['BasketItem']['model']);

		if (
			! empty($item['BasketItem']['model']) &&
			! empty($item['BasketItem']['model_id']) &&
			! empty($item['BasketItem']['quantity']) &&
			$item['quantity'] > 0 &&
			method_exists($Model, 'getUnitPrice')
		) {
			// check we have enough stock
			if (! $this->hasEnoughStock(EvClassRegistry::init($item['BasketItem']['model']), $item)) {
				$this->_controller->Flash->fail(
					Configure::read('EvInventory.notEnoughStockMessage')
				);
				return false;
			}

			return true;
		}

		return false;
	}

	/**
	 * check to see if the item has enough stock (assuming inventories are enabled)
	 *
	 * @param 	object 	$Model 	The model object of the item we are checking
	 * @param 	array 	$item 	The basket item
	 * @return 	bool
	 */
	public function hasEnoughStock($Model, $item) {
		if (CakePlugin::loaded('EvInventory') && $Model->hasBehavior('EvInventory.Inventories')) {
			App::uses('InventoryLib', 'EvInventory.Lib');

			if (! isset($item['BasketItem'])) {
				$item['BasketItem'] = $item;
			}

			if (! InventoryLib::hasEnoughStock($Model->readForEdit($item['BasketItem']['model_id']), $item['BasketItem']['quantity'])) {
				return false;
			}
		}

		return true;
	}

	/**
	 * find a total row given the name
	 *
	 * @param 	string 		$totalName 		Row name for the total
	 * @return 	array|bool 	$BasketTotal	BasketTotal array row
	 */
	public function findTotalRow($name) {
		$this->getBasket();

		$rows = Hash::combine(
			$this->basketCache['BasketTotal'],
			'{n}.name',
			'{n}.id'
		);

		if (isset($rows[$name]) && ! empty($rows[$name])) {
			return $this->basketCache['BasketTotal'][$rows[$name]];
		}

		return false;
	}

	/**
	 * reset a basket - clear the hash and basket
	 *
	 * @return 	bool
	 */
	public function resetBasket() {
		$this->hash = null;
		$this->basketCache = null;

		return true;
	}

	/**
	 * get a single basket item
	 *
	 * @param 	string|int|array 	$key 	Either Basket row ID / compiled basket key string / Array with two elements of model and model_id
	 * @return  array|bool 			$item 	Item array or bool false
	 */
	public function getBasketItem($key) {
		$this->getBasket();

		// set the key as the itemKey as a fallback
		$itemKey = $this->getItemKey($key);

		if ($this->basketItemExists($itemKey)) {
			return $this->basketCache['BasketItem'][$itemKey];
		}

		return false;
	}

	/**
	 * get the basket
	 *
	 * @param 	bool 	$rebuild	Whether to force a rebuild of the basket or not
	 * @return 	array 	$basket 	Basket Data
	 */
	public function getBasket($rebuild = false) {
		$this->setupHash();
		$this->setupBasket($this->hash);

		if ($rebuild === false && ! empty($this->basketCache)) {
			return $this->basketCache;
		}

		$itemContain = $this->buildItemContain(
			Configure::read('EvBasket.BasketItemContains'),
			Configure::read('EvBasket.BasketItemBelongsTo')
		);

		$this->basketCache = $this->Basket->getFullBasket(
			$this->hash,
			$itemContain
		);

		return $this->basketCache;
	}

	/**
	 * add an item to the basket
	 *
	 * @param 	array 		$item 		Array with two elements of model and model_id / or multidimensional array of model, model_id, quantity, unitPrice, taxRate
	 * @param 	decimal 	$unitPrice 	Unit price of the item
	 * @param 	int 		$quantity	Amount of items to add
	 * @param 	decimal		$taxRate 	The tax rate to add
	 * @param 	bool 		$dispatchEvent 	Whether to disaptch the itemAdd event - used to prevent loops
	 * @return 	bool
	 */
	public function addItem($items, $unitPrice = 0, $quantity = 1, $taxRate = 0, $dispatchEvent = true) {
		$this->getBasket();

		$toUpdate = $this->buildUpdateList($items, $quantity, $unitPrice, $taxRate);

		$result = false;
		$updated = array();

		foreach ($toUpdate as $item) {

			$itemKey = $this->buildItemKey($item);

			// check if we are updating an existing item or creating a new row
			if ($this->basketItemExists($itemKey)) {
				$itemAdded = $this->BasketItem->updateItem(
					$this->basketCache['BasketItem'][$itemKey],
					($this->basketCache['BasketItem'][$itemKey]['quantity'] + $item['quantity']),
					$item['unitPrice'],
					$item['taxRate']
				);
			} else {
				$itemAdded = $this->BasketItem->addItem(
					$this->basketCache['Basket']['id'],
					$item,
					$item['unitPrice'],
					$item['quantity'],
					$item['taxRate']
				);
			}

			if ($itemAdded === true) {
				$result = true;
				$updatedItems[] = $item;
			}
		}

		if ($result === true) {
			if ($dispatchEvent) {
				// dispatch item added to basket event
				$this->_controller->getEventManager()->dispatch(
					new CakeEvent('EvBasket.Component.Basket.itemAdd', $this, array(
						'basketId' => $this->basketCache['Basket']['id'],
						'items' => $updatedItems
					))
				);
			}

			// call out to recompile all the basket totals
			return $this->rebuildTotals();
		}

		return false;
	}

	/**
	 * update an item in the basket
	 *
	 * @param 	array 		$item 		Array with two elements of model and model_id / or multidimensional array of model, model_id, quantity, unitPrice, taxRate
	 * @param 	int 		$quantity	Amount of items to add
	 * @param 	decimal 	$unitPrice 	Unit price of the item
	 * @param 	decimal		$taxRate 	The tax rate to update
	 * @param 	bool 		$dispatchEvent 	Whether to disaptch the itemAdd event - used to prevent loops
	 * @return 	bool
	 */
	public function updateItem($items, $quantity = null, $unitPrice = null, $taxRate = 0, $dispatchEvent = true) {
		$this->getBasket();

		$toUpdate = $this->buildUpdateList($items, $quantity, $unitPrice, $taxRate);

		$result = false;
		$updated = array();

		foreach ($toUpdate as $item) {
			$itemKey = $this->buildItemKey($item);

			// check if the row actually exists
			if (! $this->basketItemExists($itemKey)) {
				continue;
			}

			// check if we're wanting to delete
			if ($item['quantity'] <= 0) {
				$this->deleteItem($item);
				continue;
			}

			// check we have enough stock
			if (! $this->hasEnoughStock(EvClassRegistry::init($item['model']), $item)) {
				$this->_controller->Flash->fail(
					Configure::read('EvInventory.notEnoughStockMessage')
				);
				continue;
			}

			$updateItem = $this->BasketItem->updateItem(
				$this->basketCache['BasketItem'][$itemKey],
				$item['quantity'],
				(isset($item['unitPrice'])) ? $item['unitPrice'] : null,
				(isset($item['taxRate'])) ? $item['taxRate'] : null
			);

			if ($updateItem === true) {
				$result = true;
				$updatedItems[] = $item;
			}
		}

		if ($result === true) {
			if ($dispatchEvent) {
				// dispatch item added to basket event
				$this->_controller->getEventManager()->dispatch(
					new CakeEvent('EvBasket.Component.Basket.itemUpdated', $this, array(
						'basketId' => $this->basketCache['Basket']['id'],
						'items' => $updatedItems
					))
				);
			}

			// call out to recompile all the basket totals
			return $this->rebuildTotals();
		}

		return false;
	}

	/**
	 * delete and item from the basket
	 *
	 * @param 	array 		$item 		Array with two elements of model and model_id
	 * @param 	bool 		$dispatchEvent 	Whether to disaptch the itemAdd event - used to prevent loops
	 * @return 	bool
	 */
	public function deleteItem($item, $dispatchEvent = true) {
		$this->getBasket();

		$itemKey = $this->buildItemKey($item);

		// check if the row actually exists
		if (! $this->basketItemExists($itemKey)) {
			return false;
		}

		$result = $this->BasketItem->deleteItem(
			$this->basketCache['BasketItem'][$itemKey]
		);

		if ($result === true) {
			if ($dispatchEvent) {
				// dispatch item added to basket event
				$this->_controller->getEventManager()->dispatch(
					new CakeEvent('EvBasket.Component.Basket.itemDeleted', $this, array(
						'basketId' => $this->basketCache['Basket']['id'],
						'itemKey' => $itemKey,
						'BasketItem' => $this->BasketItem
					))
				);
			}

			// call out to recompile all the basket totals
			return $this->rebuildTotals();
		}

		return false;
	}

	/**
	 * manage a total row
	 *
	 * @param 	string 	$name 			The name label for the row
	 * @param 	decimal $amount 		The amount to add
	 * @param 	int 	$sequence 		The sequence order number
	 * @param 	bool 	$displayOnly 	Whether it's a displayOnly row (not added n grand total calcs)
	 * @return 	bool
	 */
	public function manageTotalRow($name, $amount, $sequence = null, $displayOnly = false) {
		$this->getBasket();

		if (empty($name)) {
			return false;
		}

		$totalRow = $this->findTotalRow($name);

		$data = array(
			'name' => $name,
			'amount' => $amount,
			'display_only' => $displayOnly
		);

		if (is_null($sequence) && ! empty($totalRow['sequence'])) {
			$data['sequence'] = $totalRow['sequence'];
		} elseif (! empty($sequence)) {
			$data['sequence'] = $sequence;
		}

		$basketTotalId = null;
		if ($totalRow !== false && ! empty($totalRow['id'])) {
			$basketTotalId = $totalRow['id'];
		}

		return $this->BasketTotal->manageTotalRow(
			$this->basketCache['Basket']['id'],
			$data,
			$basketTotalId
		);
	}

	/**
	 * get basket summary for headers / sidebar
	 *
	 * @return 	array 	$summary 	summary of amount and no. items
	 */
	public function getBasketSummary() {
		$this->getBasket();

		$noItems = 0;
		$total = 0;

		foreach ($this->basketCache['BasketItem'] as $BasketItem) {
			$noItems += $BasketItem['quantity'];
			$total += $BasketItem['row_total'];
		}

		return array(
			'total' => $total,
			'items' => $noItems
		);
	}

	/**
	 * build transaction items from basket
	 *
	 * @return 	array 	$transactionItems
	 */
	public function buildTransactionItems() {
		$this->getBasket();

		$items = array();
		foreach ($this->basketCache['BasketItem'] as $BasketItem) {
			$items[] = array(
				'model' => $BasketItem['model'],
				'model_id' => $BasketItem['model_id'],
				'description' => BasketLib::getItemName($BasketItem),
				'amount' => $BasketItem['row_total'],
				'quantity' => $BasketItem['quantity'],
				'unitPrice' => $BasketItem['unit_price']
			);
		}

		return $items;
	}

	/**
	 * build transaction totals from basket
	 *
	 * @return 	array 	$transactionTotals
	 */
	public function buildTransactionTotals() {
		$this->getBasket();

		$grandTotal = $this->findTotalRow(Configure::read('EvBasket.labels.grandtotal'));
		$subTotal = $this->findTotalRow(Configure::read('EvBasket.labels.subtotal'));
		$taxTotal = $this->findTotalRow(Configure::read('EvBasket.labels.tax'));
		$deliveryTotal = $this->findTotalRow(Configure::read('EvBasket.labels.delivery'));
		$discount = $this->findTotalRow('Discount');

		return array(
			'delivery' => $deliveryTotal['amount'],
			'tax' => $taxTotal['amount'],
			'subtotal' => $subTotal['amount'],
			'grandtotal' => $grandTotal['amount'],
			'discount' => (! empty($discount['amount'])) ? $discount['amount'] : false
		);
	}
}
