<?php

App::uses('AuditableBehavior', 'AuditLog.Model/Behavior');

App::uses('CakeSession', 'Model/Datasource');

class EvAuditLogAuditableBehavior extends AuditableBehavior {

/**
 * Get the current user from the auth component. Have to use the cake session statically as this is a
 * behavior. This isn't the proper way to access the auth session but is the only way we are able to
 * detect which user has made a change. If it isn't possible to acquire the user or a user didn't
 * make the change then it defaults to a system change.
 *
 * @param Model $Model Cake model the behavior is running on.
 * @return array Containing the id and description of the current user. Id is only set if a user is found.
 */
	public function currentUser(Model $Model) {
		$currentUser = [
			'id' => null,
			'description' => null,
		];

		$sessionUser = CakeSession::read('Auth.User');

		if (!empty($sessionUser)) {
			$currentUser['id'] = $sessionUser['User']['id'];
			$currentUser['description'] = $sessionUser['User']['email'];
		} else {
			$currentUser['description'] = 'System Change';
		}

		return $currentUser;
	}

/**
 * Get the audits for the current model. Can be limited to just create or edit events. Audits from associated
 * models can be contained by adding the models to the containLog behavior setting.
 *
 * @param Model       $Model Cake model to get audits of.
 * @param int         $id    The id of the model to find audits of.
 * @param null|string $event Empty to get all audits, create or edit to limit to those events.
 * @return array Found audits for the current model.
 */
	public function getAudits(Model $Model, $id, $event = null) {
		if ($this->_isAuditLogModel($Model) || empty($id)) {
			return null;
		}

		$Model->bindModel(
			[
				'hasMany' => [
					'AuditLog.Audit'
				]
			]
		);

		$containLogs = !empty($this->settings[$Model->alias]['containLog']) ? $this->settings[$Model->alias]['containLog'] : null;

		//Get the ids of the audits we want to get.
		$auditIds = $this->getAuditIds($Model, $id, $event, $containLogs);

		$audits = [];
		if (!empty($auditIds)) {
			//Find the actual audits to get with the deltas and order them.
			$audits = $Model->Audit->find(
				'all',
				[
					'conditions' => [
						'Audit.id' => $auditIds,
					],
					'order' => 'Audit.created DESC, Audit.event DESC',
					'group' => 'Audit.id',
					'contain' => [
						'AuditDelta' => [
							'order' => 'property_name ASC'
						]
					]
				]
			);
		}

		return $audits;
	}

/**
 * Get a list of audit ids for the current instance of the model. Limit to create or edit events and contain
 * associated model audits by providing a contain array. The list is used to get a full search for audits
 * that contains the deltas and is ordered.
 *
 * @param Model       $Model       Cake model of the current instance to get audit ids for.
 * @param int|array   $id          Id of the model to get audits for. If an array of ids is passed then all audits will be found.
 * @param null|string $event       Empty to get all audits, create or edit to limit to those events.
 * @param array       $containLogs The audit logs of associated models to contain in the audit list.
 * @return array List of audit ids that the current model instance has.
 */
	public function getAuditIds(Model $Model, $id, $event, $containLogs = []) {
		if ($this->_isAuditLogModel($Model) || empty($id)) {
			return null;
		}

		if (!isset($Model->hasMany['Audit'])) {
			$Model->bindModel(
				[
					'hasMany' => [
						'AuditLog.Audit',
					],
				]
			);
		}

		//Get the audit logs for the current model id.
		$auditConditions = [
			'Audit.model' => $Model->getLinkedAuditModels(),
			'Audit.entity_id' => $id,
		];

		if (!empty($event)) {
			$auditConditions['Audit.event'] = strtoupper($event);
		}

		$auditIds = $Model->Audit->find(
			'all',
			[
				'fields' => [
					'Audit.id',
				],
				'conditions' => $auditConditions,
			]
		);

		$auditIds = Hash::extract($auditIds, '{n}.Audit.id');

		//Get any associated audit logs that are to be contained.
		$associatedModels = $Model->getAssociated();
		if (!empty($containLogs) && is_array($containLogs) && !empty($associatedModels)) {
			foreach ($containLogs as $containIndex => $contain) {
				$containModel = $contain;
				if (is_array($contain)) {
					$containModel = $containIndex;
				}

				if (!isset($containModel, $associatedModels)) {
					continue;
				}

				if (
					!$this->_isAuditLogModel($Model->{$containModel}) &&
					$Model->{$containModel}->hasMethod('getAuditIds')
				) {
					$associatedModel = $Model->{$containModel};

					//Get the ids of the assocaited data via the foreign key.
					$associatedForeignKey = $Model->{$associatedModels[$containModel]}[$containModel]['foreignKey'];

					$query = [
						'fields' => [
							$associatedModel->alias . '.id',
						],
					];

					$relationship = $associatedModels[$containModel];

					$relationshipInfo = $Model->{$relationship}[$containModel];

					switch($relationship) {
						case 'hasAndBelongsToMany':
							list($withPlugin, $withAlias) = pluginSplit($relationshipInfo['with']);

							$query['joins'][] = [
								'table' => $relationshipInfo['joinTable'],
								'alias' => $withAlias,
								'conditions' => array_merge(
									[
										$withAlias . '.' . $relationshipInfo['associationForeignKey'] . ' = ' . $associatedModel->alias . '.id',
										$withAlias . '.' . $relationshipInfo['foreignKey'] => $id,
									],
									! empty($relationshipInfo['conditions']) ? $relationshipInfo['conditions'] : []
								),
							];

							break;
						case 'belongsTo':
							$query['joins'][] = [
								'table' => $Model->table,
								'alias' => $Model->alias,
								'conditions' => array_merge(
									[
										$Model->alias . '.' . $relationshipInfo['foreignKey'] . ' = ' . $associatedModel->alias . '.' . $associatedModel->primaryKey,
										$Model->alias . '.' . $Model->primaryKey => $id,
									],
									! empty($relationshipInfo['conditions']) ? $relationshipInfo['conditions'] : []
								),
							];

							break;
						default:
							$query['conditions'][$associatedModel->alias . '.' . $associatedForeignKey] = $id;
							break;
					}

					$assocaitedIds = $associatedModel->find(
						'all',
						$query
					);

					$assocaitedIds = Hash::extract($assocaitedIds, '{n}.' . $associatedModel->alias . '.id');
					if (!empty($assocaitedIds)) {
						$assocaitedAuditIds = $associatedModel->getAuditIds($assocaitedIds, $event, $contain);

						if (!empty($assocaitedAuditIds)) {
							$auditIds = Hash::merge($auditIds, $assocaitedAuditIds);
						}
					}
				}
			}
		}

		return $auditIds;
	}

/**
 * Executed after a save operation completes. Overriden to fix the issue with audit logs not being created when the
 * original data isn't set and the new data is.
 * Issue opened in repo: https://github.com/robwilkerson/CakePHP-Audit-Log-Plugin/issues/119
 * Once fixed this method can be removed.
 *
 * @param Model $Model The model that is used for the save operation.
 * @param bool $created True, if the save operation was an insertion, false otherwise.
 * @param array $options The options data (unused).
 * @return true Always true.
 */
	public function afterSave(Model $Model, $created, $options = array()) {
		// Do not act on the AuditLog related models.
		if ($this->_isAuditLogModel($Model)) {
			return true;
		}

		$modelData = $this->_getModelData($Model);
		if (!$modelData) {
			$this->afterDelete($Model);

			return true;
		}

		$audit[$Model->alias] = $modelData;
		$audit[$Model->alias][$Model->primaryKey] = $Model->id;

		// Create a runtime association with the Audit model
		$Model->bindModel(
			array('hasMany' => array('AuditLog.Audit'))
		);

		// If a currentUser() method exists in the model class (or, of
		// course, in a superclass) the call that method to pull all user
		// data. Assume than an ID field exists.
		$source = array();
		if ($Model->hasMethod('currentUser')) {
			$source = $Model->currentUser();
		} elseif ($Model->hasMethod('current_user')) {
			$source = $Model->current_user();
		}

		$data = array(
			'Audit' => array(
				'event' => $created ? 'CREATE' : 'EDIT',
				'model' => $Model->getAuditModelName(),
				'entity_id' => $Model->id,
				'request_id' => self::_requestId(),
				'json_object' => json_encode($audit),
				'source_id' => isset($source['id']) ? $source['id'] : null,
				'description' => isset($source['description']) ? $source['description'] : null,
			),
		);

		// We have the audit_logs record, so let's collect the set of
		// records that we'll insert into the audit_log_deltas table.
		$updates = array();
		foreach ($audit[$Model->alias] as $property => $value) {
			$delta = array();

			// Ignore virtual fields (Cake 1.3+) and specified properties.
			if (($Model->hasMethod('isVirtualField') && $Model->isVirtualField($property))
				|| in_array($property, $this->settings[$Model->alias]['ignore'])
			) {
				continue;
			}

			if ($created) {
				$delta = array(
					'AuditDelta' => array(
						'property_name' => $property,
						'old_value' => '',
						'new_value' => $value,
					),
				);
			} else {
				//Modifying this check so new data on edits creates audits.
				$originalData = $this->_getOriginalDataForModel($Model);

				if (array_key_exists($property, $originalData) && $originalData[$property] != $value) {
					// If the property exists in the original _and_ the
					// value is different, store it.
					$delta = array(
						'AuditDelta' => array(
							'property_name' => $property,
							'old_value' => $originalData[$property],
							'new_value' => $value,
						),
					);
				}
			}

			if (!empty($delta)) {
				array_push($updates, $delta);
			}
		}

		// Insert an audit record if a new model record is being created
		// or if something we care about actually changed.
		if ($created || count($updates)) {
			$Model->Audit->create();
			$Model->Audit->save($data);

			if ($created) {
				if ($Model->hasMethod('afterAuditCreate')) {
					$Model->afterAuditCreate($Model);
				}
			} else {
				if ($Model->hasMethod('afterAuditUpdate')) {
					$Model->afterAuditUpdate($Model, $this->_original, $updates, $Model->Audit->id);
				}
			}
		}

		// Insert a delta record if something changed.
		if (count($updates)) {
			foreach ($updates as $delta) {
				$delta['AuditDelta']['audit_id'] = $Model->Audit->id;

				$Model->Audit->AuditDelta->create();
				$Model->Audit->AuditDelta->save($delta);

				if (!$created && $Model->hasMethod('afterAuditProperty')) {
					$Model->afterAuditProperty(
						$Model,
						$delta['AuditDelta']['property_name'],
						Hash::get($this->_getOriginalDataForModel($Model), $delta['AuditDelta']['property_name']),
						$delta['AuditDelta']['new_value']
					);
				}
			}
		}

		// Destroy the runtime association with the Audit.
		$Model->unbindModel(
			array('hasMany' => array('Audit'))
		);

		$this->_unsetOriginalDataForModel($Model);

		return true;
	}

/**
 * The method that can be overridden by the model to provide an alternative model name
 *
 * @param Model $Model The model being logged
 * @return string
 */
	public function getAuditModelName($Model) {
		return $Model->plugin ? $Model->plugin . '.' . $Model->name : $Model->name;
	}

/**
 * Override this to attach extra model names to the main audit query when fetching records
 *
 * @param Model $Model The model being logged
 * @return string
 */
	public function getLinkedAuditModels($Model) {
		return [$Model->getAuditModelName()];
	}

/**
 * Add a message to the audit log for the associated model.
 *
 * @param Model  $Model       The model the write an audit message for.
 * @param string $message     The message to write to the audit log.
 * @param array  $auditParams The parameters to write the audit log with.
 * @return bool.
 */
	public function auditMessage(Model $Model, string $message, array $auditParams = []) {
		$auditModel = $auditParams['model'] ?? ($Model->plugin ? $Model->plugin . '.' . $Model->alias : $Model->alias);
		$auditEntityId = $auditParams['entity_id'] ?? $Model->id;

		$source = $Model->currentUser();

		$audit = [
			'Audit' => [
				'event' => 'MESSAGE',
				'model' => $auditModel,
				'entity_id' => $auditEntityId,
				'request_id' => self::_requestId(),
				'json_object' => json_encode($message),
				'source_id' => isset($source['id']) ? $source['id'] : null,
				'description' => isset($source['description']) ? $source['description'] : null,
			],
			'AuditDelta' => [
				[
					'property_name' => 'Message',
					'old_value' => null,
					'new_value' => $message,
				],
			],
		];

		// Create a runtime association with the Audit model
		$Model->bindModel(
			array('hasMany' => array('AuditLog.Audit'))
		);

		$saved = $Model->Audit->saveAssociated($audit, ['deep' => true]);

		// Destroy the runtime association with the Audit.
		$Model->unbindModel(
			array('hasMany' => array('Audit'))
		);

		return $saved;
	}
}
