<?php

/**
 * This class is used to present data in an audit log in a way that is useful to a user.
 * Mainly this will deal with assocaited ids and lookup the actual value of the record instead
 * of just the id being displayed. When presenting an audit delta value, the presenter will
 * attempt to call out to the plugin that the audit belongs so that the value can be found
 * within that plugin. If no presenter is found in the plugin then the value is returned without
 * modification.
 */
class AuditLogPresenter {

	// A modelAlias => AuditLogPresenter registry
	private static $__presenterRegistry;

	// An instance of AuditLogPresenter
	private static $__defaultPresenter;

/**
 * Gets the default presenter instantiating one if none exists
 *
 * @return AuditLogPresenter The default presenter
 */
	public static function getDefaultPresenter() {
		if (empty(self::$__defaultPresenter)) {
			self::$__defaultPresenter = new AuditLogPresenter();
		}
		return self::$__defaultPresenter;
	}

/**
 * Gets an instance of a presenter for a given model
 *
 * @param string $model The plugin.model name.
 * @return AuditLogPresenter A presenter for the given model
 */
	public static function init($model) {
		if (empty($model)) {
			// Won't be able to find an override without a model so return the default
			return self::getDefaultPresenter();
		}

		list($plugin, $modelAlias) = pluginSplit($model);

		// If we've used this one before use the existing instance
		if (!empty(self::$__presenterRegistry[$modelAlias])) {
			return self::$__presenterRegistry[$modelAlias];
		}

		// Check for an extended plugin presenter first. This will also find presenters for non-plugin models
		$presenterClassName = $plugin . $modelAlias . 'AuditLogPresenter';
		App::uses($presenterClassName, 'Lib/AuditLog');
		if (class_exists($presenterClassName)) {
			self::$__presenterRegistry[$modelAlias] = new $presenterClassName();
			return self::$__presenterRegistry[$modelAlias];
		}

		//If an extended version couldn't be found check for a presenter in the plugin
		$presenterClassName = $modelAlias . 'AuditLogPresenter';
		App::uses($presenterClassName, $plugin . '.Lib/AuditLog');
		if (class_exists($presenterClassName)) {
			self::$__presenterRegistry[$modelAlias] = new $presenterClassName();
			return self::$__presenterRegistry[$modelAlias];
		}

		return self::getDefaultPresenter();
	}

/**
 * Should present Item
 *
 * @param array $audit the audit.
 * @return bool True if the audit item should appear in the log, false otherwise
 */
	public function shouldPresentItem($audit) {
		return true;
	}

/**
 * The item name identifies the item that is changing in the audit
 *
 * @param array $audit he audit data.
 * @return string Humanised item name. Defaults to "ModelName #id"
 */
	public function presentItemName($audit) {
		list($plugin, $modelAlias) = pluginSplit($audit['model']);
		return InflectorExt::humanize($modelAlias) . ' #' . $audit['entity_id'];
	}

/**
 * The property name is the name of the database field that changed. This method turns the field name
 * into something readable.
 *
 * @param array $audit      The audit data.
 * @param array $auditDelta The aduit delta array.
 * @return string Humanised property name.
 */
	public function presentPropertyName($audit, $auditDelta) {
		if (empty($auditDelta['property_name'])) {
			return '';
		} else {
			return InflectorExt::humanize($auditDelta['property_name']);
		}
	}

/**
 * Present an old value of an audit delta.
 *
 * @param array $audit      The audit data.
 * @param array $auditDelta The aduit delta array.
 * @return string The value to present.
 */
	public function presentOldValue($audit, $auditDelta) {
		return $this->presentValue($audit, $auditDelta, $auditDelta['old_value']);
	}

/**
 * Present an new value of an audit delta.
 *
 * @param array $audit      The audit data.
 * @param array $auditDelta The aduit delta array.
 * @return string The value to present.
 */
	public function presentNewValue($audit, $auditDelta) {
		return $this->presentValue($audit, $auditDelta, $auditDelta['new_value']);
	}

/**
 * Present a audit delta value. If the value is null return an empty string. If the model is present
 * on the audit then an attempt to use a plugin presenter will occur, otherwise the value is returned
 * withou any modification.
 *
 * @param array $audit      The audit data.
 * @param array $auditDelta The aduit delta array.
 * @param string $valueType Whether the value to display is old or new.
 * @return string The value to present.
 */
	public function presentValue($audit, $auditDelta, $value) {
		if (is_bool($value)) {
			return $this->_presentBoolean($value);
		} elseif (empty($value)) {
			$value = '-';
		} elseif ($auditDelta['property_name'] === 'model_id') {
			$value = $this->_lookupPolymorphicValue($audit, $value);

			if (empty($value)) {
				$value = '-';
			}
		}

		return $value;
	}

/**
 * Check if a value should appear in the audit log
 *
 * @param array $audit      The audit data.
 * @param array $auditDelta The aduit delta array.
 * @return bool Whether to present. Defaults to true unless overridden.
 */
	public function shouldPresentProperty($audit, $auditDelta) {
		return true;
	}

/**
 * Presenter method to present a boolean value which can be used in plugin presenters.
 *
 * @param string $value The current value to present.
 * @return string The value to present.
 */
	protected function _presentBoolean($value) {
		if (empty($value)) {
			return 'No';
		} else {
			return 'Yes';
		}
	}

/**
 * Presenter method to replace an id field with the displayname of the field.
 * Initialises the model to find and then searches using the current value. Returns the displayField value
 * of the search. If no record is found then the value is returned without modification.
 *
 * @param string $modelClass The class of the model to initialise.
 * @param string $modelId The current value to present.
 * @return string The value to present.
 */
	protected function _lookupValue($modelClass, $modelId) {
		$Model = EvClassRegistry::init($modelClass);
		$modelData = $Model->findById($modelId);

		if (!empty($modelData)) {
			return $modelData[$Model->alias][$Model->displayField];
		}

		return $modelId;
	}

/**
 * Presenter method to replace an polymorphic id field with the displayname of
 * the field. The model is found from the object data stored against the audit
 * which is then used to initialises the model to find and then searches using
 * the current value. Returns the displayField value of the search. If no record
 * is found then the value is returned without modification.
 *
 * @param array $audit Audit data.
 * @param string $modelId The current value to present.
 * @return string The value to present.
 */
	protected function _lookupPolymorphicValue($audit, $modelId) {
		if (empty($audit['model']) || empty($audit['json_object'])) {
			return null;
		}

		$auditObject = json_decode($audit['json_object'], true);

		list($auditPlugin, $auditModel) = pluginSplit($audit['model']);

		if (empty($auditModel) || empty($auditObject) || empty($auditObject[$auditModel]['model'])) {
			return null;
		}

		return $this->_lookupValue($auditObject[$auditModel]['model'], $modelId);
	}
}
