<?php
App::uses('EvAddressBookAppModel', 'EvAddressBook.Model');
/**
 * Address Model
 *
 * @property Country $Country
 */
class Address extends EvAddressBookAppModel {

/**
 * Display field
 *
 * @var string
 */
	public $displayField = 'name';

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'model' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				//'message' => 'Your custom message here',
				//'allowEmpty' => false,
				//'required' => false,
				//'last' => false, // Stop validation after this rule
				//'on' => 'create', // Limit validation to 'create' or 'update' operations
			),
		),
		'model_id' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				//'message' => 'Your custom message here',
				//'allowEmpty' => false,
				//'required' => false,
				//'last' => false, // Stop validation after this rule
				//'on' => 'create', // Limit validation to 'create' or 'update' operations
			),
		),
		'address1' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'Please fill out address line 1'
			),
		),
		'city' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'Please fill out the city'
			),
		),
		'post_code' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'Please fill out the postcode'
			),
			'checkPostcode' => array(
				'rule' => array('validatePostcode'),
				'message' => 'Please enter a valid postal code'
			),
		),
		'country_id' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'Please select a country'
			),
			'checkShipping' => array(
				'rule' => array('hasShipping'),
				'message' => 'This country is not available for delivery'
			),
		),
		'state' => array(
			'checkCountryState' => array(
				'rule' => array('validateCountryState'),
				'message' => 'Please select a state'
			),
		),
	);

	//The Associations below have been created with all possible keys, those that are not needed can be removed

/**
 * belongsTo associations
 *
 * @var array
 */
	public $belongsTo = array(
		'Country' => array(
			'className' => 'EvCountry.Country'
		)
	);

/**
 * {@inheritDoc}
 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);
	}

/**
 * Get all the addresses for a given model / model id.
 *
 * @param string $model   The model the data belongs to.
 * @param int    $modelId The id the address relates to.
 * @return array
 */
	public function getAddresses($model, $modelId) {
		return $this->getAddressesForModel($model, $modelId);
	}

/**
 * Get all the addresses for a given model / model id. Extend this if you want to modify the query
 * parameters when getting the addresses.
 *
 * @param string $model   The model the data belongs to.
 * @param int 	 $modelId The id the address relates to.
 * @param array  $query   Extra query parameters.
 * @return array
 */
	public function getAddressesForModel($model, $modelId, $query = []) {
		$queryParams = Hash::merge(
			[
				'conditions' => [
					'Address.model' => $model,
					'Address.model_id' => $modelId
				],
				'contain' => [
					'Country'
				]
			],
			$query
		);

		return $this->find(
			'all',
			$queryParams
		);
	}
/**
 * Get a single address.
 *
 * @param int $id The address id.
 * @return array
 */
	public function getAddress($id) {
		return $this->getAddressById($id);
	}

/**
 * Get a single address. Extend this if you want to modify the query
 * parameters when getting the address.
 *
 * @param int   $id    The address id.
 * @param array $query Extra query parameters.
 * @return array
 */
	public function getAddressById($id, $query = []) {
		$queryParams = Hash::merge(
			[
				'conditions' => [
					'Address.id' => $id
				],
				'contain' => [
					'Country'
				]
			],
			$query
		);

		return $this->find(
			'first',
			$queryParams
		);
	}

/**
 * Validate US states. If the country provided isn't US then it passes. The state must be provided and be a length of 2
 * uppercase letters.
 */
	public function validateCountryState($check) {
		//Check the country selected is the US
		if (
			$this->data['Address']['country_id'] === $this->Country->findByIso2('US')['Country']['id'] ?? null
			|| $this->data['Address']['country_id'] === $this->Country->findByIso2('CA')['Country']['id'] ?? null
		) {
			//Check that a state has been supplied
			if (isset($this->data['Address']['state']) && !empty($this->data['Address']['state'])) {
				//Check if it is an uppercase 2 character string for US state code
				if (!preg_match('/^[A-Z]{2}$/', $this->data['Address']['state'])) {
					return false;
				}
			} else {
				//State hasn't been supplied
				return false;
			}
		}

		return true;
	}

/**
 * Validate  postcodes. If the country provided isn't US, CA or UK then it passes.
 *
 * @param array $check Field to validate
 * @return bool
 */
	public function validatePostcode($check) {
		// Only validating postcodes on production as test payments can require specific test address details
		if (
			Configure::read('app.environment') === 'DEVELOPMENT'
			&& ! Configure::read('EvAddressBook.validatePostcodeInDevelopment')
		) {
			return true;
		}

		$rules = [
			'CA' => '/^(\d{5}-\d{4}|\d{5}|\d{9})$|^([a-zA-Z]\d[a-zA-Z]( )?\d[a-zA-Z]\d)$/',
			// UK rule does not support overseas teritories, it supports postcodes with and without spaces
			'GB' => '/^(([gG][iI][rR] {0,}0[aA]{2})|((([a-pr-uwyzA-PR-UWYZ][a-hk-yA-HK-Y]?[0-9][0-9]?)|(([a-pr-uwyzA-PR-UWYZ][0-9][a-hjkstuwA-HJKSTUW])|([a-pr-uwyzA-PR-UWYZ][a-hk-yA-HK-Y][0-9][abehmnprv-yABEHMNPRV-Y]))) {0,}[0-9][abd-hjlnp-uw-zABD-HJLNP-UW-Z]{2}))$/',
			'US' => '/^\d{5}$/',
		];

		if (!array_key_exists('Address', $this->data) || !array_key_exists('country_id', $this->data['Address'])) {
			return true;
		}

		$country = $this->Country->findById($this->data['Address']['country_id']);
		if (empty($country) || empty($rules[$country['Country']['iso2']])) {
			return true;
		}

		return (bool)preg_match($rules[$country['Country']['iso2']], $this->data['Address']['post_code']);
	}

/**
 * Validate the country to see if any shipping is available for the current address or not.
 * Can be skipped if config setting, `checkCountryForShipping`, is set to false
 */
	public function hasShipping($check) {
		if (CakePlugin::loaded('EvShipping') && Configure::read('EvAddressBook.checkCountryForShipping')) {
			$ShippingRule = EvClassRegistry::init('EvShipping.ShippingRule');
			$rule = $ShippingRule->find(
				'first',
				[
					'joins' => [
						[
							'table' => 'ev_shipping_shipping_rules_zones',
							'alias' => 'ShippingRuleZone',
							'conditions' => [
								'ShippingRuleZone.shipping_rule_id = ShippingRule.id'
							]
						],
						[
							'table' => 'ev_shipping_zones',
							'alias' => 'Zone',
							'conditions' => [
								'Zone.id = ShippingRuleZone.zone_id',
								'Zone.is_active' => true
							]
						],
						[
							'table' => 'ev_shipping_zones_countries',
							'alias' => 'ZoneCountry',
							'conditions' => [
								'ZoneCountry.zone_id = Zone.id',
								'ZoneCountry.country_id' => $this->data['Address']['country_id']
							]
						]
					],
					'conditions' => [
						'ShippingRule.is_active' => true
					]
				]
			);

			return !empty($rule);
		}

		return true;
	}
}
