<?php

App::uses('AppComponent', 'Controller/Component');

class AddressManagerComponent extends AppComponent {

	/**
	 * get all the address for a given model / model_id
	 *
	 * @param 	object 	$AddressModel 	The model object to use for loading
	 * @param 	string 	$model  		The model the data belongs to
	 * @param 	int 	$modelId 		The id the address relates to
	 * @return 	array
	 */
	public function getAddresses($AddressModel, $model, $modelId) {
		return $AddressModel->getAddresses($model, $modelId);
	}

	/**
	 * get specific address
	 *
	 * @param 	object 	$AddressModel 	The model object to use for loading
	 * @param 	int 	$addressId 		The id the address
	 * @return 	array
	 */
	public function getAddress($AddressModel, $addressId) {
		return $AddressModel->getAddress($addressId);
	}

	/**
	 * save a new address
	 *
	 * @param 	object 	$AddressModel 	The model object to use for loading
	 * @param 	string 	$model  		The model the data belongs to
	 * @param 	int 	$modelId 		The id the address relates to
	 * @return 	array 	$result 		Two element array - result(bool) / data(array)
	 */
	public function add($AddressModel, $model, $modelId) {
		//If data is passed as associative data then check the Address index.
		if (!empty($this->_controller->request->data['Address'])) {
			$addressData = $this->_controller->request->data;
		} else {
			//Otherwise all data is address data.
			$addressData['Address'] = $this->_controller->request->data;
		}

		$addressData['Address']['model'] = $model;
		$addressData['Address']['model_id'] = $modelId;

		if ($AddressModel->saveAssociated($addressData, ['deep' => true])) {
			$action = (!empty($addressData['Address']['id'])) ? 'edited' : 'added';

			$result = true;
			$returnData = array(
				'message' => "Address has been $action",
				'Address' => array(
					'id' => $AddressModel->id
				)
			);
		} else {

			$validationErrors = $AddressModel->validationErrors;

			$errors = array(
				'description' => 'Please correct the following issues and try again',
				'list' => $validationErrors
			);

			$result = false;
			$returnData = array(
				'errors' => $errors
			);
		}

		return array(
			'result' => $result,
			'data' => $returnData
		);
	}

	/**
	 * check the given address id belongs to the given user (usually from auth->user())
	 *
	 * @param 	object 	$AddressModel 	The model object to use for loading
	 * @param 	int 	$addressId	 	The address ID to validdate
	 * @param 	int 	$userId 		The user Id to check against
	 * @return 	bool
	 */
	public function addressBelongsToUser($AddressModel, $addressId, $userId) {
		$result = (bool)$AddressModel->find(
			'count',
			array(
				'conditions' => array(
					'Address.id' => $addressId,
					'Address.model' => 'EvCore.User',
					'Address.model_id' => $userId
				),
				'callbacks' => false
			)
		);

		if ($result === true) {
			$returnData = array(
				'message' => 'Address selected'
			);
		} else {
			$returnData = array(
				'errors' => 'The selected address does not belong to the supplied user'
			);
		}

		return array(
			'result' => $result,
			'data' => $returnData
		);
	}

/**
 * Check the given address has an active shipping rule applicable to it so addresses
 * can't be selected if they can't be delivered to.
 * @param  obj   $AddressModel The address model to find the address with
 * @param  int   $addressId    The id of the address to check
 * @return array               Whether the address has applicable shipping or not
 */
	public function addressHasShipping($AddressModel, $addressId) {
		if (CakePlugin::loaded('EvShipping') && Configure::read('EvAddressBook.checkCountryForShipping')) {
			$address = $AddressModel->findById($addressId);

			if (!empty($address)) {
				$ShippingRule = EvClassRegistry::init('EvShipping.ShippingRule');
				$rule = $ShippingRule->find(
					'first',
					[
						'joins' => [
							[
								'table' => 'ev_shipping_shipping_rules_zones',
								'alias' => 'ShippingRuleZone',
								'conditions' => [
									'ShippingRuleZone.shipping_rule_id = ShippingRule.id'
								]
							],
							[
								'table' => 'ev_shipping_zones',
								'alias' => 'Zone',
								'conditions' => [
									'Zone.id = ShippingRuleZone.zone_id'
								]
							],
							[
								'table' => 'ev_shipping_zones_countries',
								'alias' => 'ZoneCountry',
								'conditions' => [
									'ZoneCountry.zone_id = Zone.id',
									'ZoneCountry.country_id' => $address['Address']['country_id']
								]
							]
						],
						'conditions' => [
							'ShippingRule.is_active' => true,
						],
					]
				);

				if (!empty($rule)) {
					return [
						'result' => true,
					];
				} else {
					return [
						'result' => false,
						'data' => [
							'errors' => 'Sorry, we no longer ship to this country'
						]
					];
				}
			} else {
				return [
					'result' => false,
					'data' => [
						'errors' => 'Could not find the selected address'
					]
				];
			}
		} else {
			return [
				'result' => true
			];
		}
	}

	/**
	 * set a selected address to session
	 *
	 * @param 	string 	$type 		Type of address (billing / devliery)
	 * @param 	int 	$addressId 	Address row ID
	 * @return 	bool
	 */
	public function selectAddress($type, $addressId) {
		return $this->_controller->Session->write(
			'EvCheckout.address.' . $type,
			$addressId
		);
	}

	/**
	 * find selected addresses
	 *
	 * @return 	array
	 */
	public function findAddresses() {
		// if the addresses have been loaded and saved to the template get from there
		if (isset($this->_controller->viewVars['selectedAddresses'])) {
			$addresses = $this->_controller->viewVars['selectedAddresses'];
		} else {
			// read for the session and process each id
			$addressIds = $this->_controller->Session->read('EvCheckout.address');
			$addresses = array();

			if (! empty($addressIds)) {
				foreach ($addressIds as $key => $addressId) {
					$addresses[$key] = $this->getAddress(
						EvClassRegistry::init('EvAddressBook.Address'),
						$addressIds[$key]
					);
				}
			}
		}

		// Reformat postcodes to put the space in the correct place (UK postcodes only)
		if (!empty($addresses)) {
			foreach ($addresses as $index => $address) {
				if (!empty($address['Address']['post_code']) && !empty($address['Address']['country_id']) && $address['Address']['country_id'] == Configure::read('EvAddressBook.ukCountryId')) {
					$postcode = $address['Address']['post_code'];
					$postcode = str_replace(' ', '', $postcode);
					$postcode = substr_replace($postcode, ' ', strlen($postcode) - 3, 0);

					$addresses[$index]['Address']['post_code'] = $postcode;
				}
			}
		}

		return $addresses;
	}

/**
 * Get the countries to be used in the country select dropdown for a new address
 * @return array List of countries and ids to be used in country select drop down
 */
	public function getCountriesForNewAddress() {
		$Country = EvClassRegistry::init('EvCountry.Country');

		$query = [];

		if (Configure::read('EvAddressBook.checkCountryForShipping') && CakePlugin::loaded('EvShipping')) {
			$query['joins'] = [
				[
					'table' => 'ev_shipping_zones_countries',
					'alias' => 'ZoneCountry',
					'conditions' => [
						'ZoneCountry.country_id = Country.id',
					]
				],
				[
					'table' => 'ev_shipping_zones',
					'alias' => 'Zone',
					'conditions' => [
						'Zone.id = ZoneCountry.zone_id',
						'Zone.is_active' => true
					]
				],
				[
					'table' => 'ev_shipping_shipping_rules_zones',
					'alias' => 'ShippingRuleZone',
					'conditions' => [
						'ShippingRuleZone.zone_id = Zone.id'
					]
				],
				[
					'table' => 'ev_shipping_shipping_rules',
					'alias' => 'ShippingRule',
					'conditions' => [
						'ShippingRule.id = ShippingRuleZone.shipping_rule_id',
						'ShippingRule.is_active' => true
					]
				],
			];
		}

		return $Country->getForDropDown($query);
	}

}
