<?php
namespace Ev3Users\Controller\Admin;

use App\Controller\Admin\AppController;
use CakeDC\Users\Controller\Traits\LoginTrait;
use CakeDC\Users\Controller\Traits\ProfileTrait;
use CakeDC\Users\Controller\Traits\ReCaptchaTrait;
use CakeDC\Users\Controller\Traits\RegisterTrait;
use CakeDC\Users\Controller\Traits\SocialTrait;
use Cake\Core\Configure;
use Cake\Utility\Hash;

class UsersController extends AppController
{
    use LoginTrait;
    use ProfileTrait;
    use ReCaptchaTrait;
    use RegisterTrait {
        changePassword as updatePassword;
    }
    use SocialTrait;

    /**
     * Initialize and set defaults for model class
     *
     * @return void
     */
    public function initialize()
    {
        parent::initialize();
        $this->_setModelClass(Configure::read('Users.table'));
    }

    /**
     * Before filter
     * @param \Cake\Event\Event $event Event
     * @return void
     */
    public function beforeFilter(\Cake\Event\Event $event)
    {
        $this->Crud->mapAction('index', 'Crud.Index');
        $this->Crud->mapAction('add', 'Crud.Add');
        $this->Crud->mapAction('edit', 'Crud.Edit');
        $this->Crud->mapAction('delete', 'Ev3Crud.Delete');
        parent::beforeFilter($event);
    }

    /**
     * Paginated list of users.
     * @return \Cake\Network\Response|null
     */
    public function index()
    {
        $action = $this->Crud->action();
        $fields = [
            'username',
            'email',
            'is_active',
            'created',
            'modified'
        ];
        $username = Configure::read('Auth.authenticate.Form.fields.username');
        if ($username === 'email') {
            // Remove the username column as this isn't in use on this site.
            array_shift($fields);
        }
        $action->config('scaffold.fields', $fields);

        return $this->Crud->execute();
    }

    /**
     * Configure the add user action.
     * @return \Cake\Network\Response|null
     */
    public function add()
    {
        $action = $this->Crud->action();
        // We want to hide some of the automated fields that won't be set until after user creation.
        // We will handle these as display only on the edit form.
        $action->config('scaffold.fields_blacklist', [
            'tos_date',
            'activation_date'
        ]);

        return $this->Crud->execute();
    }

    /**
     * Configure the edit user action.
     * @param string $id User ID
     * @return \Cake\Network\Response|null
     */
    public function edit($id)
    {
        $action = $this->Crud->action();
        // We want to hide some of the automated fields that won't be set until after user creation.
        // We will handle these as display only on the edit form.
        $action->config('scaffold.fields_blacklist', [
            'password'
        ]);

        return $this->Crud->execute();
    }

    /**
     * Form defaults.
     * @return void
     */
    protected function _form()
    {
        $action = $this->Crud->action();
        // Configure the field blacklist.
        $defaultFieldsBlacklist = [
            'token',
            'token_expires',
            'api_token'
        ];
        // If we're using an email address as the username then we want to hide this field from the
        // form.
        $username = Configure::read('Auth.authenticate.Form.fields.username');
        if ($username === 'email') {
            $defaultFieldsBlacklist[] = 'username';
        }
        $fieldsBlacklist = (array)$action->config('scaffold.fields_blacklist');
        $action->config('scaffold.fields_blacklist', array_merge($defaultFieldsBlacklist, $fieldsBlacklist));
        // Configure the field settings.
        $defaultFieldSettings = [
            'tos_date' => [
                'type' => 'displayDate',
                'label' => __d('ev3_users', 'Terms of Service Date')
            ],
            'activation_date' => [
                'type' => 'displayDate'
            ]
        ];
        $fieldSettings = $action->config('scaffold.field_settings');
        $action->config('scaffold.field_settings', Hash::merge($defaultFieldSettings, $fieldSettings));
    }

    /**
     * Login page
     * @return \Cake\Network\Response|null
     */
    public function login()
    {
        $this->viewBuilder()->templatePath('Users');
    }

    /**
     * Redirects to the current user's edit page.
     * @return \Cake\Network\Response|null
     */
    public function profile()
    {
        return $this->redirect(['action' => 'edit', $this->Auth->user('id')]);
    }

    /**
     * Change current users password.
     * @return \Cake\Network\Response|null
     */
    public function changePassword()
    {
        $this->Breadcrumb->addCrumb(__d('ev3_users', 'Users'), [
            'action' => 'index'
        ]);
        $this->Breadcrumb->addCrumb(__d('ev3_users', 'Change Password'));
        // We need to change where we want the trait's method to redirect us after successfully
        // saving the new password.
        Configure::write('Users.Profile.route', ['action' => 'profile']);
        // Call the RegisterTrait's `changePassword` method (renamed to `updatePassword` so that we
        // could extend it here).
        $this->updatePassword();
        $this->viewBuilder()->theme('CoreUI');
    }
}
