<?php
namespace Ev3Menu\View\Helper;

use Cake\View\Helper;

class MenuHelper extends Helper
{
    public $helpers = ['Html'];

    protected $_defaultConfig = [
        'depth' => null,
        'tag' => 'li',
        'wraptag' => 'ul',
        'class' => 'nav',
        'childrenClass' => 'nav',
        'activeClass' => 'active',
        'activeTrailClass' => 'active-trail',
        'element' => false,
        'disableParentLinks' => true
    ];

    /**
     * Returns a rendered menu.
     *
     * @param array $data Menu data
     * @param array $options Options specific to this render
     * @return string Menu
     */
    public function render(array $data, array $options = [])
    {
        $options = array_merge($this->config(), $options);

        $activeParent = false;
        $data = $this->buildMenu($data, $activeParent, $options['depth'], $options);

        return $this->_formatMenu($data, $options, true);
    }

    /**
     * Format the menu
     *
     * @param array $data Menu data
     * @param array $options Menu options
     * @param bool $primary Determines if this is the first call to the method
     * @return string HTML markup for menu
     */
    protected function _formatMenu(array $data, array $options, $primary = false)
    {
        $html = '';
        foreach ($data as $menu) {
            if (!empty($options['element'])) {
                $html .= $this->_View->element($options['element'], compact('menu', 'options'));
            } else {
                $html .= $this->_formatMenuItem($menu, $options);
            }
        }

        return $this->Html->tag(
            $options['wraptag'],
            $html,
            [
                'class' => $primary ? $options['class'] : $options['childrenClass']
            ]
        );
    }

    /**
     * Format a menu item.
     *
     * @param object $item Menu item
     * @param array $options Menu options
     * @return string HTML markup for menu item
     */
    protected function _formatMenuItem($item, array $options)
    {
        $class = [];
        $class[] = $item['class'];
        if ($item['active']) {
            $class[] = $options['activeClass'];
        }
        if ($item['active_trail']) {
            $class[] = $options['activeTrailClass'];
        }

        return $this->Html->tag(
            $options['tag'],
            $this->_formatMenuItemLink($item, $options) . (!empty($item['children']) ? $this->_formatMenu($item['children'], $options) : ''),
            [
                'class' => implode(' ', $class)
            ]
        );
    }

    /**
     * Format a menu item link.
     *
     * @param object $item Menu item
     * @param array $options Menu options
     * @return string HTML markup for menu item
     */
    protected function _formatMenuItemLink($item, array $options)
    {
        // If we haven't got a route then we don't want to output a link so just return the menu
        // item as a span. If parent links are disabled we also want to output the menu item as a
        // span if it has children.
        if (empty($item->route) || ($options['disableParentLinks'] === true && !empty($item['children']))) {
            return $this->Html->tag(
                'span',
                h($item->name)
            );
        }

        return $this->Html->link(
            $item->name,
            $item->route,
            ['target' => $item['new_window'] ? '_blank' : null]
        );
    }

    /**
     * Build up the menu array with the correct active/active-trail flags set.
     * @param array &$data Menu data
     * @param bool &$activeParent Parent ID of active item
     * @param int $depth Depth of menu
     * @param array $options Menu options
     * @return array Menu
     */
    public function buildMenu(&$data, &$activeParent, $depth = null, array $options = [])
    {
        foreach ($data as &$item) {
            if (!empty($item['children']) && ($depth === null | $depth > 1)) {
                $this->buildMenu(
                    $item['children'],
                    $activeParent,
                    ($depth === null ? null : $depth - 1),
                    $options
                );

            } else {
                // We've reached the maximum depth of the current menu so
                // get rid of the children.
                unset($item['children']);

            }

            // Check for active states.
            $item['active'] = false;
            $item['active_trail'] = false;
            if ($this->checkActive($item) === true) {
                $item['active'] = true;
                $activeParent = $item['parent_id'];

            } elseif ($activeParent === $item['id']) {
                $item['active_trail'] = true;
                $activeParent = $item['parent_id'];

            }

        }

        return $data;
    }

    /**
     * Checks the active state of a menu item.
     *
     * @param array $item Menu item
     * @return bool Returns true if active
     */
    public function checkActive($item)
    {
        return !empty($item['active'])
            || $this->matchUri($item['route'])
            || (!empty($item['pattern']) && $this->matchRegex($item['pattern']));
    }

    /**
     * Checks if we've got a matching URI.
     *
     * @param string $uri Menu item's URI
     * @return bool Returns true for a match
     */
    public function matchUri($uri)
    {
        return $this->getUri() === $uri;
    }

    /**
     * Checks if the URI matches the menu items pattern.
     *
     * @param string $pattern Menu item's Regex
     * @return bool Returns true for a match
     */
    public function matchRegex($pattern)
    {
        return preg_match($pattern, $this->getUri()) === 1;
    }

    public function getUri()
    {
        return $this->request->here();
    }

    /**
     * Returns markup from a menu item's class.
     *
     * @param string $class Class
     *
     * @return string
     */
    public function icon($class, $iconClass = 'fa', $regex = '/fa-([\w\-]+)/i')
    {
        preg_match($regex, $class, $match);
        if (!empty($match)) {
            return '<i class="' . $iconClass . ' ' . $match[0] . '"></i>';
        }
    }
}
