<?php
namespace Ev3Img\View\Helper;

use Cake\View\Helper;
use Intervention\Image\ImageManagerStatic as Image;

/**
 * @property \Cake\View\Helper\HtmlHelper $Html
 */
class ImgHelper extends Helper
{

    /**
     * Helpers.
     *
     * @var array
     */
    public $helpers = ['Html'];

    /**
     * Constructor hook method.
     *
     * @param array $config The configuration settings provided to this helper.
     * @return void
     */
    public function initialize(array $config)
    {
        parent::initialize($config);
        $defaultConfig = [
            'uploadDirectory' => 'files' . DS . 'images',
            'cacheDirectory' => 'files' . DS . 'cache',
            'memoryLimit' => '1024M',
            'driver' => 'imagick'
        ];
        $this->_config = array_merge($defaultConfig, $config);
    }

    /**
     * Returns a cached image.
     *
     * @param object/string $image Image to cache
     * @param array $options Resize options
     * @param array $attributes Img tag attributes
     * @param string $context Image context (defines a directory to place image)
     * @return string Fully qualified img tag
     */
    public function resize($image, $options = [], $attributes = [], $context = null)
    {
        $cache = $this->_cache($image, $options, $context);

        return $this->Html->image($cache['path'], $attributes);
    }

    /**
     * Returns the path to a cached image.
     *
     * @param object/string $image Image to cache
     * @param array $options Resize options
     * @param string $context Image context (defines a directory to place image)
     * @return string Path to cached image
     */
    public function path($image, $options = [], $context = null)
    {
        return $this->_cache($image, $options, $context)['path'];
    }

    /**
     * Generate a cached image path.
     *
     * @param object/string $image Image to cache
     * @param array $options Resize options
     * @param string $context Image context (defines a directory to place image)
     *
     * @return string Path to cached image
     */
    protected function _cache($image, $options, $context = null)
    {
        $cacheFilepath = $this->_config['cacheDirectory'] . DS;

        if (!empty($image)) {
            if (is_object($image)) {
                $originalFilename = $image->filename;
                $originalFilepath = $image->dir;
                $cacheFilename = $this->_cacheFilename($image);
                $cacheFilepath .= $context ?: $image->id . DS;
            } else {
                $file = pathinfo($image);
                $originalFilename = $file['basename'];
                $originalFilepath = DS . $file['dirname'] . DS;
                $cacheFilename = $originalFilename;
                $cacheFilepath .= $context !== null ? $context . DS : '';
            }
        }

        // If `$image` hasn't been set or the image source image doesn't exist provide a fallback.
        if (empty($image) || !file_exists(ROOT . DS . $originalFilepath . $originalFilename)) {
            $originalFilename = 'no-image.png';
            $originalFilepath = 'webroot' . DS . 'img' . DS;
        }

        // Determine the cached image's filepath/filename.
        $cacheOptions = md5(serialize($options));
        $cacheFile = $cacheFilepath . $cacheOptions . DS . $cacheFilename;

        if (!file_exists($cacheFile)) {
            Image::configure(['driver' => $this->_config['driver']]);

            $cacheImg = Image::make(ROOT . DS . $originalFilepath . $originalFilename);

            $height = $cacheImg->height();
            $width = $cacheImg->width();

            if (!empty($options['crop'])) {
                $cacheImg->crop($options['width'], $options['height']);
            } elseif (!empty($options['fit'])) {
                $cacheImg->fit($options['width'], $options['height']);
            } else {
                if ($height <= $width && (int)$options['width'] < $width) {
                    $cacheImg->resize($options['width'], 0);
                } elseif ($height > $width && (int)$options['height'] < $height) {
                    $cacheImg->resize(0, $options['height']);
                }
            }

            // Convert image to greyscale.
            if (!empty($options['greyscale'])) {
                $cacheImg->greyscale();
            }

            // Check if we need to create the cache directory for this image.
            if (!file_exists(WWW_ROOT . $cacheFilepath)) {
                mkdir(WWW_ROOT . $cacheFilepath);
            }
            if (!file_exists(WWW_ROOT . $cacheFilepath . $cacheOptions . DS)) {
                mkdir(WWW_ROOT . $cacheFilepath . $cacheOptions . DS);
            }

            $cacheImg->save(WWW_ROOT . $cacheFile);
        }

        // Return the image path (make sure we use forward and not back slashes for the image, this
        // is an issue when using a Windows server).
        return [
            'path' => '/' . str_replace('\\', '/', $cacheFile)
        ];
    }

    /**
     * Determine a cached images filename from an image object. If the image has a `name` value then
     * we want to use this for SEO.
     *
     * @param object $image Image object
     * @return string Filename for the cached image
     */
    protected function _cacheFilename($image)
    {
        $cacheFilename = $image->filename;
        if (!empty($image->name)) {
            $ext = pathinfo($image->filename, PATHINFO_EXTENSION);
            $cacheFilename = urlencode($image->name) . '.' . $ext;
        }

        return $cacheFilename;
    }
}
