<?php
namespace Ev3Assets\Model\Table;

use Cake\ORM\Query;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;

/**
 * Images Model
 *
 * @method \Ev3Assets\Model\Entity\Image get($primaryKey, $options = [])
 * @method \Ev3Assets\Model\Entity\Image newEntity($data = null, array $options = [])
 * @method \Ev3Assets\Model\Entity\Image[] newEntities(array $data, array $options = [])
 * @method \Ev3Assets\Model\Entity\Image|bool save(\Cake\Datasource\EntityInterface $entity, $options = [])
 * @method \Ev3Assets\Model\Entity\Image patchEntity(\Cake\Datasource\EntityInterface $entity, array $data, array $options = [])
 * @method \Ev3Assets\Model\Entity\Image[] patchEntities($entities, array $data, array $options = [])
 * @method \Ev3Assets\Model\Entity\Image findOrCreate($search, callable $callback = null)
 *
 * @mixin \Cake\ORM\Behavior\TimestampBehavior
 */
class ImagesTable extends Table
{

    /**
     * Initialize method
     *
     * @param array $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config)
    {
        parent::initialize($config);

        $this->table('images');
        $this->displayField('name');
        $this->primaryKey('id');

        $this->addBehavior('Josegonzalez/Upload.Upload', [
            'filename' => [
                'path' => 'webroot{DS}files{DS}images{DS}{year}{DS}{month}{DS}{day}{DS}',
                'nameCallback' => ['\Ev3Assets\Model\Table\ImagesTable', 'fixUploadFilename'],
                'keepFilesOnDelete' => false
            ]
        ]);

        $this->addBehavior('Timestamp');
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator)
    {
        $validator
            ->integer('id')
            ->allowEmpty('id', 'create');

        $validator
            ->allowEmpty('name');

        $validator
            ->allowEmpty('filename');

        $validator
            ->allowEmpty('alt');

        $validator
            ->allowEmpty('caption');

        $validator
            ->allowEmpty('source_text');

        $validator
            ->allowEmpty('source_url');

        $validator
            ->allowEmpty('type');

        $validator
            ->allowEmpty('size');

        $validator
            ->allowEmpty('dir');

        $validator
            ->boolean('is_active')
            ->requirePresence('is_active', 'create')
            ->notEmpty('is_active');

        return $validator;
    }

    /**
     * Upload callback to prevent duplicate filename collisions.
     *
     * @param array $data File data
     * @param array $settings Upload settings
     * @return string Filename to save
     */
    public static function fixUploadFilename($data, $settings)
    {
        $filename = preg_replace('/[^A-Za-z0-9\-_.]/', '', $data['name']);
        return time() . '_' . str_replace(' ', '_', $filename);
    }

    /**
     * Modifies the entity before it is saved so that uploaded file data is persisted in the
     * database too.
     *
     * @param \Cake\Event\Event $event The beforeSave event that was fired
     * @param \Cake\ORM\Entity $entity The entity that is going to be saved
     * @param \ArrayObject $options the options passed to the save method
     * @return void|false
     */
    public function beforeSave(\Cake\Event\Event $event, \Cake\ORM\Entity $entity, \ArrayObject $options)
    {
        $file = $_FILES['filename'];
        // If we're uploading a new file we want to set the image dimensions.
        if (!empty($file['tmp_name'])) {
            list($width, $height) = getimagesize($file['tmp_name']);
            $entity->width = $width;
            $entity->height = $height;
        }
        // We want to make sure the entity has a name so fallback to the filename as the default.
        $entity->name = $entity->name ?: $entity->filename;
    }
}
