<?php

App::uses('BuzzSourceAppModel', 'BuzzSource.Model');

class WaiversApi extends BuzzSourceAppModel {

/**
 * Get booking
 *
 * @param int $salesOrderId Sales order ID
 * @param string $bookingDate Booking date
 * @return array
 */
	public function getBooking($salesOrderId, $bookingDate = null) {
		$result = $this->find('all', array(
			'method' => 'GetBooking',
			'conditions' => array(
				'salesorderId' => $salesOrderId,
				'bookingdate' => $bookingDate
			)
		));

		return !empty($result['WaiversApi']['d']['Bookings']) ? $this->_processBookings($result['WaiversApi']['d']['Bookings']) : [];
	}

/**
 * Returns the booking waiver details, along with an overall waiver signed status
 * to save having to do additional calls to getBooking and getBookingParticipants
 *
 * @param  string $email     Optional. Email address of one of the participants in a booking
 * @param  string $telephone Optional. Telephone number of one of the participants (generally not used on our sites)
 * @return array
 */
	public function confirmBookingWaiver($email = null, $telephone = null) {
		$result = $this->find('all', array(
			'method' => 'ConfirmBookingWaiver',
			'conditions' => [
				'email' => $email,
				'telephone' => $telephone
			]
		));

		return ! empty($result['WaiversApi']['d']) ? $this->_processConfirmBookingWaiver($result['WaiversApi']['d']) : [];
	}

/**
 * Get the participants for a diary booking
 * @param int $bookingId Booking ID
 * @return array
 */
	public function getBookingParticipants($bookingId) {
		$result = $this->find('all', array(
			'method' => 'GetBookingParticipants',
			'conditions' => array(
				'bookingid' => $bookingId
			)
		));

		return ! empty($result['WaiversApi']['d']) ? $this->_processBookingParticipants($result['WaiversApi']['d']) : [];
	}


/**
 * Get the participant for a user
 * @param int $userId User ID
 * @return array
 */
	public function getBookingParticipant($userId) {
		$result = $this->find('all', array(
			'method' => 'GetBookingParticipant',
			'conditions' => array(
				'userId' => $userId
			)
		));

		return ! empty($result['WaiversApi']['d']) ? $this->_processBookingParticipant($result['WaiversApi']['d']) : [];
	}

/**
 * Sign Waiver
 *
 * @param string $model     Model name
 * @param int    $foreignId Foreign ID
 * @param array  $data      Waiver data
 * @param string $signature (Optional) Signature
 * @return int|bool Waiver ID or false if failed
 */
	public function signWaiver($model, $foreignId, $data, $signature = null) {
		$conditions = Hash::extract($data, 'Condition.{n}.title');

		$signedName = $data['Participant']['first_name'] . ' ' . $data['Participant']['last_name'];
		if (! empty($data['Participant']['adult_first_name'])) {
			$signedName = $data['Participant']['adult_first_name'] . ' ' . $data['Participant']['adult_last_name'];
		}

		$dob = $data['Participant']['date_of_birth'];

		if (! empty($data['Participant']['adult_date_of_birth'])) {
			$dob = $data['Participant']['adult_date_of_birth'];
		}

		$userId = ! empty($data['Participant']['adult_first_name']) ? 0 : $data['Participant']['waiver_user_id'];

		$params = [
			'dob' => $dob,
			'email' => $data['Participant']['email'],
			'forename' => $data['Participant']['first_name'],
			'surname' => $data['Participant']['last_name'],
			'signedname' => $signedName,
			'ipaddress' => $_SERVER['REMOTE_ADDR'],
			'conditions' => $conditions,
			'UserId' => $userId
		];
		if (! empty($signature)) {
			$params['signature'] = $signature;
		}

		try {
			$result = $this->post('all', array(
				'method' => 'SignWaiver',
				'conditions' => $params,
				'apiLog' => array(
					'model' => $model,
					'foreign_id' => $foreignId
				)
			));

			return !empty($result->SignWaiverResult) ? $result->SignWaiverResult : false;
		} catch (Exception $e) {
			CakeLog::write('debug', 'Sign Waiver failed');
			CakeLog::write('debug', $e->getMessage());
		}

		return false;
	}

/**
 * Add Waiver photo
 *
 * @param string $model     Model name
 * @param int    $foreignId Foreign ID
 * @param array  $data      Photo data
 * @return void
 */
	public function addWaiverPhoto($model, $foreignId, $data) {
		try {
			$this->post('all', array(
				'method' => 'AddWaiverPhoto',
				'conditions' => array(
					'waiverId' => $data['waiverId'],
					'photo' => $data['photo']
				),
				'apiLog' => array(
					'model' => $model,
					'foreign_id' => $foreignId
				)
			));

		} catch (Exception $e) {
			CakeLog::write('debug', 'Photo Upload to API failed');
			CakeLog::write('debug', $e->getMessage());
		}
	}

/**
 * Add Waiver signature
 *
 * @param string $model     Model name
 * @param int    $foreignId Foreign ID
 * @param array  $data      Signature data
 * @return void
 * @deprecated Use signWaiver instead
 */
	public function addWaiverSignature($model, $foreignId, $data) {
		try {
			$this->post('all', array(
				'method' => 'AddWaiverSignature',
				'conditions' => array(
					'waiverId' => $data['waiverId'],
					'signature' => $data['signature']
				),
				'apiLog' => array(
					'model' => $model,
					'foreign_id' => $foreignId
				)
			));

		} catch (Exception $e) {
			CakeLog::write('debug', 'Photo Upload to API failed');
			CakeLog::write('debug', $e->getMessage());
		}
	}

/**
 * Signs a waiver for a child
 *
 * @param  string $model         Name of the model, used for logging
 * @param  integer $foreignId    Foreign id of the record, used for logging
 * @param  array $data           Data containing the participant / waiver details
 * @param  integer $adultWaiverId Id of the adult waiver being used to co-sign for the child waiver
 * @return array
 */
	public function signChildWaiver($model, $foreignId, $data, $adultWaiverId = null) {
		$conditions = Hash::extract($data, 'Condition.{n}.title');

		$result = $this->find('all', array(
			'method' => 'SignChildWaiver',
			'conditions' => array(
				'parentId' => $adultWaiverId ?: $data['Participant']['waiver_id'],
				'dob' => $data['Participant']['date_of_birth'],
				'forename' => $data['Participant']['first_name'],
				'surname' => $data['Participant']['last_name'],
				'conditions' => json_encode($conditions)
			),
			'apiLog' => array(
				'model' => $model,
				'foreign_id' => $foreignId
			)
		));

		return !empty($result['WaiversApi']['d']) ? $result['WaiversApi']['d'] : false;
	}

/**
 * Checks a waiver exists and returns its details based on the users dob and email.
 *
 * @param  string $dob   User date of birth in Y-m-d H:i:s format
 * @param  string $email Email address of the user
 * @return array
 */
	public function checkWaiver($dob, $email) {
		$result = $this->find('all', array(
			'method' => 'CheckWaiver',
			'conditions' => array(
				'dob' => $dob,
				'email' => $email
			)
		));

		return !empty($result['WaiversApi']['d']) ? $result['WaiversApi']['d'] : false;
	}

/**
 * Checks a child waiver exists and returns its details. Unlike the standard
 * checkWaiver method, child waivers dont have an email address so require their
 * name be used instead.
 *
 * @param  string $dob   User date of birth in Y-m-d H:i:s format
 * @param  string $forename Users forename
 * @param  string $surname  Users surname
 * @return array
 */
	public function checkChildWaiver($dob, $forename, $surname) {
		$result = $this->find('all', array(
			'method' => 'CheckChildWaiver',
			'conditions' => array(
				'dob' => $dob,
				'forename' => $forename,
				'surname' => $surname
			)
		));

		return !empty($result['WaiversApi']['d']) ? $result['WaiversApi']['d'] : false;
	}

/**
 * Process bookings data from the API
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processBookings($data) {
		$result = [];

		foreach ($data as $booking) {
			$result[] = [
				'contact_name' => $booking['ContactName'],
				'location' => $booking['Location'],
				'reference' => $booking['Reference'],
				'date' => $booking['BookingDate'],
				'date' => $this->cleanApiDate($booking['BookingDate']),
				'check_in' => $booking['CheckInMins'],
				'description' => $booking['Description'],
				'participants' => $booking['Participants'],
				'message' => $booking['FailureDescription']
			];
		}

		return $result;
	}

/**
 * Process participants data from the API
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processBookingParticipants($data) {
		$result = [];

		foreach ($data as $participant) {
			$result[] = [
				'date_of_birth' => $participant['DateOfBirth'],
				'email_address' => $participant['EmailAddress'],
				'first_name' => preg_match('/^Users\ \d+$/', $participant['Forename']) ? null : $participant['Forename'],
				'last_name' => $participant['Surname'],
				'requires_waiver' => $participant['RequiresWaiver'],
				'user_id' => $participant['UserId'],
			];
		}

		return $result;
	}


/**
 * Process participant data from the API
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processBookingParticipant($data) {
		$result = [
			'Participant' => [
				'date_of_birth' => $data['DateOfBirth'],
				'email_address' => $data['EmailAddress'],
				'first_name' => preg_match('/^Users\ \d+$/', $data['Forename']) ? null : $data['Forename'],
				'last_name' => $data['Surname'],
				'requires_waiver' => $data['RequiresWaiver'],
				'user_id' => $data['UserId'],
				'display' => $data['Display'],
				'business_entity_id' => $data['BusinessEntityId']
			]
		];

		return $result;
	}

/**
 * Process participants data from the API
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processConfirmBookingWaiver($data) {
		$result = [];

		foreach ($data as $bookings) {
			if (! empty($bookings['Bookings'])) {
				$items = [];
				foreach ($bookings['Bookings'] as $booking) {
					$items[] = [
						'contact_name' => $booking['ContactName'],
						'location' => $booking['Location'],
						'reference' => $bookings['SalesOrderId'],
						'date' => $this->cleanApiDate($booking['BookingDate']),
						'check_in' => $booking['CheckInMins'],
						'description' => $booking['Description'],
						'participants' => $booking['Participants'],
						'message' => $booking['FailureDescription'],
						'status' => $booking['Status'],
						'requires_waiver' => $booking['RequiresWaiver'],
						'booking_ref' => $booking['Id'],
						'display' => $bookings['Display']
					];
				}

				$result[$bookings['SalesOrderId']] = $items;
			}
		}

		return $result;
	}

}
