<?php
App::uses('BuzzWaiversAppController', 'BuzzWaivers.Controller');
App::uses('CakeEvent', 'Event');

class WaiversController extends BuzzWaiversAppController {

	public $kiosk = false;

	public $participant;

	public $booking;

	public $activeStage;

	public function __construct(CakeRequest $Request, CakeResponse $Response) {
		$this->components['PreStage'] = array(
			'className' => 'BuzzWaivers.PreStage'
		);
		$this->components['PostStage'] = array(
			'className' => 'BuzzWaivers.PostStage'
		);

		parent::__construct($Request, $Response);
	}

	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(array(
			'complete_booking',
			'manage',
			'participants',
			'participant',
			'stage',
			'create_waivers'
		));

		// default stage process
		$stageProcess = Configure::read('BuzzWaivers.stageProcess');

		if ($this->kiosk) {
			$stageProcess = 'kiosk';
		}

		$stages = Configure::read('BuzzWaivers.stages');
		// check for an overriding process
		if ($this->Session->check('BuzzWaivers.stageProcess')) {
			$stageProcess = $this->Session->read('BuzzWaivers.stageProcess');
		}

		// check the selected process to use is valid
		if (empty($stages[$stageProcess]) || ! is_array($stages[$stageProcess])) {
			throw new InvalidStageProcessException(array($stageProcess));
		}
		$this->activeStageProcess = $stageProcess;
	}

/**
 * loop the given array and try to run the methods on the given
 * component - for processing the stages
 *
 * @param 	string  $componentKey 	The component key to run methods on
 * @param 	array 	$processors 		Array of any methods to run
 * @return 	bool
 */
	protected function _callStageProcessors($componentKey, $processors) {
		$result = true;
		if (! empty($processors)) {
			foreach ($processors as $method) {
				if (method_exists($this->{$componentKey}, $method)) {
					$result = $result && $this->{$componentKey}->{$method}();
				}
			}
		}
		return $result;
	}

/**
 * preStage failed, calculate the redirect route
 *
 * @param 	string 	$stageId 	The current stage ID
 * @return 	array 	$route 		The route array to redirect to
 */
	public function calculatePreStageRedirect($stageId) {
		$stages = Configure::read('BuzzWaivers.stages.' . $this->activeStageProcess);
		// all processors returned true, proceed
		$position = array_search($stageId, array_keys($stages));
		$position--;
		if ($position >= 0) {
			$stageIterator = new ArrayIterator($stages);
			$stageIterator->seek($position);
			$prevStageKey = $stageIterator->key();
			// redirect to the next stage
			if (! empty($prevStageKey)) {

				$route = array(
					'admin' => false,
					'plugin' => 'buzz_waivers',
					'controller' => 'waivers',
					'action' => 'stage',
				);

				$route[] = $prevStageKey;
			}
		}
		if (empty($route)) {
			// no stage found - start from beginning

			$route = array(
				'admin' => false,
				'plugin' => 'buzz_waivers',
				'controller' => 'waivers',
				'action' => 'participants'
			);
		}

		return $route;
	}

/**
 * postStage succeeded, calculate the redirect route
 *
 * @param 	string 	$stageId 	The current stage ID
 * @return 	array 	$route 	The route array to redirect to
 */
	public function calculatePostStageRedirect($stageId) {
		$stages = Configure::read('BuzzWaivers.stages.' . $this->activeStageProcess);
		// all processors returned true, proceed
		$position = array_search($stageId, array_keys($stages));
		$stageIterator = new ArrayIterator($stages);
		$stageIterator->seek($position);
		$stageIterator->next();
		$nextStageKey = $stageIterator->key();
		// redirect to the next stage
		if (! empty($nextStageKey)) {
			$route = array(
				'admin' => false,
				'plugin' => 'buzz_waivers',
				'controller' => 'waivers',
				'action' => 'stage',
			);
			$route[] = $nextStageKey;
		} else {
			// no more stages but everything was successful.
			// complete
			$route = array(
				'admin' => false,
				'plugin' => 'buzz_waivers',
				'controller' => 'waivers',
				'action' => 'participants'
			);
		}

		return $route;
	}

/**
 * the main stage method to load each stage of the waiver
 *
 */
	public function stage($stageId, $id) {
		$stageInfo = Configure::read('BuzzWaivers.stages.' . $this->activeStageProcess . '.' . $stageId);
		$stages = Configure::read('BuzzWaivers.stages.' . $this->activeStageProcess);

		// Load up participant waiver and see if they've already completed the photo and/or signature steps.
		// If so, skip them.
		$this->loadModel('BuzzWaivers.Participant');
		$participant = $this->Participant->find('first', array(
			'conditions' => array(
				'id' => $id
			)
		));

		$this->set('logoRoute', array(
			'plugin' => 'buzz_waivers',
			'controller' => 'waivers',
			'action' => 'participants',
			$id
		));

		$disablePhoto = false;
		$disableSignature = false;

		if (! empty($participant)) {
			if ($participant['Participant']['photo_complete'] == '1') {
				$disablePhoto = true;
			}

			if ($participant['Participant']['signature_complete'] == '1') {
				$disableSignature = true;
			}
		}

		// Check if this stage has been disabled in site settings
		if (Configure::read('SiteSetting.waivers_disable_' . $stageId) == '1' || ($stageId == 'photo' && $disablePhoto) || ($stageId == 'signature' && $disableSignature)) {
			$route = $this->calculatePostStageRedirect($stageId);
			$route[] = $id;

			if ($this->kiosk) {
				$route = '/kiosk' . Router::url($route);
			}

			return $this->redirect($route);
		}

		// Remove any stages that are disabled by site settings
		foreach ($stages as $name => $stage) {
			if (Configure::read('SiteSetting.waivers_disable_' . $name) == '1') {
				unset($stages[$name]);
			}
		}

		$stageCount = count($stages) - 1; // We dont count the last stage

		$currentStageCount = 1;
		$i = 1;
		foreach ($stages as $key => $stage) {
			if ($key == $stageId) {
				$currentStageCount = $i;
			}
			$i++;
		}

		if ($currentStageCount > 1 && $currentStageCount < count($stages)) {
			$firstStage = $stages;
			reset($firstStage);
			$this->set('logoRoute', array(
				'plugin' => 'buzz_waivers',
				'controller' => 'waivers',
				'action' => 'stage',
				key($firstStage),
				$id
			));
		}

		$this->set(compact('currentStageCount', 'stageCount'));

		if (empty($stageInfo)) {

			$route = array(
				'action' => 'index'
			);

			if ($this->kiosk) {
				$route = '/kiosk' . Router::url($route);
			}

			$this->redirect($route);
		}
		// set the active stage
		$this->activeStage = $stageId;
		// check the preStage calls
		$result = $this->_callStageProcessors('PreStage', $stageInfo['pre']);
		// go back a step if the results are false
		if ($result === false) {
			$route = $this->calculatePreStageRedirect($stageId);
			$route[] = $id;

			$route = Router::url($route);

			if ($this->kiosk) {
				$route = '/kiosk' . $route;
			}

			$this->redirect(
				$route
			);
		}

		if ($this->request->is('post') || $this->request->is('put')) {
			$result = $this->_callStageProcessors('PostStage', $stageInfo['post']);
			// proceed to next step
			if ($result === true) {
				$route = $this->calculatePostStageRedirect($stageId);
				$route[] = $id;

				$route = Router::url($route);

				if ($this->kiosk) {
					$route = '/kiosk' . $route;
				}

				$this->redirect(
					$route
				);
			}
		}

		$this->view = 'BuzzWaivers.' . $stageInfo['template'];

		$route = array(
			'plugin' => 'buzz_waivers',
			'controller' => 'waivers',
			'action' => 'participants',
			$id
		);

		$cancelUrl = Router::url($route);

		if ($this->kiosk) {
			$cancelUrl = '/kiosk' . $cancelUrl;
		}

		$this->set(compact('cancelUrl'));
	}

/**
 * Create participants for waivers at the end of the booking process.
 * @param string $salesRef Sales ref from the API
 * @return void
 */
	public function complete_booking($salesRef) {
		if ($this->request->is('post')) {
			$this->loadModel('BuzzBookings.Booking');
			$this->loadModel('BuzzWaivers.Participant');
			$booking = $this->Booking->findBySalesRef($salesRef);
			if (empty($booking)) {
				$this->Session->setFlash(
					__d('buzz_waivers', 'There was a problem validating the participant details.'),
					'flash_fail'
				);

				return $this->redirect($this->referer());
			}

			// Check for existing waiver participant records
			$storedParticipants = $this->Participant->find('count', [
				'conditions' => ['sales_ref' => $salesRef]
			]);
			if ($storedParticipants > 0) {
				$route = [
					'plugin' => 'buzz_check_bookings',
					'controller' => 'check_bookings',
					'action' => 'check'
				];
				if ($this->kiosk) {
					$route['kiosk'] = 'kiosk';
				}
				$checkBookingUrl = Router::url($route);
				$this->Session->setFlash(
					__d('buzz_waivers', 'All participant slots for this booking have already been allocated. If you wish to change a participant, please do so from your <a href="' . $checkBookingUrl . '"><strong>Check My Booking</strong></a> section.'),
					'flash_fail'
				);

				return $this->redirect($this->referer());
			}

			$waivers = [];
			$bookingItems = $this->Booking->BookingItem->find('all', [
				'contain' => 'BookingItemPlace',
				'conditions' => [
					'BookingItem.booking_id' => $booking['Booking']['id']
				]
			]);
			$bookingItems = Hash::combine($bookingItems, '{n}.BookingItem.id', '{n}');
			// We need to attach the booking IDs from the API to the participants and take into
			// account multi-activity packages that may have multiple IDs per participant (each will
			// need a waiver).
			foreach ($this->request->data['Participant'] as $participant) {
				if (! empty($bookingItems[$participant['booking_item_id']])) {
					$bookingItem = $bookingItems[$participant['booking_item_id']];
					foreach ($bookingItem['BookingItemPlace'] as $bookingItemPlace) {
						$participant['api_booking_ref'] = $bookingItemPlace['api_reference'];
						unset($participant['booking_item_id']);

						// Add the booking date in so we can do more accurate lookups later.
						$participant['booking_date'] = $booking['Booking']['booking_date'];
						$waivers[] = $participant;
					}
				}
			}
			$waivers = Hash::insert($waivers, '{n}.sales_ref', $salesRef);

			if ($this->Participant->saveMany($waivers)) {
				$event = new CakeEvent('Model.Participant.afterCreate', $this, array(
					'salesRef' => $salesRef
				));
				$this->getEventManager()->dispatch($event);
				$this->Session->setFlash(
					__d('buzz_waivers', 'Emails with instructions on how to complete the waiver have been successfully sent to the participant(s) you entered.'),
					'flash_success'
				);

				return $this->redirect('/');
			}

		}

		return $this->redirect(['controller' => 'check_bookings', 'action' => 'check', 'plugin' => 'buzz_check_bookings']);
	}

	public function participant($id) {
		$this->loadModel('BuzzWaivers.Participant');
		$this->loadModel('BuzzWaivers.WaiversApi');

		$participant = $this->WaiversApi->getBookingParticipant($id);

		if (empty($participant)) {
			throw new NotFoundException;
		}

		//$dob = date('Y-m-d H:i:s', strtotime($participant['Participant']['date_of_birth']));
		$dob = \DateTime::createFromFormat('d/m/Y', $participant['Participant']['date_of_birth']);
		$dob = $dob->format('Y-m-d h:i:s');

		// Check if the participant already exists in our db.
		$ourParticipant = $this->Participant->find('first', [
			'conditions' => [
				'waiver_user_id' => $id
			]
		]);

		if (! empty($ourParticipant)) {
			$participant = $ourParticipant;
		} else {
			// Create a record
			$participant = $this->Participant->save([
				'first_name' => $participant['Participant']['first_name'],
				'last_name' => $participant['Participant']['last_name'],
				'email' => $participant['Participant']['email_address'],
				'date_of_birth' => $dob,
				'waiver_user_id' => $id,
				'waiver_complete' => (empty($participant['Participant']['requires_waiver']) ? true : false)
			]);
		}

		if ($this->request->is('post') || $this->request->is('put')) {
			$failed = false;
			$i = 0;

			if (! isset($this->request->data['Participant'][$participant['Participant']['id']])) {
				throw new NotFoundException;
			}
			$data = $this->request->data['Participant'][$participant['Participant']['id']];


			$particpant['Participant']['first_name'] = $data['first_name'];
			$particpant['Participant']['last_name'] = $data['last_name'];
			$particpant['Participant']['date_of_birth'] = $data['date_of_birth'];
			$particpant['Participant']['email'] = $data['email'];
			$particpant['Participant']['adult_first_name'] = $data['adult_first_name'];
			$particpant['Participant']['adult_last_name'] = $data['adult_last_name'];
			$particpant['Participant']['adult_date_of_birth'] = $data['adult_date_of_birth'];
			$particpant['Participant']['adult_email'] = $data['adult_email'];

			if (! empty($participant['Participant']['adult_email']) && empty($participant['Participant']['email'])) {
				$participant['Participant']['email'] = $participant['Participant']['adult_email'];
			}

			if (! $this->Participant->save($participant)) {
				$failed = true;
			}

			if ($failed) {
				$this->Session->setFlash(__d('buzz_waivers', 'There was a problem saving your waiver. Please check your details and try again.'), 'flash_fail');
				return $this->redirect($this->here);
			}

			// Check if there is already a waiver with these details first.
			// If there is, we'll skip signing and mark signed, then redirect
			// back with a flash notice.
			$postData = $this->request->data;
			$participantId = $participant['Participant']['id'];
			$validWaiverId = false;

			// Check if this is a child waiver
			$dob = new DateTime($data['date_of_birth']);
			$minAge = new DateTime();
			$minAge->sub(new DateInterval('P18Y'));

			$formattedDob = $dob->format('Y-m-d H:i:s');

			if ($dob >= $minAge) {
				// Child waiver
				$childWaiver = $this->WaiversApi->checkChildWaiver($formattedDob, $data['first_name'], $data['last_name']);

				if (! empty($childWaiver['Status']) & $childWaiver['Status'] == 'Valid'
					&& ! empty($childWaiver['WaiverId']) && $childWaiver['WaiverId'] > 0) {
					// Looks like they already have a valid waiver, as an extra
					// precaution we'll check the expiry date as we dont know if
					// the api is updating the status value on expiry.
					$waiverDate = DateTime::createFromFormat('d/m/Y H:i:s', $childWaiver['ExpiryDate']);
					$now = new DateTime();

					if ($waiverDate >= $now) {
						// The waiver isnt expired, so we can use it
						$validWaiverId = $childWaiver['WaiverId'];
					}
				}

			} else {
				// Adult waiver
				$adultWaiver = $this->WaiversApi->checkWaiver($formattedDob, $data['email']);

				if (! empty($adultWaiver['Status']) && $adultWaiver['Status'] == 'Valid'
					&& ! empty($adultWaiver['WaiverId']) && $adultWaiver['WaiverId'] > 0) {

					// Looks like they already have a valid waiver, as an extra
					// precaution we'll check the expiry date as we dont know if
					// the api is updating the status value on expiry.

					$waiverDate = DateTime::createFromFormat('d/m/Y H:i:s', $adultWaiver['ExpiryDate']);

					$now = new DateTime();

					if ($waiverDate >= $now) {
						// The waiver isnt expired, so we can use it
						$validWaiverId = $adultWaiver['WaiverId'];
					}
				}
			}

			if ($validWaiverId > 0) {
				// We found a valid waiver for the participant details. Instead
				// of letting them fill out a new one, we're now going to
				// check to make sure they provided a photo and signature
				// (if those sections are enabled), and if so we'll then
				// update the record to store the waiver id and the fact
				// that it has already been signed. We'll then redirect them
				// back to the main participant info screen along with a
				// message to let them know that they already have a valid
				// waiver in the system.
				//
				$participant = $this->Participant->find('first', array(
					'conditions' => array(
						'email' => $data['email'],
						'date_of_birth' => $formattedDob,
						'waiver_complete' => 1
					)
				));

				if (empty($participant)) {
					$participant = $this->Participant->find('first', array(
						'conditions' => array(
							'email' => $data['email'],
							'date_of_birth' => $formattedDob
						)
					));
				}

				if (
					(Configure::read('SiteSetting.waivers_disable_photo') != '1' && $participant['Participant']['photo_complete'] == '1')
					&& (Configure::read('SiteSetting.waivers_disable_signature') != '1' && $participant['Participant']['signature_complete'] == '1')) {
					$data = array(
						'id' => $participantId,
						'waiver_id' => $validWaiverId,
						'waiver_complete' => 1
					);

					$this->Participant->clear();
					$this->Participant->id = $participantId;

					if ($this->Participant->save($data, array('deep' => true))) {
						$this->Session->setFlash(__d('buzz_waivers', 'A valid waiver for this participant was found in our system and does not need to be completed. The participant record has been updated below.'), 'flash_success');
					} else {
						$this->Session->setFlash(__d('buzz_waivers', 'A valid waiver was found for the participant, however there was a problem retrieving the details. Please try again.'), 'flash_fail');
					}
					return $this->redirect($this->here);
				}
			}

			$route = array(
				'admin' => false,
				'plugin' => 'buzz_waivers',
				'controller' => 'waivers',
				'action' => 'stage',
				'terms',
				$participantId
			);

			$route = Router::url($route);

			if ($this->kiosk) {
				$route = '/kiosk' . $route;
			}

			return $this->redirect($route);
		}

		$this->set('participant', $participant);
	}

/**
 * Participants page for a booking
 * @param string $id UUID for participant
 * @return void
 */
	public function participants($id, $single = false) {
		$this->loadModel('BuzzWaivers.Participant');
		$this->loadModel('BuzzWaivers.WaiversApi');

		$participant = $this->Participant->findById($id);
		$participantId = $id;

		if (empty($participant)) {
			throw new NotFoundException();
		}

		$participants = $this->Participant->getBookingParticipants(
			(isset($participant['Participant']['sales_ref']) ? $participant['Participant']['sales_ref'] : null),
			$participant['Participant']['api_booking_ref']
		);

		// The Twinwoods api in some situations (cause unknown) creates new participants
		// with a name of 'Users <number>'. We dont want this displaying so we'll
		// run though each participany and null the first name if it contains
		// any instance of this behaviour.
		if (! empty($participants)) {
			foreach ($participants as $key => $participant) {
				$participants[$key]['Participant']['first_name'] = preg_match('/^Users\ \d+$/', $participant['Participant']['first_name']) ? null : $participant['Participant']['first_name'];
			}
		}

		$this->set('salesRef', $participant['Participant']['sales_ref']);

		if (empty($participants)) {
			throw new NotFoundException();
		}

		$this->set('activity', $participants[0]['BookingItemPlace']['name']);

		$bookingParticipants = $this->WaiversApi->getBookingParticipants(
			$participant['Participant']['api_booking_ref']
		);

		if (empty($bookingParticipants)) {
			$this->Session->setFlash(__d('buzz_waivers', 'A booking for the specified participant could not be found.'), 'flash_fail');
			throw new NotFoundException();
		}

		// Sometimes new participants are added to the diary and returned by the API that we've not
		// previously encountered so we need to include those here if need be.
		$participantsDiff = count($bookingParticipants) - count($participants);
		if ($participantsDiff > 0) {
			$newParticipants = [];
			for ($i = 0; $i < $participantsDiff; $i++) {
				// There api will return the users first name as "Users <number>"
				// with an incrementing id if no user details are provided. We
				// override this and strip out any instance of "Users <number>"
				// from the first name field.
				$newParticipants[] = [
					'sales_ref' => $participants[0]['Participant']['sales_ref'],
					'api_booking_ref' => $participants[0]['Participant']['api_booking_ref'],
					'booking_date' => $participants[0]['Participant']['booking_date'],
					'first_name' => preg_match('/^Users\ \d+$/', $participants[0]['Participant']['first_name']) ? null : $participants[0]['Participant']['first_name']
				];
			}

			$this->Participant->saveMany($newParticipants);
			$participants = $this->Participant->getBookingParticipants(
				$participant['Participant']['sales_ref'],
				$participant['Participant']['api_booking_ref']
			);
		}

		// Now we want to check for any participants that we've got locally that
		// have missing data that we can grab from the above bookingParticipant
		// data retrieved from the api. Generally this happens if a booking was
		// creted elsewhere and is just being retrieved. As our site didnt handle
		// the booking we'd have no record of the data, so this will fix that.
		$emptyParticipantKeys = [];
		$linkedParticipants = 0; // Maintain a count of participants that have a valid api link.
		$usedUserIds = [];

		foreach ($participants as $key => $participant) {
			if (empty($participant['Participant']['waiver_user_id']) && empty($participant['Participant']['first_name'])) {
				// Looks like the participant isnt complete. Add its index to the
				// list of indexids we'll try to fill out automatically using the
				// api data.
				$emptyParticipantKeys[] = $key;
			} else {
				$usedUserIds[] = $participant['Participant']['waiver_user_id'];
				$linkedParticipants++;
			}
		}

		if (! empty($emptyParticipantKeys)) {
			// We've got one or more participants that are incomplete, lets see
			// if we can populate them from left over api data.

			// First we just want to generate a new array containing just the
			// unused api participants as they are free to be assigned to one of
			// the empty participant keys.
			$availableBookingParticipants = [];

			foreach ($bookingParticipants as $key => $bookingParticipant) {
				if (empty($bookingParticipant['user_id']) || ! in_array($bookingParticipant['user_id'], $usedUserIds)) {
					$availableBookingParticipants[] = $key;
				}
			}

			foreach ($emptyParticipantKeys as $key) {
				if (! empty($availableBookingParticipants)) {
					$bookingParticipantKey = reset($availableBookingParticipants);
					$bookingParticipant = $bookingParticipants[$bookingParticipantKey];

					$participants[$key]['Participant']['first_name'] = $bookingParticipant['first_name'];
					$participants[$key]['Participant']['last_name'] = $bookingParticipant['last_name'];
					$participants[$key]['Participant']['email'] = $bookingParticipant['email_address'];
					$participants[$key]['Participant']['waiver_user_id'] = $bookingParticipant['user_id'];
					$participants[$key]['Participant']['date_of_birth'] = $bookingParticipant['date_of_birth'];
					$participants[$key]['Participant']['name'] = $bookingParticipant['first_name'] . ' ' . $bookingParticipant['last_name'];
					$this->Participant->clear();
					$this->Participant->id = $participants[$key]['Participant']['id'];
					$this->Participant->save($participants[$key]['Participant']);
				}
			}
		}

		if ($this->request->is('post') || $this->request->is('put')) {

			// Save all the waivers again

			// Check all the participants are those we found above.
			$validParticipants = Hash::extract($participants, '{n}.Participant.id');

			$failed = false;
			$i = 0;
			foreach ($this->request->data['Participant'] as $pId => $participant) {
				if (in_array($id, $validParticipants)) {
					$this->Participant->clear();
					$this->Participant->id = $pId;

					$participant['waiver_user_id'] = ! empty($bookingParticipants[$i]) ? $bookingParticipants[$i]['user_id'] : null;
					$i++;

					if (! empty($participant['adult_email']) && empty($participant['email'])) {
						$participant['email'] = $participant['adult_email'];
					}

					if (! $this->Participant->save($participant)) {
						$failed = true;
					}
				}
			}

			if ($failed) {
				$this->Session->setFlash(__d('buzz_waivers', 'There was a problem saving your waiver. Please check your details and try again.'), 'flash_fail');
				return $this->redirect($this->here);
			}

			// Check if an individual waiver is being signed - if so we'll
			// redirect to that one.
			if (! empty($this->request->data['sign_waiver'])) {

				// Check if there is already a waiver with these details first.
				// If there is, we'll skip signing and mark signed, then redirect
				// back with a flash notice.
				$postData = $this->request->data;
				$participantData = $postData['Participant'][$postData['sign_waiver']];
				$participantId = $postData['sign_waiver'];

				$validWaiverId = false;

				// Check if this is a child waiver
				$dob = new DateTime($participantData['date_of_birth']);
				$minAge = new DateTime();
				$minAge->sub(new DateInterval('P18Y'));

				$formattedDob = $dob->format('Y-m-d H:i:s');

				if ($dob >= $minAge) {
					// Child waiver
					$childWaiver = $this->WaiversApi->checkChildWaiver($formattedDob, $participantData['first_name'], $participantData['last_name']);

					if (! empty($childWaiver['Status']) & $childWaiver['Status'] == 'Valid'
						&& ! empty($childWaiver['WaiverId']) && $childWaiver['WaiverId'] > 0) {
						// Looks like they already have a valid waiver, as an extra
						// precaution we'll check the expiry date as we dont know if
						// the api is updating the status value on expiry.
						$waiverDate = DateTime::createFromFormat('d/m/Y H:i:s', $childWaiver['ExpiryDate']);
						$now = new DateTime();

						if ($waiverDate >= $now) {
							// The waiver isnt expired, so we can use it
							$validWaiverId = $childWaiver['WaiverId'];
						}
					}

				} else {
					// Adult waiver
					$adultWaiver = $this->WaiversApi->checkWaiver($formattedDob, $participantData['email']);

					if (! empty($adultWaiver['Status']) && $adultWaiver['Status'] == 'Valid'
						&& ! empty($adultWaiver['WaiverId']) && $adultWaiver['WaiverId'] > 0) {

						// Looks like they already have a valid waiver, as an extra
						// precaution we'll check the expiry date as we dont know if
						// the api is updating the status value on expiry.

						$waiverDate = DateTime::createFromFormat('d/m/Y H:i:s', $adultWaiver['ExpiryDate']);

						$now = new DateTime();

						if ($waiverDate >= $now) {
							// The waiver isnt expired, so we can use it
							$validWaiverId = $adultWaiver['WaiverId'];
						}
					}
				}

				if ($validWaiverId > 0) {
					// We found a valid waiver for the participant details. Instead
					// of letting them fill out a new one, we're now going to
					// check to make sure they provided a photo and signature
					// (if those sections are enabled), and if so we'll then
					// update the record to store the waiver id and the fact
					// that it has already been signed. We'll then redirect them
					// back to the main participant info screen along with a
					// message to let them know that they already have a valid
					// waiver in the system.
					//
					$participant = $this->Participant->find('first', array(
						'conditions' => array(
							'email' => $participantData['email'],
							'date_of_birth' => $formattedDob,
							'waiver_complete' => 1
						)
					));

					if (empty($participant)) {
						$participant = $this->Participant->find('first', array(
							'conditions' => array(
								'email' => $participantData['email'],
								'date_of_birth' => $formattedDob
							)
						));
					}

					if (
						(Configure::read('SiteSetting.waivers_disable_photo') != '1' && $participant['Participant']['photo_complete'] == '1')
						&& (Configure::read('SiteSetting.waivers_disable_signature') != '1' && $participant['Participant']['signature_complete'] == '1')) {
						$data = array(
							'id' => $participantId,
							'waiver_id' => $validWaiverId,
							'waiver_complete' => 1
						);

						$this->Participant->clear();
						$this->Participant->id = $participantId;

						if ($this->Participant->save($data, array('deep' => true))) {
							$this->Session->setFlash(__d('buzz_waivers', 'A valid waiver for this participant was found in our system and does not need to be completed. The participant record has been updated below.'), 'flash_success');
						} else {
							$this->Session->setFlash(__d('buzz_waivers', 'A valid waiver was found for the participant, however there was a problem retrieving the details. Please try again.'), 'flash_fail');
						}
						return $this->redirect($this->here);
					}
				}

				$route = array(
					'admin' => false,
					'plugin' => 'buzz_waivers',
					'controller' => 'waivers',
					'action' => 'stage',
					'terms',
					$this->request->data['sign_waiver']
				);

				$route = Router::url($route);

				if ($this->kiosk) {
					$route = '/kiosk' . $route;
				}

				return $this->redirect($route);
			}

			$this->Session->setFlash(__d('buzz_waivers', 'Participant information has been successfully updated for this booking.'), 'flash_success');
			return $this->redirect($this->here);
		}

		// Split participants into those that have completed a waiver, and those
		// that have not.
		$completedParticipants = array();

		foreach ($participants as $key => $participant) {
			// We need to do an additional check now. If the
			// bookingParticipants match up with an existng waiver record,
			// we'll use the waiver status there to see if we want to allow
			// signing of a waiver for the user.
			$matchingRecord = false;
			foreach ($bookingParticipants as $bookingParticipant) {
				// If any of the fields are empty theres no point even trying to
				// match them up as we'll need to fill them out anyway.
				if (
					empty($bookingParticipant['date_of_birth']) ||
					empty($bookingParticipant['email_address'])
				) {
					continue;
				}

				// Reformat the DOB to match how we're storing it in our db (DATETIME)
				$dob = \DateTime::createFromFormat('m/d/Y', $bookingParticipant['date_of_birth']);
				$dob = $dob->format('Y-m-d 00:00:00');
				if (
					$dob == $participant['Participant']['date_of_birth'] &&
					$bookingParticipant['email_address'] == $participant['Participant']['email'] &&
					$bookingParticipant['first_name'] == $participant['Participant']['first_name'] &&
					$bookingParticipant['last_name'] == $participant['Participant']['last_name']
				) {
					// Found a match
					$matchingRecord = $bookingParticipant;
				}
			}

			if (! empty($matchingRecord)) {
				// Found a matching record for this participant! We'll
				// use the value of the requires_waiver param to update the
				// waiver_complete and waiver_user_id values if needed

				// The first check is to see if we have a waiver user id already.
				// If we dont, add it.
				if ($participant['Participant']['waiver_user_id'] == null && $matchingRecord['user_id'] != null) {
					// Looks like we need to add the user id.
					$participants[$key]['Participant']['waiver_user_id'] = $matchingRecord['user_id'];
				}

				// Now check for the waiver status
				if ($participant['Participant']['waiver_complete'] != '1') {
					$participants[$key]['Participant']['waiver_complete'] = $matchingRecord['requires_waiver'] ? false : '1';
				}
			}

			// Now that we know for sure if a waiver has actually been completed
			// (or if its simply not needed, as determined by the api) we can
			// split these into two arrays - one for complete, one for incomplete.
			//
			// This is done to make the frontend a bit easier to work with, as
			// participants needing a waiver siging need to see a different
			// element, and need to be pushed to the top of the page. To do this
			// the participants array is used first, and the completedParticipants
			// array is used to show the details as read-only at the bottom.
			//
			// Note: we re-check this using the key as theres a chance it was
			// updated above. Saves doing an extra foreach.
			if ($participants[$key]['Participant']['waiver_complete'] == '1') {
				$completedParticipants[] = $participant;
				unset($participants[$key]);
			}
		}

		$this->set(compact('participants', 'completedParticipants'));
	}

/**
 * Create waivers - initial step of the waiver process
 * @param int $salesRef Sales order ID
 * @param string $bookingRef API's booking ID
 * @return void Redirects to next stage
 */
	public function create_waivers($salesRef, $bookingRef = null) {
		$this->loadModel('BuzzWaivers.WaiversApi');

		if (empty($bookingRef)) {
			$bookingRef = $salesRef;
			$salesRef = null;
		}

		$this->loadModel('BuzzWaivers.Participant');
		$conditions = [
			'Participant.api_booking_ref' => $bookingRef
		];

		if (! empty($salesRef)) {
			$conditions['Participant.sales_ref'] = $salesRef;
		}

		$existingParticipants = $this->Participant->find('all', [
			'conditions' => $conditions
		]);

		if (! empty($existingParticipants)) {
			return $this->redirect($this->_generateParticipantRoute(
				$existingParticipants[0]['Participant']['id']
			));
		}

		$participants = $this->WaiversApi->getBookingParticipants($bookingRef);
		if (empty($participants)) {
			$this->Session->setFlash(__d('buzz_waivers', 'Unable to find booking.'), 'flash_fail');

			return $this->redirect([
				'controller' => 'check_bookings',
				'action' => 'check',
				'plugin' => 'buzz_check_bookings'
			]);
		}

		// Create waiver records
		$data = [];
		foreach ($participants as $participant) {
			if (! empty($participant['date_of_birth'])) {
				$dob = strtotime($participant['date_of_birth']);
				$dob = date('Y-m-d 00:00:00', $dob);
			}

			// If the api reports that the user does not need a waiver, we
			// mark waiver_complete as true.
			$waiverComplete = $participant['requires_waiver'] != '1' ? true : false;

			$data[] = [
				'sales_ref' => $salesRef,
				'api_booking_ref' => $bookingRef,
				'first_name' => preg_match('/^Users\ \d+$/', $participant['first_name']) ? null : $participant['first_name'],
				'last_name' => $participant['last_name'],
				'email' => $participant['email_address'],
				'date_of_birth' => isset($dob) ? $dob : null,
				'requires_waiver' => $participant['requires_waiver'] ?: null,
				'emailed' => false,
				'waiver_complete' => $waiverComplete,
				'waiver_user_id' => $participant['user_id']
			];
		}
		$this->Participant->saveMany($data);

		return $this->redirect($this->_generateParticipantRoute($this->Participant->id));
	}

/**
 * Generates a route to the participant step
 * @param string $participantId Participant UUID
 * @return array Route
 */
	protected function _generateParticipantRoute($participantId) {
		$route = [
			'admin' => false,
			'plugin' => 'buzz_waivers',
			'controller' => 'waivers',
			'action' => 'participants',
			$participantId
		];
		$route = Router::url($route);
		if ($this->kiosk) {
			$route = '/kiosk' . $route;
		}

		return $route;
	}
}
