<?php

App::uses('BuzzSourceAppModel', 'BuzzSource.Model');

class VoucherApi extends BuzzSourceAppModel {

	/**
	 * Create a gift voucher.
	 *
	 * @param string $model Calling model for logging API call
	 * @param int $foreignId
	 * @param string $firstName
	 * @param string $lastName
	 * @param string $email
	 * @param string $telephone
	 * @param array $address Address generated by CustomerAddress::generateAddressArray()
	 * @param string $message Gift message
	 * @param string $templateId Voucher template
	 * @param int $quantity Quantity of vouchers being purchased
	 * @param bool $isEvoucher True if an e-voucher
	 * @param string $ip IP address
	 * @param array $payment Payment details for the API
	 * @param float $total Amount paid
	 * @param int $salesOrderId Sales Order ID (from the API)
	 * @return array|bool
	 */
	public function createVoucher($model, $foreignId, $firstName, $lastName, $email, $telephone, $address, $message, $templateId, $quantity, $isEvoucher, $ip, array $payment, $total = 0, $salesOrderId = null, $deliveryFirstname = false, $deliveryLastname = false, $deliveryAddress = false) {

		$params = [
			'firstname' => $firstName,
			'lastname' => $lastName,
			'email' => $email,
			'telephone' => $telephone,
			'addressline1' => $address['address_line_1'],
			'addressline2' => $address['address_line_2'],
			'addressline3' => $address['address_line_3'],
			'addressline4' => $address['address_line_4'],
			'postcode' => $address['postcode'],
			'message' => $message,
			'templateId' => $templateId,
			'quantity' => (int)$quantity,
			'isevoucher' => $isEvoucher === true ? 'true' : 'false',
			'ipaddress' => $ip === null ? $_SERVER['REMOTE_ADDR'] : $ip,
			'vendortxcode' => $payment['vendorTxCode'],
			'vpstxId' => $payment['vpstxId'],
			'status' => $payment['status'],
			'txauthno' => $payment['txauthno'],
			'totalPaid' => $total,
			'salesorderId' => $salesOrderId ?: 0
		];

		if (! empty($deliveryAddress)) {
			$params['deliveryfirstname'] = $deliveryFirstname;
			$params['deliverylastname'] = $deliveryLastname;
			$params['deliveryaddressline1'] = $deliveryAddress['address_line_1'];
			$params['deliveryaddressline2'] = $deliveryAddress['address_line_2'];
			$params['deliveryaddressline3'] = $deliveryAddress['address_line_3'];
			$params['deliveryaddressline4'] = $deliveryAddress['address_line_4'];
			$params['deliverypostcode'] = $deliveryAddress['postcode'];
		}

		$result = $this->find('all', [
			'method' => 'CreateVoucher',
			'conditions' => $params,
			'apiLog' => [
				'model' => $model,
				'foreign_id' => $foreignId
			]
		]);

		return !empty($result['VoucherApi']['d']) ? $this->_processVouchers($result['VoucherApi']['d']) : false;
	}

/**
 * Process gift vouchers data from the API
 *
 * @param array $data Response from API
 * @return array|bool Returns false if something is wrong
 */
	protected function _processVouchers($data) {
		$result = [];

		foreach ($data as $voucher) {
			if (empty($voucher['VoucherNo']) || empty($voucher['ExpiryDate'])) {
				return false;
			}
			$result['Voucher'][] = array(
				'id' => $voucher['Id'],
				'description' => $voucher['Description'],
				'code' => $voucher['VoucherNo'],
				'expiry_date' => $voucher['ExpiryDate'],
				'sales_order_id' => $voucher['SalesOrderId']
			);
		}

		return $result;
	}

	public function applyDiscountToVoucher($model, $foreignId, $voucherId, $discountCode, $amount, $type = 'F') {

		$result = $this->find('all', [
			'method' => 'DiscountVoucher',
			'conditions' => [
				'voucherId' => $voucherId,
				'discounttype' => $type,
				'amount' => $amount,
				'discountcode' => $discountCode
			],
			'apiLog' => [
				'model' => $model,
				'foreign_id' => $foreignId
			]
		]);

		return $result;
	}

}
