<?php

App::uses('BuzzVouchersAppModel', 'BuzzVouchers.Model');

class VoucherActivity extends BuzzVouchersAppModel {

	/**
	 * Behaviors
	 *
	 * @var array
	 */
	public $actsAs = array(
		'BuzzAdverts.AdvertHost',
		'BuzzTranslate.Translatable',
		'EvCore.Tabbed' => array(
			'max_tabs' => 3
		),
	);

	/**
	 * Has many associations
	 *
	 * @var array
	 */
	public $hasMany = array(
		'GiftVoucher' => array(
			'className' => 'BuzzVouchers.GiftVoucher'
		),
	);

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'validate.required'
			),
			'max' => array(
				'rule' => array('maxLength', 150),
				'message' => 'validate.maxLength150'
			)
		)
	);

	/**
	 * Image slots
	 *
	 * @var array
	 */
	public $imageSlots = [
		'listing' => [
			'slots' => 1,
			'fields' => []
		],
	];

	/**
	 * Constructor
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		if (Configure::check('BuzzVouchers.routes.activities') === true) {
			$this->addBehaviour('Routable.Routable', Configure::read('BuzzVouchers.routes.activities'));
		}

		// If BuzzGallery loaded associate activities with galleries. We're
		// handling this in the constructor so that this feature can easily
		// be disabled on a site if BuzzGallery is excluded.
		if (CakePlugin::loaded('BuzzGallery') === true) {
			$this->belongsTo['Gallery'] = array(
				'className' => 'BuzzGallery.Gallery'
			);
		}
	}

	/**
	 * Get all visible listed voucher activities.
	 *
	 * @return array
	 */
	public function getListed() {
		$params = array(
			'contain' => [
				'ListingImage'
			],
			'conditions' => [
				'VoucherActivity.is_active' => true,
				'VoucherActivity.is_removed' => false
			],
			'order' => [
				'VoucherActivity.sequence' => 'ASC',
				'VoucherActivity.created' => 'ASC'
			]
		);
		return $this->find('all', $params);
	}

	/**
	 * Returns a translated list of listed voucher activities.
	 *
	 * @return array
	 */
	public function listListed() {
		$results = $this->getListed();

		$data = [];
		foreach ($results as $result) {
			$data[$result['VoucherActivity']['id']] = TranslateUtil::translate($result, 'VoucherActivity.name');
		}

		return $data;
	}

	/**
	 * Checks if there is only one available listed activity for the site, if
	 * there's only one activity it will return the activity ID, otherwise it
	 * returns false
	 *
	 * @return bool|int
	 */
	public function onlyAvailableActivity() {
		// Get the listed activities
		$data = $this->getListed();
		return count($data) === 1 && !empty($data[0]['VoucherActivity']['id']) ? $data[0]['VoucherActivity']['id'] : false;
	}

	/**
	 * Returns a voucher activity and its vouchers
	 *
	 * @param int $voucherActivityId Voucher activity ID
	 * @return array|bool
	 */
	public function getPurchasableVoucherActivity($voucherActivityId) {
		// Want to make sure all returned vouchers have an API reference set.
		$params = array(
			'contain' => array(
				'GiftVoucher' => array(
					'Image',
					'conditions' => array(
						'GiftVoucher.is_active' => true,
						'OR' => array(
							'GiftVoucher.api_reference <>' => null,
							'GiftVoucher.api_reference <>' => ''
						)
					),
					'order' => array(
						'GiftVoucher.sequence' => 'ASC'
					)
				)
			)
		);
		// Get the activity and its vouchers.
		$data = $this->readForView($voucherActivityId, $params);
		return !empty($data) && !empty($data['GiftVoucher']) ? $data : false;
	}

	/**
	 * Read for edit
	 *
	 * @param int $id
	 * @param array $query
	 * @return array
	 */
	public function readForEdit($id, $query = []) {
		$query['contain']['Advert'] = array(
			'AdvertRelation',
			'AdvertTemplate',
			'Image',
			'order' => array('AdvertRelation.sequence' => 'ASC')
		);
		return parent::readForEdit($id, $query);
	}

}
